﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

 /**
 * @file    cdacm_Driver.h
 * @brief   Driver Abstraction
 *
 * @details This file defines the driver class for commmunicating with different supported
 *          USB-to-Serial devices.
 */

#include "cdacm_PrivateIncludes.h"
#include "cdacm_Device.h"
#include "cdacm_SafeHandleManager.h"

namespace nn {
namespace cdacm {
namespace driver {

class Driver
{
public:
    Driver();
    ~Driver();

    Result Initialize(nn::os::MultiWaitType **pDeviceAvailableEventPtr,
                      void *pBuffer,
                      uint32_t bufferLen) NN_NOEXCEPT;

    Result Finalize() NN_NOEXCEPT;

    Result OpenHandle(nn::os::SystemEventType    **pDetachEventPtr,
                      UnitProfile                 *pOutProfile,
                      nn::os::MultiWaitHolderType *holder,
                      CdAcmParameters             *parameters) NN_NOEXCEPT;
    Result CloseHandle(UnitProfile *pOutProfile) NN_NOEXCEPT;

    Result Read(size_t* pBytesRead,
                void* pOutBuffer,
                UnitHandle handle,
                uint32_t length) NN_NOEXCEPT;
    Result Write(size_t* pBytesWritten,
                 const void* pOutBuffer,
                 UnitHandle handle,
                 uint32_t length) NN_NOEXCEPT;
    Result WriteAsync(uint32_t                 *pOutXferId,
                      nn::os::SystemEventType **pCompletionEventPtr,
                      const void               *pOutBuffer,
                      UnitHandle                handle,
                      uint32_t                  length) NN_NOEXCEPT;
    Result GetWriteAsyncResult(uint32_t            *pOutCount,
                               nn::usb::XferReport *pOutReport,
                               UnitHandle           handle,
                               uint32_t             count) NN_NOEXCEPT;

    Result RegisterDevice(Device *pDev) NN_NOEXCEPT;
    void   UnregisterDevice(Device *pDev) NN_NOEXCEPT;

private:

    // Host Stack
    nn::usb::Host                          m_UsbHost;
    nn::usb::DeviceFilter                  m_UsbDeviceFilter[NumberOfSupportedUarts];
    nn::os::SystemEventType                m_UsbIfAvailableEvent[NumberOfSupportedUarts];
    nn::os::MultiWaitHolderType            m_UsbIfAvailableEventHolder[NumberOfSupportedUarts];
    nn::os::MultiWaitType                  m_MultiWait;

    nn::os::Mutex                          m_Mutex;
    uint8_t                               *m_pClientBuf;
    uint32_t                               m_ClientBufLen;
    uint8_t                               *m_pStartWrite;
    bool                                   m_HasRolledOver;
    bool                                   m_HaveAsyncData;

    Device                                 m_Device[DeviceCountMax];

    SafeHandleManager<Device, DeviceCountMax> m_DeviceHandleManager;

private:
    Result CreateAttached(nn::os::SystemEventType **pDetachEventPtr, uint8_t uartIndex) NN_NOEXCEPT;

    Result CreateDevice(nn::os::SystemEventType      **pDetachEventPtr,
                        nn::usb::InterfaceQueryOutput *pProfileArray,
                        uint32_t                       numberOfProfileEntries,
                        uint8_t                        uartIndex) NN_NOEXCEPT;

    void DestroyAllDevices() NN_NOEXCEPT;

    Result QueueWrite(const uint8_t * pOutBuffer, uint32_t length) NN_NOEXCEPT;
    Result WriteAsyncData(UnitHandle handle) NN_NOEXCEPT;
    Result MultiXferWriteAsyncData(Device        *pDevice,
                                   uint64_t       bytesToWrite,
                                   const uint8_t *pSrc) NN_NOEXCEPT;
};

} // driver
} // cdacm
} // nn
