﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "cdacm_Driver.h"

namespace nn {
namespace cdacm {

namespace driver {

NN_USB_DMA_ALIGN uint8_t g_AsyncBuf[4096];

Driver::Driver()
    : m_Mutex(false)
    , m_pClientBuf(nullptr)
    , m_ClientBufLen(0)
    , m_pStartWrite(nullptr)
    , m_HasRolledOver(false)
    , m_HaveAsyncData(false)
{
    nn::os::InitializeMultiWait(&m_MultiWait);
}

Driver::~Driver() NN_NOEXCEPT
{
    nn::os::FinalizeMultiWait(&m_MultiWait);
}

Result
Driver::Initialize(nn::os::MultiWaitType **pDeviceAvailableEventPtr,
                   void                   *pBuffer,
                   uint32_t                bufferLen) NN_NOEXCEPT
{
    Result result = ResultSuccess();

    NN_CDACM_ABORT_IF_NULL(pDeviceAvailableEventPtr);

    if (pBuffer != nullptr && bufferLen == 0)
    {
        return ResultInvalidArgs();
    }

    NN_CDACM_DO(m_UsbHost.Initialize());
    if (result.IsFailure())
    {
        return result;
    }

    for (int i = 0; i < NumberOfSupportedUarts; i++)
    {
        m_UsbDeviceFilter[i].matchFlags =
            nn::usb::DeviceFilterMatchFlags_Vendor |
            nn::usb::DeviceFilterMatchFlags_Product;
        m_UsbDeviceFilter[i].idVendor = SupportedUarts[i][FieldIndex_Vid];
        m_UsbDeviceFilter[i].idProduct = SupportedUarts[i][FieldIndex_Pid];

        NN_CDACM_DO(
            m_UsbHost.CreateInterfaceAvailableEvent(
                &m_UsbIfAvailableEvent[i],
                nn::os::EventClearMode_ManualClear,
                i,
                &m_UsbDeviceFilter[i]
            )
        );
        if (result.IsFailure())
        {
            return result;
        }
        nn::os::InitializeMultiWaitHolder(&m_UsbIfAvailableEventHolder[i],
                                          &m_UsbIfAvailableEvent[i]);
        nn::os::LinkMultiWaitHolder(&m_MultiWait, &m_UsbIfAvailableEventHolder[i]);
    }
    *pDeviceAvailableEventPtr = &m_MultiWait;

    m_pClientBuf = static_cast<uint8_t*>(pBuffer);
    m_pStartWrite = m_pClientBuf;
    m_ClientBufLen = bufferLen;

    return result;
}

Result
Driver::Finalize() NN_NOEXCEPT
{
    Result result = ResultSuccess();

    DestroyAllDevices();

    for (int i = 0; i < NumberOfSupportedUarts; i++)
    {
        nn::os::UnlinkMultiWaitHolder(&m_UsbIfAvailableEventHolder[i]);
        nn::os::FinalizeMultiWaitHolder(&m_UsbIfAvailableEventHolder[i]);
        m_UsbHost.DestroyInterfaceAvailableEvent(&m_UsbIfAvailableEvent[i], i);
    }
    m_HasRolledOver = false;
    m_HaveAsyncData = false;
    m_pClientBuf = nullptr;
    m_ClientBufLen = 0;
    m_pStartWrite = nullptr;

    NN_CDACM_DO(m_UsbHost.Finalize());

    return result;
}

Result
Driver::OpenHandle(nn::os::SystemEventType    **pDetachEventPtr,
                   UnitProfile                 *pOutProfile,
                   nn::os::MultiWaitHolderType *pHolder,
                   CdAcmParameters             *pParameters) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    UnitHandle handle;
    Device *pDevice = nullptr;
    uint8_t uartIndex;

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    for (uartIndex = 0; uartIndex < NumberOfSupportedUarts; uartIndex++)
    {
        if (pHolder == &m_UsbIfAvailableEventHolder[uartIndex])
        {
            break;
        }
    }
    NN_CDACM_ABORT_UNLESS(uartIndex < NumberOfSupportedUarts);

    result = CreateAttached(pDetachEventPtr, uartIndex);
    if (result.IsFailure())
    {
        goto Exit;
    }

    result = m_DeviceHandleManager.Discover(*pDetachEventPtr, &handle);
    if (result.IsFailure())
    {
        goto Exit;
    }

    pDevice = m_DeviceHandleManager.Acquire(handle);

    if (pDevice == nullptr)
    {
        result = ResultDeviceNotAvailable();
        goto Exit;
    }

    pDevice->GetProfile(pOutProfile);
    NN_CDACM_DO(pDevice->InitRS232(pParameters));



    if (result.IsFailure())
    {
        m_DeviceHandleManager.Release(handle);
        UnregisterDevice(pDevice);
        pDevice->Finalize();
        goto Exit;
    }

    if (m_HaveAsyncData)
    {
        WriteAsyncData(handle);
        m_HaveAsyncData = false;
    }

    m_DeviceHandleManager.Release(handle);

Exit:

    return result;
}

Result
Driver::CloseHandle(UnitProfile *pProfile) NN_NOEXCEPT
{
    Result      result  = ResultSuccess();
    UnitHandle  handle  = pProfile->handle;
    Device     *pDevice = nullptr;

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    pDevice = m_DeviceHandleManager.Acquire(handle);

    if (pDevice == nullptr)
    {
        result = ResultDeviceNotAvailable();
        goto Exit;
    }
    m_DeviceHandleManager.Release(handle);

    UnregisterDevice(pDevice);
    pDevice->Finalize();

Exit:

    return result;
}

Result
Driver::CreateAttached(nn::os::SystemEventType **pDetachEventPtr, uint8_t uartIndex) NN_NOEXCEPT
{
    Result                        result = ResultSuccess();
    int32_t                       ifCount = 0;
    nn::usb::InterfaceQueryOutput outBuffer[DeviceCountMax];

    result = m_UsbHost.QueryAvailableInterfaces(
        &ifCount, outBuffer, sizeof(outBuffer), &m_UsbDeviceFilter[uartIndex]
    );
    if (result.IsFailure())
    {
        goto Exit;
    }

    if (ifCount == 0)
    {
        nn::os::ClearSystemEvent(&m_UsbIfAvailableEvent[uartIndex]);
        result = ResultNoNewDevices();
        goto Exit;
    }
    result = CreateDevice(pDetachEventPtr, outBuffer, ifCount, uartIndex);

Exit:

    return result;
}

Result
Driver::CreateDevice(nn::os::SystemEventType       **pDetachEventPtr,
                     nn::usb::InterfaceQueryOutput  *pProfileArray,
                     uint32_t                        numberOfProfileEntries,
                     uint8_t                         uartIndex) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    uint32_t  index;

    for (index = 0; index < DeviceCountMax; index++)
    {
        if (m_Device[index].GetDeviceState() == DeviceState_Uninitialized)
        {
            break;
        }
    }

    if (index == DeviceCountMax)
    {
        NN_CDACM_WARN("CDACM: Maximum device count (%d) reached.\n", DeviceCountMax);
        result = ResultMaxDevicesReached();
        goto Exit;
    }

    result = m_Device[index].Initialize(&m_UsbHost, pProfileArray, pDetachEventPtr);
    if (result.IsFailure())
    {
        goto Exit;
    }
    NN_CDACM_INFO("Registering device\n");
    result = RegisterDevice(&m_Device[index]);
    if (result.IsFailure())
    {
        m_Device[index].Finalize();
        goto Exit;
    }

    if (numberOfProfileEntries == 1)
    {
        NN_CDACM_INFO("Driver::CreateDevice: Clearing ifc avail event @ uart index %d\n", uartIndex);
        nn::os::ClearSystemEvent(&m_UsbIfAvailableEvent[uartIndex]);
    }

Exit:

    return result;
}

Result
Driver::Read(size_t*    pBytesRead,
    void*      pOutBuffer,
    UnitHandle handle,
    uint32_t   length) NN_NOEXCEPT
{
    Result  result = ResultInvalidUnitHandle();
    Device* pDevice = m_DeviceHandleManager.Acquire(handle);

    if (pDevice == nullptr)
    {
        result = ResultInvalidUnitHandle();
        goto Exit;
    }
    result = pDevice->Read(pBytesRead, pOutBuffer, length);
    m_DeviceHandleManager.Release(handle);

Exit:

    return result;
}

Result
Driver::Write(size_t* pBytesWritten,
    const void* pOutBuffer,
    UnitHandle handle,
    uint32_t length) NN_NOEXCEPT
{
    Result  result = ResultInvalidUnitHandle();
    Device* pDevice = nullptr;

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if (handle == 0)
    {
        UnitHandle newHandle;
        result = m_DeviceHandleManager.GetFirst(&newHandle);
        if (result.IsSuccess())
        {
            pDevice = m_DeviceHandleManager.Acquire(newHandle);
        }

        if (pDevice == nullptr)
        {
            result = QueueWrite(static_cast<const uint8_t*>(pOutBuffer), length);
        }
        else
        {
            result = pDevice->Write(pBytesWritten, pOutBuffer, length);
            m_DeviceHandleManager.Release(newHandle);
        }
    }
    else
    {
        pDevice = m_DeviceHandleManager.Acquire(handle);

        if (pDevice == nullptr)
        {
            result = ResultInvalidUnitHandle();
            goto Exit;
        }

        result = pDevice->Write(pBytesWritten, pOutBuffer, length);
        m_DeviceHandleManager.Release(handle);
    }

Exit:
    return result;
}

Result
Driver::WriteAsync(
    uint32_t                 *pOutXferId,
    nn::os::SystemEventType **pCompletionEventPtr,
    const void               *pOutBuffer,
    UnitHandle                handle,
    uint32_t                  length) NN_NOEXCEPT
{
    Result  result = ResultInvalidUnitHandle();
    Device* pDevice = nullptr;

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    pDevice = m_DeviceHandleManager.Acquire(handle);

    if (pDevice == nullptr)
    {
        result = ResultInvalidUnitHandle();
        goto Exit;
    }

    result = pDevice->WriteAsync(pOutXferId, pCompletionEventPtr, pOutBuffer, length);
    m_DeviceHandleManager.Release(handle);
Exit:
    return result;
}

Result
Driver::GetWriteAsyncResult(
    uint32_t            *pOutCount,
    nn::usb::XferReport *pOutReport,
    UnitHandle           handle,
    uint32_t             count) NN_NOEXCEPT
{
    Result  result = ResultInvalidUnitHandle();
    Device* pDevice = nullptr;

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    pDevice = m_DeviceHandleManager.Acquire(handle);

    if (pDevice == nullptr)
    {
        result = ResultInvalidUnitHandle();
        goto Exit;
    }

    result = pDevice->GetWriteAsyncResult(pOutCount, pOutReport, count);

    m_DeviceHandleManager.Release(handle);
Exit:
    return result;
}

Result
Driver::MultiXferWriteAsyncData(Device *pDevice, uint64_t bytesToWrite, const uint8_t *pSrc) NN_NOEXCEPT
{
    Result   result       = ResultSuccess();
    Result   retResult    = ResultSuccess();
    size_t   bytesInStage = 0;

    do
    {
        bytesInStage = bytesToWrite > sizeof(g_AsyncBuf) ? sizeof(g_AsyncBuf) : bytesToWrite;

        std::memcpy(static_cast<void*>(g_AsyncBuf), static_cast<const void*>(pSrc), bytesInStage);
        bytesToWrite -= bytesInStage;
        pSrc += bytesInStage;

        NN_CDACM_DO(pDevice->Write(&bytesInStage, g_AsyncBuf, bytesInStage));
        if (result.IsFailure() && retResult.IsSuccess())
        {
            retResult = result;
        }
    } while (bytesToWrite > 0);
    return retResult;
}


Result
Driver::WriteAsyncData(UnitHandle handle) NN_NOEXCEPT
{
    Result result = ResultSuccess();
    uint64_t nUntilEnd = m_ClientBufLen - static_cast<uint64_t>(m_pStartWrite - m_pClientBuf);

    Device *pDevice = m_DeviceHandleManager.Acquire(handle);

    if (pDevice == nullptr)
    {
        result = ResultInvalidUnitHandle();
        goto Exit;
    }

    if (m_HasRolledOver == true)
    {
        NN_CDACM_DO(MultiXferWriteAsyncData(pDevice, nUntilEnd, m_pStartWrite));

    }
    NN_CDACM_DO(MultiXferWriteAsyncData(pDevice, m_pStartWrite - m_pClientBuf, m_pClientBuf));

    m_HaveAsyncData = false;
    m_DeviceHandleManager.Release(handle);

Exit:
    return result;
}


Result
Driver::RegisterDevice(Device *pDevice) NN_NOEXCEPT
{
    UnitHandle handle;
    Result     result;

    NN_CDACM_DO(m_DeviceHandleManager.Register(pDevice, &handle));

    if (result.IsFailure())
    {
        goto Exit;
    }

    pDevice->SetHandle(handle);

Exit:
    return result;
}

void
Driver::UnregisterDevice(Device *pDevice) NN_NOEXCEPT
{
    UnitHandle handle;

    NN_CDACM_ABORT_IF_NULL(pDevice);

    handle = pDevice->GetHandle();

    m_DeviceHandleManager.Unregister(handle);
}


void
Driver::DestroyAllDevices() NN_NOEXCEPT
{
    uint32_t  index;

    for (index = 0; index < DeviceCountMax; index++)
    {
        if (m_Device[index].GetDeviceState() == DeviceState_Initialized)
        {
            UnregisterDevice(&m_Device[index]);
            m_Device[index].Finalize();
            break;
        }
    }
}

Result
Driver::QueueWrite(const uint8_t * pOutBuffer, uint32_t length) NN_NOEXCEPT
{
    if (m_pClientBuf == nullptr)
    {
        return ResultUnsupportedOperation();
    }

    uint64_t nUntilEnd = m_ClientBufLen - static_cast<uint64_t>(m_pStartWrite - m_pClientBuf);

    if (m_HaveAsyncData == false)
    {
        m_HaveAsyncData = true;
    }

    if (length > m_ClientBufLen)
    {
        pOutBuffer += (length - m_ClientBufLen);
        length = m_ClientBufLen;
    }

    if (length <= nUntilEnd)
    {
        std::memcpy(m_pStartWrite, pOutBuffer, length);
        m_pStartWrite += length;
        if (length == nUntilEnd)
        {
            m_pStartWrite = m_pClientBuf;
            m_HasRolledOver = true;
        }
    }
    else
    {
        m_HasRolledOver = true;
        std::memcpy(m_pStartWrite, pOutBuffer, nUntilEnd);
        std::memcpy(m_pClientBuf, pOutBuffer + nUntilEnd, length - nUntilEnd);
        m_pStartWrite = m_pClientBuf + length - nUntilEnd;
    }

    return ResultPending();
}


} // end of namespace driver
} // end of namespacecdacm
} // end of namespacenn
