﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <string>
#include <nn/nn_Macro.h>
#include <nn/fs.h>
#include <nn/fs/fs_Bis.h>
#include <nn/fs/fs_SdCardForDebug.h>
#include <nn/result/result_HandlingUtility.h>

#include "cal_FileSystem.h"

namespace nn { namespace cal {

namespace {

//!< SD カードのマウント名
const char* const SdCardFileSystemMountName = "@SdCard";

} // namespace

const char* GetRawPartitionMountName() NN_NOEXCEPT
{
    if (ExistsPartition())
    {
        return ::nn::fs::GetBisMountName(::nn::fs::BisPartitionId::CalibrationBinary);
    }
    else
    {
        return SdCardFileSystemMountName;
    }
}

const char* GetFatPartitionMountName() NN_NOEXCEPT
{
    if (ExistsPartition())
    {
        return ::nn::fs::GetBisMountName(::nn::fs::BisPartitionId::CalibrationFile);
    }
    else
    {
        return SdCardFileSystemMountName;
    }
}

void InitializeFileSystem() NN_NOEXCEPT
{
    // ファイルシステム用のアロケータを設定
    ::nn::fs::SetAllocator(AllocateForFileSystem, DeallocateForFileSystem);
}

RawPartition::RawPartition() NN_NOEXCEPT
    : m_IsMounted(false)
{
    // 何もしない
}

RawPartition::~RawPartition() NN_NOEXCEPT
{
    if (m_IsMounted)
    {
        ::nn::fs::Unmount(GetRawPartitionMountName());
    }
}

::nn::Result RawPartition::Mount() NN_NOEXCEPT
{
    if (!m_IsMounted && !ExistsPartition())
    {
        NN_RESULT_DO(::nn::fs::MountSdCardForDebug(GetRawPartitionMountName()));

        m_IsMounted = true;
    }

    NN_RESULT_SUCCESS;
}

FatPartition::FatPartition() NN_NOEXCEPT
    : m_IsMounted(false)
{
    // 何もしない
}

FatPartition::~FatPartition() NN_NOEXCEPT
{
    if (m_IsMounted)
    {
        ::nn::fs::Unmount(GetFatPartitionMountName());
    }
}

::nn::Result FatPartition::Mount() NN_NOEXCEPT
{
    if (!m_IsMounted)
    {
        if (ExistsPartition())
        {
            NN_RESULT_DO(
                ::nn::fs::MountBis(::nn::fs::BisPartitionId::CalibrationFile,
                                   nullptr));
        }
        else
        {
            NN_RESULT_DO(
                ::nn::fs::MountSdCardForDebug(GetFatPartitionMountName()));
        }

        m_IsMounted = true;
    }

    NN_RESULT_SUCCESS;
}

}}
