﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

#include <nne/max7762x/max7762x_results.h>
#include <nne/max7762x/max7762x_rtc_Types.h>
#include <nne/max7762x/max7762x_rtc_api.h>

#include <nn/bpc/bpc_Result.h>
#include <nn/bpc/bpc_ResultPrivate.h>
#include <nn/bpc/bpc_WakeupConfigTypes.h>
#include <nn/bpc/driver/bpc_Rtc.h>
#include <nn/bpc/detail/bpc_Log.h>

#include "bpc_ExternalRtc.h"
#include "bpc_WakeupTimer.h"
#include "bpc_PmicAccessor-hardware.nx.h"

namespace nn { namespace bpc { namespace driver { namespace detail {

namespace {

const int WakeupTimerResourceCountMax = 16; // TORIAEZU: システム全体でタイマーで起きたいモジュールの最大数の想定
const int InvalidTimerHandle = 0;

nn::bpc::driver::detail::WakeupTimer            g_WakeupTimer[WakeupTimerResourceCountMax];
nn::bpc::driver::detail::WakeupTimerArbiter     g_WakeupTimerArbiter;

WakeupTimerList                                 g_FreeTimerList;

// 静的コンストラクタ内で g_FreeTimerList の状態を初期化するためのクラス
class FreeTimerListInitializer
{
public:
    FreeTimerListInitializer() NN_NOEXCEPT
    {
        // g_WakeupTimer の全要素が g_FreeTimerList に繋がれているのが初期状態
        for (auto& t : g_WakeupTimer)
        {
            g_FreeTimerList.push_back(t);
        }
    }
} g_FreeTimerListInitializer;

int GenerateHandle() NN_NOEXCEPT
{
    static int s_HandleBase = 1;
    do
    {
        ++s_HandleBase;
    } while (s_HandleBase == InvalidTimerHandle);

    return s_HandleBase;
}

} // anonymous namespace

WakeupTimer* AllocateWakeupTimer() NN_NOEXCEPT
{
    if (g_FreeTimerList.empty())
    {
        return nullptr;
    }
    auto& timer = g_FreeTimerList.front();
    g_FreeTimerList.pop_front();
    timer.SetHandle(GenerateHandle());
    timer.SetTargetRtcTime(0LL);
    return &timer;
}

WakeupTimer* GetWakeupTimer(int handle) NN_NOEXCEPT
{
    if (handle == InvalidTimerHandle)
    {
        return nullptr; // Invalid handle
    }
    for (auto& t : g_WakeupTimer)
    {
        if (handle == t.GetHandle())
        {
            return &t;
        }
    }
    return nullptr; // NOT FOUND
}

void FreeWakeupTimer(WakeupTimer* pTimer) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pTimer);
    pTimer->SetHandle(InvalidTimerHandle);
    g_FreeTimerList.push_back(*pTimer);
}

WakeupTimerArbiter& GetWakeupTimerArbiter() NN_NOEXCEPT
{
    return g_WakeupTimerArbiter;
}

nn::Result WakeupTimerArbiter::Register(WakeupTimer* pTimer) NN_NOEXCEPT
{
    // アラームリストに追加
    InsertByTime(pTimer);
    NN_DETAIL_BPC_TRACE("Registered: RTC time %lld sec / type %x\n", pTimer->GetTargetRtcTime(), pTimer->GetType());

    NN_RESULT_SUCCESS;
}

nn::Result WakeupTimerArbiter::Unregister(WakeupTimer* pTimer) NN_NOEXCEPT
{
    m_RegisteredTimerList.erase(m_RegisteredTimerList.iterator_to(*pTimer));
    NN_DETAIL_BPC_TRACE("Unregistered: RTC time %lld sec / type %x\n", pTimer->GetTargetRtcTime(), pTimer->GetType());

    NN_RESULT_SUCCESS;
}

nn::Result WakeupTimerArbiter::UpdateDevice(bool *pOutIsEnabled, const WakeupTimer** ppOutEnabledTimer) NN_NOEXCEPT
{
    if (m_RegisteredTimerList.empty())
    {
        // 有効なアラームがないのでオフにする
        NN_RESULT_DO(DisableRtcAlarm(RtcAlarmType::Wakeup));
        m_LastEnabledWakeupTimerType = WakeupTimerType_None;

        if ( ppOutEnabledTimer )
        {
            *ppOutEnabledTimer = nullptr;
        }
        if (pOutIsEnabled)
        {
            *pOutIsEnabled = false;
        }
    }
    else
    {
        // 登録済リストの中でもっとも近未来なものを取ってくる
        auto& timer = m_RegisteredTimerList.front();

        // すでに過去または現在に近すぎる未来の場合は、現在時刻より少し先にセットする
        // （現在から MinWakeupIntervalInSeconds 秒後とアラーム指定時刻のうち、より未来なほうにセットする）
        // XXX: RTC 時刻レジスタが一周 (100年) すると正しい比較でなくなるが、実用上ここまで到達することはないためケアしない
        int64_t currentTime;
        NN_RESULT_DO(GetRtcTime(&currentTime));
        int64_t nearestTargetTimeAllowed = currentTime + MinWakeupIntervalInSeconds;
        int64_t targetTime = (nearestTargetTimeAllowed > timer.GetTargetRtcTime()) ?
                              nearestTargetTimeAllowed : timer.GetTargetRtcTime();

        // targetTime で RTC アラームをセットし、アラームを有効にする
        // 時刻演算によって targetTime が西暦 100 年より未来を指す場合があるが、SecondsToPmicValue の中で剰余が取られて 0-99 年に丸められる
        NN_RESULT_DO(EnableRtcAlarm(targetTime, RtcAlarmType::Wakeup));
        m_LastEnabledWakeupTimerType = timer.GetType();
        NN_DETAIL_BPC_TRACE("Enabled: RTC Time %lld (%lld sec later) / type %x\n", targetTime, targetTime - currentTime, timer.GetType());

        if ( ppOutEnabledTimer )
        {
            *ppOutEnabledTimer = &timer;
        }
        if (pOutIsEnabled)
        {
            *pOutIsEnabled = true;
        }
    }

    NN_RESULT_SUCCESS;
}

nn::Result WakeupTimerArbiter::CleanAllWakeupTimers(WakeupTimerType type) NN_NOEXCEPT
{
    if ( !m_RegisteredTimerList.empty() )
    {
        // 登録済リストからフリーリストに指定タイプの要素を全て移動
        for ( auto iter = m_RegisteredTimerList.begin(); iter != m_RegisteredTimerList.end(); )
        {
            if ( iter->GetType() == type )
            {
                auto& timer = *iter;
                iter = m_RegisteredTimerList.erase(iter); // 安全にイテレータを進める
                NN_DETAIL_BPC_TRACE("Unregistered: RTC time %lld sec / type %x\n", timer.GetTargetRtcTime(), timer.GetType());
                FreeWakeupTimer(&timer);
            }
            else
            {
                ++iter;
            }
        }

        // 操作によりリストが空になった場合、デバイス側の RTC アラームも解除
        if ( m_RegisteredTimerList.empty() )
        {
            NN_RESULT_DO(DisableRtcAlarm(RtcAlarmType::Wakeup));
        }
    }

    // この時点では m_LastEnabledWakeupTimerType は更新しないこと
    NN_RESULT_SUCCESS;
}

void WakeupTimerArbiter::InsertByTime(WakeupTimer* pTimer) NN_NOEXCEPT
{
    // XXX: RTC 時刻レジスタが一周 (100年) すると正しい比較にならず正確なソートでなくなるが、実用上ここまで到達することはないためケアしない
    for (auto iter = m_RegisteredTimerList.begin(); iter != m_RegisteredTimerList.end(); ++iter)
    {
        if ( pTimer->GetTargetRtcTime() < iter->GetTargetRtcTime() )
        {
            m_RegisteredTimerList.insert(iter, *pTimer);
            return;
        }
        else if ( pTimer->GetTargetRtcTime() == iter->GetTargetRtcTime() &&
            pTimer->GetType() >= iter->GetType() )
        {
            m_RegisteredTimerList.insert(iter, *pTimer);
            return;
        }
    }
    m_RegisteredTimerList.push_back(*pTimer);
}

}}}} // namespace nn::bpc::driver::detail
