﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/os.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/bpc/bpc_ShimInternal.h>
#include <nn/bpc/bpc_Rtc.h>
#include <nn/bpc/bpc_IRtcManager.sfdl.h>

#include "bpc_GetManagerByHipc.h"

namespace nn { namespace bpc {

namespace {

// Shim ライブラリ実装用のサービスオブジェクトへの共有ポインタ
nn::sf::SharedPointer<IRtcManager> g_RtcManager;

// Initialize の参照カウント
int g_InitializeCount = 0;

// 参照カウントを守る Mutex
struct StaticMutex
{
    nn::os::MutexType mutex;
    void lock() NN_NOEXCEPT
    {
        nn::os::LockMutex(&mutex);
    }
    void unlock() NN_NOEXCEPT
    {
        nn::os::UnlockMutex(&mutex);
    }
} g_InitializeCountMutex = { NN_OS_MUTEX_INITIALIZER(false) };

} // namespace

void InitializeRtc() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    if ( g_InitializeCount == 0 )
    {
        NN_SDK_ASSERT(!g_RtcManager);

        g_RtcManager = GetRtcManagerByHipc();
    }

    g_InitializeCount++;
}

void InitializeRtcWith(nn::sf::SharedPointer<IRtcManager> rtcManager) NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount == 0);

    g_RtcManager = rtcManager;

    g_InitializeCount++;
}

nn::sf::SharedPointer<IRtcManager> GetInternalRtcManager() NN_NOEXCEPT
{
    return g_RtcManager;
}

void FinalizeRtc() NN_NOEXCEPT
{
    std::lock_guard<StaticMutex> lock(g_InitializeCountMutex);

    NN_SDK_ASSERT(g_InitializeCount > 0);

    g_InitializeCount--;

    if ( g_InitializeCount == 0 )
    {
        NN_SDK_ASSERT_NOT_NULL(g_RtcManager);

        // 共有ポインタへの nullptr の代入で解放できる。
        g_RtcManager = nullptr;
    }
}

nn::Result GetRtcTime(int64_t *pOutRtcTimeInSeconds) NN_NOEXCEPT
{
    auto pManager = GetInternalRtcManager();
    NN_SDK_ASSERT_NOT_NULL(pManager);
    int64_t time;
    NN_RESULT_DO(pManager->GetRtcTime(&time));
    *pOutRtcTimeInSeconds = time;
    NN_RESULT_SUCCESS;
}

nn::Result SetRtcTime(int64_t rtcTimeInSeconds) NN_NOEXCEPT
{
    // TORIAEZU: デバイスの有効範囲を超えた数字でもラップアラウンドするだけで許容
    // NN_SDK_REQUIRES_RANGE(rtcTimeInSeconds, 0, MaxRtcTimeInSeconds);

    auto pManager = GetInternalRtcManager();
    NN_SDK_ASSERT_NOT_NULL(pManager);
    NN_RESULT_DO(pManager->SetRtcTime(rtcTimeInSeconds));
    NN_RESULT_SUCCESS;
}

bool GetRtcResetDetected() NN_NOEXCEPT
{
    auto pManager = GetInternalRtcManager();
    bool rtcResetDetected = false;
    NN_SDK_ASSERT_NOT_NULL(pManager);
    NN_ABORT_UNLESS_RESULT_SUCCESS(pManager->GetRtcResetDetected(&rtcResetDetected));
    return rtcResetDetected;
}

void ClearRtcResetDetected() NN_NOEXCEPT
{
    auto pManager = GetInternalRtcManager();
    NN_SDK_ASSERT(pManager);
    NN_ABORT_UNLESS_RESULT_SUCCESS(pManager->ClearRtcResetDetected());
}

void SetUpRtcResetOnShutdown() NN_NOEXCEPT
{
    auto pManager = GetInternalRtcManager();
    NN_SDK_ASSERT(pManager);
    NN_ABORT_UNLESS_RESULT_SUCCESS(pManager->SetUpRtcResetOnShutdown());
}

}} // namepsace nn::bpc
