﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/bluetooth/bluetooth_UserApi.h>
#include <nn/btm/user/btm_UserApi.h>
#include <nn/bluetooth/bluetooth_Result.h>

#include <nn/bluetooth/bluetooth_GattService.h>
#include <nn/bluetooth/bluetooth_GattCharacteristic.h>
#include <nn/bluetooth/bluetooth_GattDescriptor.h>

namespace nn { namespace bluetooth {

    GattCharacteristic::GattCharacteristic() NN_NOEXCEPT :
        GattAttribute()
    {
        m_Type = GattAttributeType_Characteristic;

        m_InstanceId = 0x0000;

        m_Property = 0x00;
    }

    GattCharacteristic::GattCharacteristic(const GattAttributeUuid& uuid, uint16_t handle, uint32_t connectionHandle,
                                           uint16_t instanceId, uint8_t property) NN_NOEXCEPT :
        GattAttribute(uuid, handle, connectionHandle)
    {
        m_Type = GattAttributeType_Characteristic;

        m_InstanceId = instanceId;

        m_Property = property;
    }

    GattCharacteristic::~GattCharacteristic() NN_NOEXCEPT
    {
        // 何もしない
    }

    uint16_t GattCharacteristic::GetInstanceId() const NN_NOEXCEPT
    {
        return m_InstanceId;
    }

    uint8_t GattCharacteristic::GetProperties() const NN_NOEXCEPT
    {
        return m_Property;
    }

    void GattCharacteristic::GetService(GattService* pOutService) const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutService);

        nn::btm::user::GattService service;

        if (nn::btm::user::GetBelongingGattService(&service, m_ConnectionHandle, m_Handle))
        {
            *pOutService = GattService(service.uuid, service.handle, m_ConnectionHandle, service.instanceId, service.endGroupHandle, service.isPrimaryService);

            return;
        }

        NN_ABORT_UNLESS_RESULT_SUCCESS(ResultAttributeNotFound());
    }

    int GattCharacteristic::GetDescriptors(GattDescriptor* pOutDescriptors, int count) const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutDescriptors);

        nn::btm::user::GattDescriptor btmDescriptors[GattAttributeCountMaxClient];

        uint8_t inNum = static_cast<uint8_t>(count);
        uint8_t descriptorNum = nn::btm::user::GetGattDescriptors(btmDescriptors, NN_ARRAY_SIZE(btmDescriptors), m_ConnectionHandle, m_Handle);

        descriptorNum = (inNum < descriptorNum) ? inNum : descriptorNum;

        for (uint8_t i = 0; i < descriptorNum; ++i)
        {
            pOutDescriptors[i] = GattDescriptor(btmDescriptors[i].uuid, btmDescriptors[i].handle, m_ConnectionHandle);
        }

        return descriptorNum;
    }

    bool GattCharacteristic::GetDescriptor(GattDescriptor* pOutDescriptor, const GattAttributeUuid& uuid) const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutDescriptor);

        nn::btm::user::GattDescriptor btmDescriptors[GattAttributeCountMaxClient];

        uint8_t descriptorNum = nn::btm::user::GetGattDescriptors(btmDescriptors, NN_ARRAY_SIZE(btmDescriptors), m_ConnectionHandle, m_Handle);

        for (uint8_t i = 0; i < descriptorNum; ++i)
        {
            if (btmDescriptors[i].uuid == uuid)
            {
                *pOutDescriptor = GattDescriptor(btmDescriptors[i].uuid, btmDescriptors[i].handle, m_ConnectionHandle);

                return true;
            }
        }

        return false;
    }

    void GattCharacteristic::SetValue(const void* pValue, size_t size) NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pValue);
        NN_ABORT_UNLESS_LESS_EQUAL(size, GattAttributeValueSizeMax);

        m_ValueLength = size;
        memcpy(m_Value, pValue, m_ValueLength);
    }

    size_t GattCharacteristic::GetValue(void* pOutValue, size_t size) const NN_NOEXCEPT
    {
        NN_ABORT_UNLESS_NOT_NULL(pOutValue);

        size_t getSize = (m_ValueLength < size) ? m_ValueLength : size;

        memcpy(pOutValue, m_Value, getSize);

        return getSize;
    }

}}  // namespace nn::bluetooth
