﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/os.h>
#include <nn/dd.h>
#include <nn/os/os_TransferMemory.h>
#include <nn/os/os_SharedMemory.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/sf_HipcClientProxyByName.h> // for nn::sf::CreateHipcProxyByName
#include <nn/sf/sf_ExpHeapAllocator.h>      // for nn::sf::ExpHeapStaticAllocator
#include <nn/applet/applet.h>

#include <nn/bluetooth/bluetooth_IBluetoothDriver.sfdl.h>
#include <nn/bluetooth/bluetooth_UserApi.h>
#include <nn/bluetooth/bluetooth_ServiceName.h>

namespace nn { namespace bluetooth { namespace user {

namespace {
nn::sf::SharedPointer<IBluetoothUser> g_User;
struct  BluetoothUserByHipcTag;
typedef nn::sf::ExpHeapStaticAllocator<1024 * 16, BluetoothUserByHipcTag> BluetoothUserAllocator;
nn::os::MutexType g_Mutex = NN_OS_MUTEX_INITIALIZER(false);
uint8_t g_InitCount = 0;
} // namespace

class BluetoothUserAllocatorInitializer
{
public:

    BluetoothUserAllocatorInitializer() NN_NOEXCEPT
    {
        BluetoothUserAllocator::Initialize(nn::lmem::CreationOption_NoOption);
    }

} g_BluetoothUserAllocatorInitializer;

void InitializeBluetoothUserInterface() NN_NOEXCEPT
{
    if (g_InitCount == 0)
    {
        NN_SDK_ASSERT(!g_User);

        nn::sf::SharedPointer<IBluetoothUser> ret;
        auto result = nn::sf::CreateHipcProxyByName<IBluetoothUser, BluetoothUserAllocator::Policy>(&ret, BluetoothUserServiceName);
        NN_ABORT_UNLESS(result.IsSuccess());

        g_User = ret;

        NN_UNUSED(result);
    }

    g_InitCount++;
}

void FinalizeBluetoothUserInterface() NN_NOEXCEPT
{
    LockMutex(&g_Mutex);
    if(g_InitCount == 0)
    {
        NN_ABORT("[bt] Initialized count over 0");
    }
    else if(g_InitCount == 1)
    {
        NN_SDK_ASSERT(g_User);
        g_User = nullptr;
        g_InitCount--;
    }
    else
    {
        g_InitCount--;
    }
    UnlockMutex(&g_Mutex);
}

nn::Result RegisterBleEvent(nn::os::SystemEventType* pSystemEvent) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pSystemEvent);
    nn::sf::NativeHandle handle;
    auto result = g_User->RegisterBleEventImpl(&handle, nn::applet::GetAppletResourceUserId());

    if (result.IsSuccess())
    {
        nn::os::AttachReadableHandleToSystemEvent(pSystemEvent, handle.GetOsHandle(), handle.IsManaged(), nn::os::EventClearMode_AutoClear);
        handle.Detach();
    }
    return result;
}

nn::Result LeClientReadCharacteristic(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, uint8_t authType) NN_NOEXCEPT
{
    auto result = g_User->LeClientReadCharacteristicImpl(
        nn::applet::GetAppletResourceUserId(),
        connId, serviceId, isPrimary, charId, authType);

    return result;
}

nn::Result LeClientReadDescriptor(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, GattId descrId, uint8_t authType) NN_NOEXCEPT
{
    auto result = g_User->LeClientReadDescriptorImpl(
        nn::applet::GetAppletResourceUserId(),
        connId, serviceId, isPrimary, charId, descrId, authType);

    return result;
}

nn::Result LeClientWriteCharacteristic(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, const uint8_t* pData, uint16_t len, uint8_t authType, bool withResponse) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);
    nn::sf::InArray<uint8_t> inArray(&pData[0], len);
    auto result = g_User->LeClientWriteCharacteristicImpl(
        nn::applet::GetAppletResourceUserId(),
        connId, serviceId, isPrimary, charId, inArray, authType, withResponse);

    return result;
}

nn::Result LeClientWriteDescriptor(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId, GattId descrId, const uint8_t* pData, uint16_t len, uint8_t authType) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);
    nn::sf::InArray<uint8_t> inArray(&pData[0], len);
    auto result = g_User->LeClientWriteDescriptorImpl(
        nn::applet::GetAppletResourceUserId(),
        connId, serviceId, isPrimary, charId, descrId, inArray, authType);

    return result;
}

nn::Result LeClientRegisterNotification(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId) NN_NOEXCEPT
{
    auto result = g_User->LeClientRegisterNotificationImpl(
        nn::applet::GetAppletResourceUserId(),
        connId, serviceId, isPrimary, charId);

    return result;
}

nn::Result LeClientDeregisterNotification(uint32_t connId, GattId serviceId, bool isPrimary, GattId charId) NN_NOEXCEPT
{
    auto result = g_User->LeClientDeregisterNotificationImpl(
        nn::applet::GetAppletResourceUserId(),
        connId, serviceId, isPrimary, charId);

    return result;
}

nn::Result SetLeResponse(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid attrUuid, const uint8_t *pData, uint16_t len) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);
    nn::sf::InArray<uint8_t> inArray(&pData[0], len);
    auto result = g_User->SetLeResponseImpl(
        nn::applet::GetAppletResourceUserId(),
        serverIf, serviceUuid, attrUuid, inArray);

    return result;
}

nn::Result LeSendIndication(uint8_t serverIf, GattAttributeUuid serviceUuid, GattAttributeUuid charUuid, const uint8_t* pData, uint8_t len, bool confirm) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pData);
    nn::sf::InArray<uint8_t> inArray(&pData[0], len);
    auto result = g_User->LeSendIndicationImpl(
        nn::applet::GetAppletResourceUserId(),
        serverIf, serviceUuid, charUuid, inArray, confirm);

    return result;
}

nn::Result GetLeEventInfo(nn::bluetooth::BleEventType *pOutEventType, uint8_t outBuffer[], uint16_t sizeOfBuffer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutEventType);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    uint32_t eventType;
    nn::sf::OutArray<uint8_t> outArray(&outBuffer[0], sizeOfBuffer);
    auto result = g_User->GetLeEventInfoImpl(
        &eventType, outArray,
        nn::applet::GetAppletResourceUserId());

    if (result.IsSuccess())
    {
        *pOutEventType = static_cast<nn::bluetooth::BleEventType>(eventType);
    }

    return result;
}

}}} // namespace nn::bluetooth::user


