﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>
#include <nn/psc.h>
#include <nn/bpc/bpc_WakeupConfig.h>

#include "bgtc_Common.h"
#include "bgtc_CompositeEventHandlerThread.h"
#include "bgtc_Instance.h"
#include "bgtc_ClientBroker.h"
#include "bgtc_RealTimeClock.h"
#include "bgtc_Settings.h"

namespace nn{ namespace bgtc{

CompositeEventHandlerThread::CompositeEventHandlerThread()
    : BackgroundStateEventHandler(m_BatterySaver)
    , ThreadBase(NN_BGTC_THREAD_CONFIG(SystemEventHandler))
    , m_LastHalfAwakeTime(0)
    , m_bIsInHalfAwake(false)
    , m_bFullAwake(true)
{
}

CompositeEventHandlerThread::~CompositeEventHandlerThread()
{
}

Result CompositeEventHandlerThread::Initialize()
{
    const psc::PmModuleId pmDependencies[] = {
        psc::PmModuleId_Bpc,
        psc::PmModuleId_Psm,
    };

    Result result = PowerStateEventHandler::Initialize(
        psc::PmModuleId_Bgtc,
        pmDependencies, sizeof(pmDependencies) / sizeof(pmDependencies[0])
    );
    if (result.IsFailure())
    {
        return result;
    }

    result = ThreadBase::Initialize();
    if (result.IsFailure())
    {
        PowerStateEventHandler::Finalize();
        return result;
    }

    return ResultSuccess();
}

void CompositeEventHandlerThread::Finalize()
{
    ThreadBase::Finalize();
    PowerStateEventHandler::Finalize();
}

void CompositeEventHandlerThread::TriggerEvent(EventCode eventCode)
{
    ClientBroker& broker = g_Instance.GetClientBroker();
    switch (eventCode)
    {
    case EventCode_EnteringHalfAwake:
        NN_BGTC_INFO("Entering half-awake. (now=% 9lld)\n", GetNow());
        m_LastHalfAwakeTime = GetNow();
        m_bIsInHalfAwake = true;
        m_TaskRunnableEvent.Signal();
        broker.SetEnableScheduleEvent(true);
        break;

    case EventCode_LeavingHalfAwake:
        m_bIsInHalfAwake = false;
        broker.SetEnableScheduleEvent(false);
        m_TaskRunnableEvent.Clear();
        NN_BGTC_INFO("Left half-awake.     (now=% 9lld, duration= %d sec)\n", GetNow(), GetNow() - m_LastHalfAwakeTime);

        // 最後に半起床になった時刻を覚えておくためにここで通知（単なる MinimumAwake の場合は更新されない）
        m_BatterySaver.NotifyFullAwakeOrHalfAwakeLeaving();
        break;

    case EventCode_EteringSleep:
        g_Instance.GetClientBroker().ResetTaskProcessingState();
        break;

    default:
        break;
    }
    BackgroundStateEventHandler::TriggerEvent(eventCode);
}

void CompositeEventHandlerThread::OnReceivePowerStateEvent(psc::PmState, psc::PmState stateTo, psc::PmFlagSet)
{
    ClientBroker& broker = g_Instance.GetClientBroker();
    switch (stateTo)
    {
    case psc::PmState_FullAwake:
        m_TaskRunnableEvent.Signal();
        m_bFullAwake = true;
        broker.SetEnableScheduleEvent(true);
        break;

    case psc::PmState_MinimumAwake:
        if (m_bFullAwake)
        {
            m_BatterySaver.NotifyFullAwakeOrHalfAwakeLeaving();
            broker.SetEnableScheduleEvent(false);
            m_TaskRunnableEvent.Clear();
        }
        break;

    case psc::PmState_SleepReady:
    {
        // 間欠起動が許可されたバッテリー・充電状態かどうかをチェック
        if (!m_BatterySaver.IsStopRequired())
        {
            Time nearestScheduled;

            // FullAwake からの（初回）遷移、かつタスクの完了が宣言されていないクライアントがいる場合は即起き
            if (m_bFullAwake && broker.IsProcessingClientExists())
            {
                NN_BGTC_INFO("Processing client(s) exist when leaving FullAwake. Next wakeup is scheduled immediately.\n");
                SetupWakeTrigger(0);
            }
            // 次にタスクの実行予約が存在するかをチェックし、最も直近の時刻までの間隔を取得
            else if (broker.GetNextScheduledTime(&nearestScheduled))
            {
                broker.ShowSchedule();

                // FullAwake からの遷移では無ければ、現在の状況に合わせた引き伸ばし処理を行う
                OperationMode mode = m_BatterySaver.JudgeOperationMode();
                const bool bBatteryAware = !m_bFullAwake;
                Interval interval = m_BatterySaver.CalculateHalfAwakeInterval(mode, nearestScheduled, bBatteryAware);

                NN_BGTC_INFO("Background operation mode is %s(charge=%d%%). From FullAwake is %s. Next wakeup is scheduled after %dsec.\n",
                    m_BatterySaver.GetOperationModeString(mode), m_BatterySaver.GetBatteryChargePercentage(),
                    m_bFullAwake ? "yes" : "no", interval);
                NN_UNUSED(mode);

                // 次回起床を予約
                SetupWakeTrigger(interval);
            }
            else
            {
                NN_BGTC_WARN("Scheduled task does not exist. Periodical awake is disabled.\n");
            }
        }
        else
        {
            NN_BGTC_WARN("Half-awake is disabled because battery level is too low to execute background tasks or forbidden by configuration.\n");
        }
        m_TaskRunnableEvent.Clear();
        TriggerEvent(EventCode_EteringSleep);

        m_bFullAwake = false;
        break;
    }
    case psc::PmState_EssentialServicesSleepReady:
    case psc::PmState_EssentialServicesAwake:
    case psc::PmState_ShutdownReady:
        break;

    case psc::PmState_Unknown:
        NN_SDK_ASSERT(false);
        break;
    } // NOLINT(style/switch_default)
}

void CompositeEventHandlerThread::ThreadBody()
{
    Result result;
    ClientBroker& cb = g_Instance.GetClientBroker();
    os::SystemEvent& eventPowerState = PowerStateEventHandler::GetEvent();
    os::TimerEvent& timerWait = BackgroundStateEventHandler::GetTimerEvent();
    os::TimerEvent& timerSchedule = cb.GetNearestScheduleEvent();
    os::TimerEvent& timerBatteryPolling = m_BatterySaver.GetChangedEvent();
    while (NN_STATIC_CONDITION(true))
    {
        int index = os::WaitAny(
            eventPowerState.GetBase(),
            timerWait.GetBase(),
            timerSchedule.GetBase(),
            timerBatteryPolling.GetBase()
        );
        if (IsExiting())
        {
            break;
        }
        switch (index)
        {
        case 0:
            eventPowerState.Clear();
            PowerStateEventHandler::Dispatch();
            break;
        case 1:
            timerWait.Clear();
            BackgroundStateEventHandler::TriggerEvent(EventCode_SignaledTimer);
            break;
        case 2:
            timerSchedule.Clear();
            if (GetLastPowerState() == psc::PmState_FullAwake
                || (GetLastPowerState() == psc::PmState_MinimumAwake && m_bIsInHalfAwake))
            {
                NN_BGTC_TRACE("Dispatchig schedule event.\n");
                cb.DispatchScheduleEvent();
            }
            else
            {
                // 利用プロセスが誤って SetSchedule を呼び出さないようにクライアントへの通知を遅延させる
                timerSchedule.StartOneShot(TimeSpan::FromSeconds(1));
                NN_BGTC_TRACE("Deferred notifications of schedule event.\n");
            }
            break;
        case 3:
            timerBatteryPolling.Clear();
            if (BackgroundStateEventHandler::GetState() != State_Idle
                && m_BatterySaver.IsStopRequired())
            {
                TriggerEvent(EventCode_StopByBatterySaver);
            }
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
        }
    }
}

void CompositeEventHandlerThread::RequestExit()
{
    BackgroundStateEventHandler::GetTimerEvent().Signal();
    ThreadBase::RequestExit();
}

void CompositeEventHandlerThread::AttachTaskRunnableEvent(bgsu::MultiClientSystemEvent::Node * pEventNode)
{
    m_TaskRunnableEvent.Attach(pEventNode);
}

void CompositeEventHandlerThread::DetachTaskRunnableEvent(bgsu::MultiClientSystemEvent::Node * pEventNode)
{
    m_TaskRunnableEvent.Detach(pEventNode);
}

Result CompositeEventHandlerThread::CheckTaskProcessable()
{
    switch (GetLastPowerState())
    {
    case psc::PmState_FullAwake:
        return ResultSuccess();

    case psc::PmState_MinimumAwake:
        if (m_bIsInHalfAwake)
        {
            return ResultSuccess();
        }
        break;

    case psc::PmState_SleepReady:
    case psc::PmState_EssentialServicesSleepReady:
    case psc::PmState_EssentialServicesAwake:
    case psc::PmState_ShutdownReady:
        break;

    case psc::PmState_Unknown:
        NN_SDK_ASSERT(false);
        break;
    } // NOLINT(style/switch_default)
    return ResultNotPermitted();
}

bool CompositeEventHandlerThread::IsHalfAwake()
{
    return GetLastPowerState() == psc::PmState_MinimumAwake && m_bIsInHalfAwake;
}

bool CompositeEventHandlerThread::IsFullAwake()
{
    return GetLastPowerState() == psc::PmState_FullAwake;
}

OperationMode CompositeEventHandlerThread::JudgeOperationMode() const
{
    return m_BatterySaver.JudgeOperationMode();
}

}
}
