﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <nn/os.h>
#include <nn/psm/psm_SystemApi.h>
#include <nn/settings/system/settings_SystemApplication.h>
#include <algorithm>

#include "bgtc_Log.h"
#include "bgtc_BatterySaver.h"
#include "bgtc_Settings.h"
#include "bgtc_RealTimeClock.h"

namespace nn{ namespace bgtc{

BatterySaver::BatterySaver()
    : m_ChangeEvent(os::EventClearMode_AutoClear)
    , m_bIsUsingSleepUnsupportedDevices(false)
{
    m_ChangeEvent.StartPeriodic(PollingInterval, PollingInterval);
}

int BatterySaver::GetBatteryChargePercentage()
{
    return psm::GetBatteryChargePercentage();
}

Interval BatterySaver::GetHalfAwakeMinimumInterval(OperationMode operationMode) const
{
    switch (operationMode)
    {
    case OperationMode_Active:
        return 0;

    case OperationMode_Balance:
        break;

    case OperationMode_Save:
    case OperationMode_Stop:
        return GetSettingsFwdbgValue<int32_t>("minimum_interval_save");
    } // NOLINT(style/switch_default)
    return GetSettingsFwdbgValue<int32_t>("minimum_interval_normal");
}

Interval BatterySaver::CalculateHalfAwakeInterval(OperationMode operationMode, Time targetSchedule, bool bBatteryAware) const
{
    Interval interval;
    if (bBatteryAware)
    {
        Interval minimumInterval = GetHalfAwakeMinimumInterval(operationMode);
        // 前回の全/半起床時間から規定時間の間隔を開けた、最速の起床時刻を求める
        Time earliestBatteryAwared = m_LatestFullAwakeOrHalfAwakeLeavingTime + minimumInterval;
        // 実際のスケジュール時刻が最速のスケジュール時刻よりも前であれば、遅延させる
        Time batteryAwaredSchedule = std::max<Time>(targetSchedule, earliestBatteryAwared);

        NN_BGTC_TRACE("targetSchedule = % 6d, minimumInterval = % 6d, earliestBatteryAwared = % 6d, batteryAwaredSchedule = % 6d\n",
            targetSchedule, minimumInterval, earliestBatteryAwared, batteryAwaredSchedule);

        interval = static_cast<Interval>(batteryAwaredSchedule - GetNow());
    }
    else
    {
        interval = static_cast<Interval>(targetSchedule - GetNow());
    }
    return std::max<Interval>(interval, 0);
}

bool BatterySaver::IsEnabled()
{
    return GetSettingsFwdbgValue<bool>("enable_battery_saver");
}

void BatterySaver::NotifyFullAwakeOrHalfAwakeLeaving()
{
    m_LatestFullAwakeOrHalfAwakeLeavingTime = GetNow();
}

bool BatterySaver::SetIsUsingSleepUnsupportedDevices(bool bIsUsingSleepUnsupportedDevices)
{
    if (m_bIsUsingSleepUnsupportedDevices != bIsUsingSleepUnsupportedDevices)
    {
        m_bIsUsingSleepUnsupportedDevices = bIsUsingSleepUnsupportedDevices;
        return true;
    }
    else
    {
        return false;
    }
}

bool BatterySaver::IsUsingSleepUnsupportedDevices() const
{
    return m_bIsUsingSleepUnsupportedDevices;
}

bool BatterySaver::ShouldCareAboutSleepUnsupportedDevices()
{
    return m_bIsUsingSleepUnsupportedDevices && JudgeInternalOperationModeByPowerState() == OperationMode_Active;
}

const char * BatterySaver::GetOperationModeString(OperationMode mode)
{
    const char* ppStrings[] =
    {
        "OperationMode_Active",
        "OperationMode_Balance",
        "OperationMode_Save",
        "OperationMode_Stop"
    };
    int index = static_cast<int>(mode);
    const int count = sizeof(ppStrings) / sizeof(ppStrings[0]);
    if (index < 0 || index >= count)
    {
        return "unknown";
    }
    return ppStrings[index];
}

OperationMode BatterySaver::JudgeInternalOperationModeByPowerState()
{
    if (!IsEnabled())
    {
        return OperationMode_Active;
    }

    psm::ChargerType chargerType = psm::GetChargerType();
    int              battery     = psm::GetBatteryChargePercentage();

    if (chargerType == psm::ChargerType_EnoughPower
        || chargerType == psm::ChargerType_LowPower)
    {
        return OperationMode_Active;
    }
    else if (battery <= GetSettingsFwdbgValue<int32_t>("battery_threshold_stop"))
    {
        return OperationMode_Stop;
    }
    else if (battery <= GetSettingsFwdbgValue<int32_t>("battery_threshold_save"))
    {
        return OperationMode_Save;
    }
    else
    {
        return OperationMode_Balance;
    }
}

OperationMode BatterySaver::JudgeOperationMode() const
{
    if (GetSettingsFwdbgValue<bool>("enable_halfawake", true)
        && !settings::system::GetQuestFlag())
    {
        return JudgeInternalOperationModeByPowerState();
    }
    else
    {
        return OperationMode_Stop;
    }
}

}}


