﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/service/bcat_ArchiveVerifier.h>
#include <nn/bcat/detail/service/core/bcat_PublicKey.h>
#include <nn/bcat/detail/service/util/bcat_SignatureVerifier.h>
#include <nn/fs.h>
#include <nn/util/util_FormatString.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace bcat { namespace service {

ArchiveVerifier::ArchiveVerifier() NN_NOEXCEPT :
    m_Destructor(nullptr),
    m_UpdateFunction(nullptr),
    m_VerifyFunction(nullptr),
    m_IsInitialized(false)
{
    NN_STATIC_ASSERT(sizeof (m_VerifierStorage) >= sizeof (detail::service::util::SignatureVerifierRsaPkcs1Sha1));
    NN_STATIC_ASSERT(sizeof (m_VerifierStorage) >= sizeof (detail::service::util::SignatureVerifierRsaPkcs1Sha256));
    NN_STATIC_ASSERT(sizeof (m_VerifierStorage) >= sizeof (detail::service::util::SignatureVerifierRsaPssSha1));
    NN_STATIC_ASSERT(sizeof (m_VerifierStorage) >= sizeof (detail::service::util::SignatureVerifierRsaPssSha256));
}

ArchiveVerifier::~ArchiveVerifier() NN_NOEXCEPT
{
    if (m_Destructor)
    {
        m_Destructor(this);
    }
}

void ArchiveVerifier::Initialize(SignatureType signatureType) NN_NOEXCEPT
{
    Initialize(signatureType,
        detail::service::core::PublicKey::GetModulus(), detail::service::core::PublicKey::GetModulusSize(),
        detail::service::core::PublicKey::GetPublicExponent(), detail::service::core::PublicKey::GetPublicExponentSize());
}

void ArchiveVerifier::Initialize(SignatureType signatureType,
    const void* pubkeyModulus, size_t pubkeyModulusSize, const void* pubkeyExponent, size_t pubkeyExponentSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pubkeyModulus);
    NN_SDK_REQUIRES(pubkeyModulusSize > 0);
    NN_SDK_REQUIRES_NOT_NULL(pubkeyExponent);
    NN_SDK_REQUIRES(pubkeyExponentSize > 0);

    NN_SDK_ASSERT(!m_IsInitialized);

    switch (signatureType)
    {
    case SignatureType_RsaPkcs1Sha1:
        {
            new (&m_VerifierStorage) detail::service::util::SignatureVerifierRsaPkcs1Sha1;

            reinterpret_cast<detail::service::util::SignatureVerifierRsaPkcs1Sha1*>
                (&m_VerifierStorage)->Initialize(pubkeyModulus, pubkeyModulusSize, pubkeyExponent, pubkeyExponentSize);

            m_Destructor = [](ArchiveVerifier* pThis)
            {
                reinterpret_cast<detail::service::util::SignatureVerifierRsaPkcs1Sha1*>(&pThis->m_VerifierStorage)->~RsaPkcs1Verifier();
            };
            m_UpdateFunction = [](ArchiveVerifier* pThis, const void* buffer, size_t size)
            {
                reinterpret_cast<detail::service::util::SignatureVerifierRsaPkcs1Sha1*>(&pThis->m_VerifierStorage)->Update(buffer, size);
            };
            m_VerifyFunction = [](ArchiveVerifier* pThis, const void* signature, size_t size)
            {
                return reinterpret_cast<detail::service::util::SignatureVerifierRsaPkcs1Sha1*>(&pThis->m_VerifierStorage)->Verify(signature, size);
            };
        }
        break;
    case SignatureType_RsaPkcs1Sha256:
        {
            new (&m_VerifierStorage) detail::service::util::SignatureVerifierRsaPkcs1Sha256;

            reinterpret_cast<detail::service::util::SignatureVerifierRsaPkcs1Sha256*>
                (&m_VerifierStorage)->Initialize(pubkeyModulus, pubkeyModulusSize, pubkeyExponent, pubkeyExponentSize);

            m_Destructor = [](ArchiveVerifier* pThis)
            {
                reinterpret_cast<detail::service::util::SignatureVerifierRsaPkcs1Sha256*>(&pThis->m_VerifierStorage)->~RsaPkcs1Verifier();
            };
            m_UpdateFunction = [](ArchiveVerifier* pThis, const void* buffer, size_t size)
            {
                reinterpret_cast<detail::service::util::SignatureVerifierRsaPkcs1Sha256*>(&pThis->m_VerifierStorage)->Update(buffer, size);
            };
            m_VerifyFunction = [](ArchiveVerifier* pThis, const void* signature, size_t size)
            {
                return reinterpret_cast<detail::service::util::SignatureVerifierRsaPkcs1Sha256*>(&pThis->m_VerifierStorage)->Verify(signature, size);
            };
        }
        break;
    case SignatureType_RsaPssSha1:
        {
            new (&m_VerifierStorage) detail::service::util::SignatureVerifierRsaPssSha1;

            reinterpret_cast<detail::service::util::SignatureVerifierRsaPssSha1*>
                (&m_VerifierStorage)->Initialize(pubkeyModulus, pubkeyModulusSize, pubkeyExponent, pubkeyExponentSize);

            m_Destructor = [](ArchiveVerifier* pThis)
            {
                reinterpret_cast<detail::service::util::SignatureVerifierRsaPssSha1*>(&pThis->m_VerifierStorage)->~RsaPssVerifier();
            };
            m_UpdateFunction = [](ArchiveVerifier* pThis, const void* buffer, size_t size)
            {
                reinterpret_cast<detail::service::util::SignatureVerifierRsaPssSha1*>(&pThis->m_VerifierStorage)->Update(buffer, size);
            };
            m_VerifyFunction = [](ArchiveVerifier* pThis, const void* signature, size_t size)
            {
                return reinterpret_cast<detail::service::util::SignatureVerifierRsaPssSha1*>(&pThis->m_VerifierStorage)->Verify(signature, size);
            };
        }
        break;
    case SignatureType_RsaPssSha256:
        {
            new (&m_VerifierStorage) detail::service::util::SignatureVerifierRsaPssSha256;

            reinterpret_cast<detail::service::util::SignatureVerifierRsaPssSha256*>
                (&m_VerifierStorage)->Initialize(pubkeyModulus, pubkeyModulusSize, pubkeyExponent, pubkeyExponentSize);

            m_Destructor = [](ArchiveVerifier* pThis)
            {
                reinterpret_cast<detail::service::util::SignatureVerifierRsaPssSha256*>(&pThis->m_VerifierStorage)->~RsaPssVerifier();
            };
            m_UpdateFunction = [](ArchiveVerifier* pThis, const void* buffer, size_t size)
            {
                reinterpret_cast<detail::service::util::SignatureVerifierRsaPssSha256*>(&pThis->m_VerifierStorage)->Update(buffer, size);
            };
            m_VerifyFunction = [](ArchiveVerifier* pThis, const void* signature, size_t size)
            {
                return reinterpret_cast<detail::service::util::SignatureVerifierRsaPssSha256*>(&pThis->m_VerifierStorage)->Verify(signature, size);
            };
        }
        break;
    default:
        NN_SDK_REQUIRES(0, "The signature type(%d) is not supported.", signatureType);
    }

    m_IsInitialized = true;
} // NOLINT(impl/function_size)

void ArchiveVerifier::Update(const void* buffer, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(buffer);

    NN_SDK_ASSERT(m_IsInitialized);

    if (size == 0)
    {
        return;
    }

    m_UpdateFunction(this, buffer, size);
}

bool ArchiveVerifier::Verify(const void* signature, size_t size) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(signature);

    NN_SDK_ASSERT(m_IsInitialized);

    if (size == 0)
    {
        return false;
    }

    return m_VerifyFunction(this, signature, size);
}

nn::Result ArchiveVerifier::VerifyFile(bool* outResult, const char* path,
    const ArchiveHeader& header, void* work, size_t workSize) NN_NOEXCEPT
{
    return VerifyFile(outResult, path, header, work, workSize,
        detail::service::core::PublicKey::GetModulus(), detail::service::core::PublicKey::GetModulusSize(),
        detail::service::core::PublicKey::GetPublicExponent(), detail::service::core::PublicKey::GetPublicExponentSize());
}

nn::Result ArchiveVerifier::VerifyFile(bool* outResult, const char* path,
    const ArchiveHeader& header, void* work, size_t workSize,
    const void* pubkeyModulus, size_t pubkeyModulusSize, const void* pubkeyExponent, size_t pubkeyExponentSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outResult);
    NN_SDK_REQUIRES_NOT_NULL(path);
    NN_SDK_REQUIRES_NOT_NULL(work);
    NN_SDK_REQUIRES(workSize > 0);

    ArchiveVerifier verifier;
    verifier.Initialize(header.signatureType, pubkeyModulus, pubkeyModulusSize, pubkeyExponent, pubkeyExponentSize);

    nn::fs::FileHandle handle = {};
    NN_RESULT_DO(nn::fs::OpenFile(&handle, path, nn::fs::OpenMode_Read));

    NN_UTIL_SCOPE_EXIT
    {
        nn::fs::CloseFile(handle);
    };

    int64_t offset = 0;

    while (NN_STATIC_CONDITION(true))
    {
        size_t read;
        NN_RESULT_DO(nn::fs::ReadFile(&read, handle, offset, work, workSize));

        if (read == 0)
        {
            break;
        }

        verifier.Update(work, read);
        offset += read;
    }

    *outResult = verifier.Verify(header.signature, sizeof (header.signature));

    NN_RESULT_SUCCESS;
}

}}}
