﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/msgpack/bcat_MsgpackReader.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/util/util_StringUtil.h>
#include <nn/nn_SdkAssert.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace msgpack {

MsgpackReader::MsgpackReader() NN_NOEXCEPT
{
}

nn::Result MsgpackReader::Parse(nne::nlib::InputStream& stream, MsgpackEventHandler& handler, size_t tokenSize) NN_NOEXCEPT
{
    nne::nlib::msgpack::JsonStreamParserSettings parserSettings;

    parserSettings.format = nne::nlib::msgpack::JsonStreamParserSettings::kFormatMsgpack;
    parserSettings.max_depth = 12;
    parserSettings.token_buffer_size = tokenSize;

    nne::nlib::msgpack::JsonStreamParser parser;

    parser.Init(parserSettings);
    parser.Open(&stream);

    NN_UTIL_SCOPE_EXIT
    {
        parser.Close();
    };

    return ParseImpl(parser, handler);
}

nn::Result MsgpackReader::ParseImpl(nne::nlib::msgpack::JsonStreamParser& parser, MsgpackEventHandler& handler) NN_NOEXCEPT
{
    while (parser.HasNext())
    {
        nne::nlib::msgpack::JsonStreamParser::Event event = parser.Next();

        switch (event)
        {
        case nne::nlib::msgpack::JsonStreamParser::kEventTrue:
        case nne::nlib::msgpack::JsonStreamParser::kEventFalse:
            {
                m_JsonPath.UpdateCurrent();

                if (!handler.OnBoolean(m_JsonPath, event == nne::nlib::msgpack::JsonStreamParser::kEventTrue))
                {
                    NN_RESULT_THROW(ResultAbortedByCallback());
                }
                m_JsonPath.PopIfKey();
            }
            break;
        case nne::nlib::msgpack::JsonStreamParser::kEventString:
            {
                m_JsonPath.UpdateCurrent();

                if (!handler.OnString(m_JsonPath,
                    parser.GetToken().buf, parser.GetToken().number.size, parser.GetToken().token_toobig))
                {
                    NN_RESULT_THROW(ResultAbortedByCallback());
                }
                m_JsonPath.PopIfKey();
            }
            break;
        case nne::nlib::msgpack::JsonStreamParser::kEventNumberInt32:
        case nne::nlib::msgpack::JsonStreamParser::kEventNumberInt64:
            {
                AnyInteger integer;

                nne::nlib::msgpack::JsonStreamParser::ToInt64(parser.GetToken(), &integer.s);
                integer.isSigned = true;

                m_JsonPath.UpdateCurrent();

                if (!handler.OnInteger(m_JsonPath, integer))
                {
                    NN_RESULT_THROW(ResultAbortedByCallback());
                }
                m_JsonPath.PopIfKey();
            }
            break;
        case nne::nlib::msgpack::JsonStreamParser::kEventNumberUint32:
        case nne::nlib::msgpack::JsonStreamParser::kEventNumberUint64:
            {
                AnyInteger integer;

                nne::nlib::msgpack::JsonStreamParser::ToUint64(parser.GetToken(), &integer.u);
                integer.isSigned = false;

                m_JsonPath.UpdateCurrent();

                if (!handler.OnInteger(m_JsonPath, integer))
                {
                    NN_RESULT_THROW(ResultAbortedByCallback());
                }
                m_JsonPath.PopIfKey();
            }
            break;
        case nne::nlib::msgpack::JsonStreamParser::kEventStartArray:
            {
                if (!m_JsonPath.TryPushArray() || !handler.OnStartArray(m_JsonPath))
                {
                    NN_RESULT_THROW(ResultAbortedByCallback());
                }
            }
            break;
        case nne::nlib::msgpack::JsonStreamParser::kEventEndArray:
            {
                m_JsonPath.Pop();

                if (!handler.OnEndArray(m_JsonPath))
                {
                    NN_RESULT_THROW(ResultAbortedByCallback());
                }
                m_JsonPath.PopIfKey();
            }
            break;
        case nne::nlib::msgpack::JsonStreamParser::kEventStartMap:
            {
                if (!m_JsonPath.TryPushObject() || !handler.OnStartObject(m_JsonPath))
                {
                    NN_RESULT_THROW(ResultAbortedByCallback());
                }
            }
            break;
        case nne::nlib::msgpack::JsonStreamParser::kEventEndMap:
            {
                m_JsonPath.Pop();

                if (!handler.OnEndObject(m_JsonPath))
                {
                    NN_RESULT_THROW(ResultAbortedByCallback());
                }
                m_JsonPath.PopIfKey();
            }
            break;
        case nne::nlib::msgpack::JsonStreamParser::kEventKeyName:
            {
                if (!m_JsonPath.TryPushKey(parser.GetToken().buf, parser.GetToken().number.size, parser.GetToken().token_toobig))
                {
                    NN_RESULT_THROW(ResultAbortedByCallback());
                }
            }
            break;
        case nne::nlib::msgpack::JsonStreamParser::kEventBinary:
            {
                m_JsonPath.UpdateCurrent();

                if (!handler.OnBinary(m_JsonPath,
                    parser.GetToken().buf, parser.GetToken().number.size, parser.GetToken().token_toobig))
                {
                    NN_RESULT_THROW(ResultAbortedByCallback());
                }
                m_JsonPath.PopIfKey();
            }
            break;
        case nne::nlib::msgpack::JsonStreamParser::kEventExt:
            {
                m_JsonPath.UpdateCurrent();

                if (!handler.OnExtension(m_JsonPath, parser.GetToken().ext,
                    parser.GetToken().buf, parser.GetToken().number.size, parser.GetToken().token_toobig))
                {
                    NN_RESULT_THROW(ResultAbortedByCallback());
                }
                m_JsonPath.PopIfKey();
            }
            break;
        default:
            break;
        }
    }

    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

}}}}}
