﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/core/bcat_ExecutionQueue.h>
#include <nn/bcat/detail/service/core/bcat_DeliveryCacheProgressManager.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

ExecutionQueue::ExecutionQueue() NN_NOEXCEPT :
    m_Mutex(true),
    m_Event(nn::os::EventClearMode_ManualClear),
    m_Count(0)
{
}

void ExecutionQueue::Clear() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_Count = 0;
}

bool ExecutionQueue::AddToHead(TaskId* outTaskId, SessionId sessionId, nn::ApplicationId appId, uint32_t appVersion,
    const char* dirName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outTaskId);
    NN_SDK_REQUIRES_NOT_EQUAL(sessionId, InvalidSessionId);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    DirectoryName d = {};

    if (dirName)
    {
        int length = nn::util::Strlcpy(d.value, dirName, sizeof (d.value));

        NN_SDK_ASSERT(length < sizeof (d.value) && d.IsValid());
        NN_UNUSED(length);
    }

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    TaskId taskId = IdIssuer::GetInstance().IssueTaskId();

    if (m_Count == NN_ARRAY_SIZE(m_Items))
    {
        return false;
    }

    if (m_Count > 0)
    {
        std::memmove(&m_Items[1], &m_Items[0], sizeof (Item) * m_Count);
    }

    Item item = {taskId, sessionId, appId, appVersion, d};

    m_Items[0] = item;
    m_Count++;

    *outTaskId = taskId;

    DeliveryCacheProgressManager::GetInstance().Register(taskId);

    m_Event.Signal();

    return true;
}

bool ExecutionQueue::AddToTail(TaskId* outTaskId, SessionId sessionId, nn::ApplicationId appId, uint32_t appVersion,
    const char* dirName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outTaskId);
    NN_SDK_REQUIRES_NOT_EQUAL(sessionId, InvalidSessionId);
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());

    DirectoryName d = {};

    if (dirName)
    {
        int length = nn::util::Strlcpy(d.value, dirName, sizeof (d.value));

        NN_SDK_ASSERT(length < sizeof (d.value) && d.IsValid());
        NN_UNUSED(length);
    }

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    TaskId taskId = IdIssuer::GetInstance().IssueTaskId();

    if (m_Count == NN_ARRAY_SIZE(m_Items))
    {
        return false;
    }

    Item item = {taskId, sessionId, appId, appVersion, d};

    m_Items[m_Count++] = item;

    *outTaskId = taskId;

    DeliveryCacheProgressManager::GetInstance().Register(taskId);

    m_Event.Signal();

    return true;
}

void ExecutionQueue::Remove(TaskId taskId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(taskId, InvalidTaskId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = Search(taskId);

    if (index == -1)
    {
        return;
    }

    RemoveItem(index);
}

void ExecutionQueue::CancelAndRemove(TaskId taskId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(taskId, InvalidTaskId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = Search(taskId);

    if (index == -1)
    {
        return;
    }

    DeliveryCacheProgressImpl progress = {};

    progress.NotifyDone(ResultCanceledByUser());
    DeliveryCacheProgressManager::GetInstance().SetProgress(taskId, progress);

    RemoveItem(index);
}

bool ExecutionQueue::Pop(Item* outItem) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outItem);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (m_Count == 0)
    {
        return false;
    }

    *outItem = m_Items[0];

    RemoveItem(0);

    return true;
}

bool ExecutionQueue::IsEmpty() const NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    return (m_Count == 0);
}

bool ExecutionQueue::Exists(TaskId taskId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(taskId, InvalidTaskId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    return (Search(taskId) != -1);
}

int ExecutionQueue::GetCount() const NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    return m_Count;
}

nn::os::Event& ExecutionQueue::GetEvent() NN_NOEXCEPT
{
    return m_Event;
}

int ExecutionQueue::Search(TaskId taskId) const NN_NOEXCEPT
{
    for (int i = 0; i < m_Count; i++)
    {
        if (taskId == m_Items[i].taskId)
        {
            return i;
        }
    }

    return -1;
}

void ExecutionQueue::RemoveItem(int index) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER(m_Count, 0);

    int moveCount = m_Count - 1 - index;

    if (moveCount > 0)
    {
        std::memmove(&m_Items[index], &m_Items[index + 1], sizeof (Item) * moveCount);
    }

    m_Count--;
}

}}}}}
