﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/core/bcat_DeliveryListSubtractor.h>
#include <nn/bcat/detail/service/msgpack/bcat_FileInputStream.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

DeliveryListSubtractor::DeliveryListSubtractor() NN_NOEXCEPT :
    m_Mutex(true),
    m_AppId(nn::ApplicationId::GetInvalidId()),
    m_AppVersion(0),
    m_TargetDirectory(nullptr),
    m_DirCount(0),
    m_WholeDownloadSize(0),
    m_IncreasingDirCount(0),
    m_ReducingDirCount(0)
{
}

void DeliveryListSubtractor::SetTargetDirectory(const char* dirName) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    m_TargetDirectory = dirName;
}

nn::Result DeliveryListSubtractor::Analyze(nn::ApplicationId appId, uint32_t appVersion, void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(appId, nn::ApplicationId::GetInvalidId());
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES_GREATER(bufferSize, 0u);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    NN_DETAIL_BCAT_INFO("[bcat] DeliveryListSubtractor::Analyze(%016llx) ...\n", appId.value);

    m_AppId = appId;
    m_AppVersion = appVersion;

    NN_RESULT_DO(StepAnalyze(buffer, bufferSize));

    NN_RESULT_SUCCESS;
}

int64_t DeliveryListSubtractor::GetWholeDownloadSize() const NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    return m_WholeDownloadSize;
}

void DeliveryListSubtractor::GetDirectoryView(DirectoryMetaView* outMetaView, DirectoryDiffView* outDiffView) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outMetaView);
    NN_SDK_REQUIRES_NOT_NULL(outDiffView);

    *outMetaView = m_DirectoryMetaView;
    *outDiffView = m_DirectoryDiffView;
}

void DeliveryListSubtractor::GetDirectoryList(int* outCount, ListDirectory* dirs, int count) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(dirs);
    NN_SDK_REQUIRES_GREATER(count, 0);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int actualCount = 0;

    for (int i = 0; i < m_DirCount; i++)
    {
        if (count-- == 0)
        {
            break;
        }
        dirs[actualCount++] = m_Dirs[i];
    }

    *outCount = actualCount;
}

void DeliveryListSubtractor::GetIncreasingDirectoryIndexList(int* outCount, int* indexes, int count) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(indexes);
    NN_SDK_REQUIRES_GREATER(count, 0);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int actualCount = 0;

    for (int i = 0; i < m_IncreasingDirCount; i++)
    {
        if (count-- == 0)
        {
            break;
        }
        indexes[actualCount++] = m_IncreasingDirIndexes[i];
    }

    *outCount = actualCount;
}

void DeliveryListSubtractor::GetReducingDirectoryIndexList(int* outCount, int* indexes, int count) const NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(indexes);
    NN_SDK_REQUIRES_GREATER(count, 0);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int actualCount = 0;

    for (int i = 0; i < m_ReducingDirCount; i++)
    {
        if (count-- == 0)
        {
            break;
        }
        indexes[actualCount++] = m_ReducingDirIndexes[i];
    }

    *outCount = actualCount;
}

nn::Result DeliveryListSubtractor::StepAnalyze(void* buffer, size_t bufferSize) NN_NOEXCEPT
{
    NN_DETAIL_BCAT_INFO("[bcat] - DeliveryListSubtractor::StepAnalyze ...\n");

    NN_RESULT_DO(DirectoryMetaView::LoadFromDeliveryCacheStorage(&m_DirectoryMetaView, m_AppId));

    char listPath[64] = {};
    DeliveryCacheStorageManager::GetInstance().MakeListPath(listPath, sizeof (listPath), m_AppId);

    detail::service::msgpack::FileInputStream stream;

    stream.SetBuffer(buffer, bufferSize);

    NN_RESULT_DO(stream.Open(listPath));

    m_DirCount = 0;
    m_WholeDownloadSize = 0;

    DeliveryListReader reader;
    ListHeader header;

    reader.SetupForWalking(AnalyzeCallback, this, m_WalkWorkBuffer);

    NN_RESULT_TRY(reader.Read(&header, stream))
        NN_RESULT_CATCH(ResultInvalidFormat)
        {
            // リストフォーマットに互換のない更新が入ったため、不正なフォーマット扱いとなる。
            NN_DETAIL_BCAT_WARN("[bcat] The list format is invalid.\n");
            NN_RESULT_RETHROW;
        }
    NN_RESULT_END_TRY;

    NN_RESULT_THROW_UNLESS(header.requiredAppVersion <= m_AppVersion, ResultApplicationUpdateRequired());
    NN_RESULT_THROW_UNLESS(header.isInService, ResultApplicationServiceExpired());

    char naRequiredPath[64] = {};
    DeliveryCacheStorageManager::GetInstance().MakeNintendoAccountRequiredPath(naRequiredPath, sizeof (naRequiredPath), m_AppId);

    bool isWritten = true;

    if (header.isNintendoAccountRequired)
    {
        NN_RESULT_TRY(FileSystem::CreateFile(naRequiredPath, 0, false))
            NN_RESULT_CATCH(nn::fs::ResultPathAlreadyExists)
            {
                isWritten = false;
            }
        NN_RESULT_END_TRY;
    }
    else
    {
        NN_RESULT_TRY(nn::fs::DeleteFile(naRequiredPath))
            NN_RESULT_CATCH(nn::fs::ResultPathNotFound)
            {
                isWritten = false;
            }
        NN_RESULT_END_TRY;
    }

    if (isWritten)
    {
        NN_ABORT_UNLESS_RESULT_SUCCESS(DeliveryCacheStorageManager::GetInstance().Commit(m_AppId));
    }

    if (header.isNintendoAccountRequired)
    {
        NN_RESULT_THROW_UNLESS(detail::service::util::Account::IsNetworkServiceAccountAvailable(), ResultNintendoAccountNotLinked());
    }

    NN_DETAIL_BCAT_INFO("[bcat] Whole download size = %lld\n", m_WholeDownloadSize);

    NN_RESULT_DO(m_DirectoryDiffView.Make(m_DirectoryMetaView, m_Dirs, m_DirCount));

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryListSubtractor::AnalyzeCallback(const ListDirectory& dir, const ListFile* files, int count) NN_NOEXCEPT
{
    if (m_DirCount >= NN_ARRAY_SIZE(m_Dirs))
    {
        NN_RESULT_SUCCESS;
    }

    int dirIndex = m_DirCount++;

    m_Dirs[dirIndex] = dir;

    // ダウンロード対象かどうかの確認。
    {
        DirectoryMeta entry = {};

        if (m_DirectoryMetaView.Search(nullptr, &entry, dir.name.value).IsSuccess())
        {
            // 更新がなければ何もしない。
            if (std::memcmp(entry.digest.value, dir.digest.value, sizeof (entry.digest.value)) == 0)
            {
                NN_RESULT_SUCCESS;
            }
        }
    }

    NN_RESULT_DO(FileMetaView::LoadFromDeliveryCacheStorage(&m_FileMetaView, m_AppId, dir.name.value));
    NN_RESULT_DO(m_FileDiffView.Make(m_FileMetaView, files, count));

    int64_t dirDiffSize = 0;

    int fileIndexes[DeliveryCacheFileCountMaxPerDirectory] = {};
    int fileCount = 0;

    m_FileDiffView.GetIndexListFromRemote(&fileCount, fileIndexes, NN_ARRAY_SIZE(fileIndexes), DiffType_Add);

    if (fileCount < NN_ARRAY_SIZE(fileIndexes))
    {
        int c = 0;

        m_FileDiffView.GetIndexListFromRemote(&c, fileIndexes + fileCount, NN_ARRAY_SIZE(fileIndexes) - fileCount, DiffType_Update);
        fileCount += c;
    }

    int64_t dirDownloadSize = 0;

    for (int i = 0; i < fileCount; i++)
    {
        dirDownloadSize += files[fileIndexes[i]].size;
        dirDiffSize += FileSystem::CalculateFileSizeOnStorage(files[fileIndexes[i]].size);
    }

    if (m_TargetDirectory)
    {
        if (nn::util::Strnicmp(m_TargetDirectory, dir.name.value, sizeof (dir.name.value)) == 0)
        {
            m_WholeDownloadSize = dirDownloadSize;
        }
    }
    else
    {
        m_WholeDownloadSize += dirDownloadSize;
    }

    m_FileDiffView.GetIndexListFromLocal(&fileCount, fileIndexes, NN_ARRAY_SIZE(fileIndexes), DiffType_Remove);

    if (fileCount < NN_ARRAY_SIZE(fileIndexes))
    {
        int c = 0;

        m_FileDiffView.GetIndexListFromLocal(&c, fileIndexes + fileCount, NN_ARRAY_SIZE(fileIndexes) - fileCount, DiffType_Update);
        fileCount += c;
    }

    for (int i = 0; i < fileCount; i++)
    {
        FileMeta entry = {};
        m_FileMetaView.GetEntry(&entry, fileIndexes[i]);

        dirDiffSize -= FileSystem::CalculateFileSizeOnStorage(entry.size);
    }

    if (dirDiffSize >= 0)
    {
        m_IncreasingDirIndexes[m_IncreasingDirCount++] = dirIndex;
    }
    else
    {
        m_ReducingDirIndexes[m_ReducingDirCount++] = dirIndex;
    }

    NN_RESULT_SUCCESS;
}

void DeliveryListSubtractor::AnalyzeCallback(const ListDirectory& dir, const ListFile* files, int count, void* param) NN_NOEXCEPT
{
    reinterpret_cast<DeliveryListSubtractor*>(param)->AnalyzeCallback(dir, files, count);
}

}}}}}
