﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/core/bcat_DeliveryCacheProgressManager.h>

namespace nn { namespace bcat { namespace detail { namespace service { namespace core {

DeliveryCacheProgressManager::DeliveryCacheProgressManager() NN_NOEXCEPT :
    m_Mutex(true)
{
    std::memset(m_Records, 0, sizeof (m_Records));
}

DeliveryCacheProgressManager::~DeliveryCacheProgressManager() NN_NOEXCEPT
{
    for (int i = 0; i < CountMax; i++)
    {
        if (m_Records[i].taskId != InvalidTaskId && m_Records[i].isSystemEventAttached)
        {
            nn::os::DestroySystemEvent(&m_Records[i].systemEvent);
        }
    }
}

void DeliveryCacheProgressManager::Register(TaskId taskId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(taskId, InvalidTaskId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    for (int i = 0; i < CountMax; i++)
    {
        if (m_Records[i].taskId == InvalidTaskId)
        {
            std::memset(&m_Records[i], 0, sizeof (m_Records[i]));
            m_Records[i].taskId = taskId;
            break;
        }
    }
}

void DeliveryCacheProgressManager::Unregister(TaskId taskId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(taskId, InvalidTaskId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = Search(taskId);

    NN_SDK_ASSERT(index != -1);

    if (m_Records[index].isSystemEventAttached)
    {
        nn::os::DestroySystemEvent(&m_Records[index].systemEvent);
    }

    m_Records[index].taskId = InvalidTaskId;
}

void DeliveryCacheProgressManager::AttachUpdateDetectionEvent(TaskId taskId, nn::os::NativeHandle eventHandle) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(taskId, InvalidTaskId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = Search(taskId);

    NN_SDK_ASSERT(index != -1);

    nn::os::AttachWritableHandleToSystemEvent(&m_Records[index].systemEvent, eventHandle, false, nn::os::EventClearMode_ManualClear);
    m_Records[index].isSystemEventAttached = true;
}

void DeliveryCacheProgressManager::GetProgress(DeliveryCacheProgressImpl* outProgress, TaskId taskId) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outProgress);
    NN_SDK_REQUIRES_NOT_EQUAL(taskId, InvalidTaskId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    int index = Search(taskId);

    NN_SDK_ASSERT(index != -1);

    outProgress->status = m_Records[index].status;
    outProgress->result = m_Records[index].result;
    outProgress->dirName = m_Records[index].dirName;
    outProgress->fileName = m_Records[index].fileName;
    outProgress->currentDownloaded = m_Records[index].currentDownloaded;
    outProgress->currentTotal = m_Records[index].currentTotal;
    outProgress->wholeDownloaded = m_Records[index].wholeDownloaded;
    outProgress->wholeTotal = m_Records[index].wholeTotal;
}

void DeliveryCacheProgressManager::SetProgress(TaskId taskId, const DeliveryCacheProgressImpl& progress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_EQUAL(taskId, InvalidTaskId);

    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    // NN_DETAIL_BCAT_INFO("[bcat] Progress: status = %d, current = (%s/%s, %lld/%lld), whole = (%lld/%lld)\n",
    //     progress.status, progress.dirName.value, progress.fileName.value,
    //     progress.currentDownloaded, progress.currentTotal, progress.wholeDownloaded, progress.wholeTotal);

    NN_SDK_ASSERT_LESS_EQUAL(progress.currentDownloaded, progress.currentTotal);
    NN_SDK_ASSERT_LESS_EQUAL(progress.currentDownloaded + progress.wholeDownloaded, progress.wholeTotal);

    int index = Search(taskId);

    if (index != -1)
    {
        m_Records[index].status = progress.status;
        m_Records[index].result = progress.result;
        m_Records[index].dirName = progress.dirName;
        m_Records[index].fileName = progress.fileName;
        m_Records[index].currentDownloaded = progress.currentDownloaded;
        m_Records[index].currentTotal = progress.currentTotal;
        m_Records[index].wholeDownloaded = progress.wholeDownloaded;
        m_Records[index].wholeTotal = progress.wholeTotal;

        Signal(index);
    }
}

int DeliveryCacheProgressManager::Search(TaskId taskId) const NN_NOEXCEPT
{
    for (int i = 0; i < CountMax; i++)
    {
        if (taskId == m_Records[i].taskId)
        {
            return i;
        }
    }

    return -1;
}

void DeliveryCacheProgressManager::Signal(int index) NN_NOEXCEPT
{
    if (m_Records[index].isSystemEventAttached)
    {
        nn::os::SignalSystemEvent(&m_Records[index].systemEvent);
    }
}

}}}}}
