﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/detail/service/bcat_ServiceCreator.h>
#include <nn/bcat/detail/service/bcat_BcatService.h>
#include <nn/bcat/detail/service/bcat_DeliveryCacheStorageService.h>
#include <nn/bcat/detail/service/bcat_Common.h>
#include <nn/bcat/detail/service/bcat_ServiceMemoryManager.h>
#include <nn/sf/sf_ObjectFactory.h>
#include <nn/arp/arp_Api.h>
#include <nn/ns/ns_ApplicationControlDataApi.h>

#define CHECK_CAPABILITY(flag) \
    NN_RESULT_THROW_UNLESS(m_Capability.IsPermitted(flag), ResultNotPermitted())

namespace nn { namespace bcat { namespace detail { namespace service {

namespace
{
    nn::os::SdkMutexType g_Mutex = NN_OS_SDK_MUTEX_INITIALIZER();

    nn::ns::ApplicationControlProperty g_AppControlProperty;
};

ServiceCreator::ServiceCreator(const char* serviceName, const Capability& capability) NN_NOEXCEPT :
    m_ServiceName(serviceName),
    m_Capability(capability)
{
    NN_UNUSED(m_ServiceName);
}

ServiceCreator::~ServiceCreator() NN_NOEXCEPT
{
}

nn::Result ServiceCreator::CreateBcatService(nn::sf::Out<nn::sf::SharedPointer<nn::bcat::detail::ipc::IBcatService>> outService, nn::Bit64 processId) NN_NOEXCEPT
{
    nn::os::ProcessId osProcessId = {processId};

    nn::ApplicationId appId = nn::ApplicationId::GetInvalidId();
    uint32_t appVersion = 0;

    char passphrase[PassphraseLengthMax + 1] = {};

    {
        nn::arp::ApplicationLaunchProperty appLaunchProperty = {};

        if (nn::arp::GetApplicationLaunchProperty(&appLaunchProperty, osProcessId).IsSuccess())
        {
            appId.value = appLaunchProperty.id.value;

            // appLaunchProperty.version は (ReleaseVersion << 16 | PrivateVersion) で構成される。
            // BCAT システムは ReleaseVersion のみを利用する。
            appVersion = (appLaunchProperty.version) >> 16;
        }
    }

    if (appId != nn::ApplicationId::GetInvalidId())
    {
        std::lock_guard<decltype (g_Mutex)> lock(g_Mutex);

        // アプリケーション管理情報（nmeta）からパスフレーズを取得する。
        if (nn::arp::GetApplicationControlProperty(&g_AppControlProperty, osProcessId).IsSuccess())
        {
            nn::util::Strlcpy(passphrase, g_AppControlProperty.bcatPassphrase, sizeof (passphrase));
        }
    }

    // アプリケーションである、かつ、パスフレーズが設定されている場合、データ配信機能を利用しているとみなす。
    if (appId != nn::ApplicationId::GetInvalidId() && passphrase[0] != '\0')
    {
        NN_RESULT_DO(detail::service::core::DeliveryCacheStorageManager::GetInstance().RequestSavePassphrase(appId, passphrase));
        NN_RESULT_DO(CreateBcatServiceImpl(outService, appId, appVersion));
    }
    else
    {
        NN_RESULT_DO(CreateBcatServiceImpl(outService));
    }

    NN_RESULT_SUCCESS;
}

nn::Result ServiceCreator::CreateDeliveryCacheStorageService(nn::sf::Out<nn::sf::SharedPointer<nn::bcat::detail::ipc::IDeliveryCacheStorageService>> outService, nn::Bit64 processId) NN_NOEXCEPT
{
    nn::os::ProcessId osProcessId = {processId};

    nn::arp::ApplicationLaunchProperty appLaunchProperty;

    NN_RESULT_TRY(nn::arp::GetApplicationLaunchProperty(&appLaunchProperty, osProcessId))
        NN_RESULT_CATCH_ALL
        {
            NN_RESULT_THROW(ResultNotFound());
        }
    NN_RESULT_END_TRY;

    nn::ApplicationId appId = {appLaunchProperty.id.value};

    NN_RESULT_DO(CreateDeliveryCacheStorageServiceImpl(outService, appId));

    NN_RESULT_SUCCESS;
}

nn::Result ServiceCreator::CreateDeliveryCacheStorageServiceWithApplicationId(nn::sf::Out<nn::sf::SharedPointer<nn::bcat::detail::ipc::IDeliveryCacheStorageService>> outService, nn::ApplicationId appId) NN_NOEXCEPT
{
    CHECK_CAPABILITY(Capability::Flag_System);

    NN_RESULT_DO(CreateDeliveryCacheStorageServiceImpl(outService, appId));

    NN_RESULT_SUCCESS;
}

nn::Result ServiceCreator::CreateBcatServiceImpl(nn::sf::Out<nn::sf::SharedPointer<nn::bcat::detail::ipc::IBcatService>> outService) NN_NOEXCEPT
{
    auto& allocator = ServiceMemoryManager::GetInstance().GetBcatServiceAllocator();

    auto p = nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::
        CreateSharedEmplaced<detail::ipc::IBcatService, BcatService>(&allocator, m_Capability);

    // NN_DETAIL_BCAT_INFO("[bcat] Create bcat service. (%s)\n", m_ServiceName ? m_ServiceName : "");

    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionResource());

    outService.Set(std::move(p));

    NN_RESULT_SUCCESS;
}

nn::Result ServiceCreator::CreateBcatServiceImpl(nn::sf::Out<nn::sf::SharedPointer<nn::bcat::detail::ipc::IBcatService>> outService, nn::ApplicationId appId, uint32_t appVersion) NN_NOEXCEPT
{
    auto& allocator = ServiceMemoryManager::GetInstance().GetBcatServiceAllocator();

    auto p = nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::
        CreateSharedEmplaced<detail::ipc::IBcatService, BcatService>(&allocator, appId, appVersion, m_Capability);

    // NN_DETAIL_BCAT_INFO("[bcat] Create bcat service. (%s)\n", m_ServiceName ? m_ServiceName : "");

    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionResource());

    outService.Set(std::move(p));

    NN_RESULT_SUCCESS;
}

nn::Result ServiceCreator::CreateDeliveryCacheStorageServiceImpl(nn::sf::Out<nn::sf::SharedPointer<nn::bcat::detail::ipc::IDeliveryCacheStorageService>> outService, nn::ApplicationId appId) NN_NOEXCEPT
{
    NN_RESULT_DO(detail::service::core::DeliveryCacheStorageManager::GetInstance().Mount(appId));

    bool isSuccess = false;

    NN_UTIL_SCOPE_EXIT
    {
        if (!isSuccess)
        {
            detail::service::core::DeliveryCacheStorageManager::GetInstance().Unmount(appId);
        }
    };

    char naRequiredPath[64] = {};
    detail::service::core::DeliveryCacheStorageManager::GetInstance().MakeNintendoAccountRequiredPath(naRequiredPath, sizeof (naRequiredPath), appId);

    if (detail::service::core::FileSystem::Exists(naRequiredPath))
    {
        NN_RESULT_THROW_UNLESS(detail::service::util::Account::IsNetworkServiceAccountAvailable(), ResultNintendoAccountNotLinked());
    }

    auto& allocator = ServiceMemoryManager::GetInstance().GetDeliveryCacheStorageServiceAllocator();

    auto p = nn::sf::ObjectFactory<nn::sf::ExpHeapAllocator::Policy>::
        CreateSharedEmplaced<detail::ipc::IDeliveryCacheStorageService, DeliveryCacheStorageService>(&allocator, appId, m_Capability);

    // NN_DETAIL_BCAT_INFO("[bcat] Create delivery cache storage service. (%s)\n", m_ServiceName ? m_ServiceName : "");

    NN_RESULT_THROW_UNLESS(p, ResultOutOfSessionResource());

    outService.Set(std::move(p));

    isSuccess = true;

    NN_RESULT_SUCCESS;
}

}}}}
