﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/bcat_DeliveryCacheProgress.h>
#include <nn/bcat/detail/bcat_TypesDetail.h>
#include <nn/bcat/detail/bcat_ErrorHandler.h>
#include <nn/bcat/detail/bcat_ShimLibraryGlobal.h>
#include <mutex>

#define GET_IMPL() reinterpret_cast<detail::DeliveryCacheProgressImpl*>(m_Storage)
#define CONST_GET_IMPL() reinterpret_cast<const detail::DeliveryCacheProgressImpl*>(m_Storage)

namespace nn { namespace bcat {

DeliveryCacheProgress::DeliveryCacheProgress() NN_NOEXCEPT :
    m_Mutex(true),
    m_Handle(nullptr)
{
    NN_STATIC_ASSERT(sizeof (detail::DeliveryCacheProgressImpl) == sizeof (m_Storage));

    std::memset(m_Storage, 0, sizeof (m_Storage));
}

DeliveryCacheProgress::~DeliveryCacheProgress() NN_NOEXCEPT
{
    Detach();
}

void DeliveryCacheProgress::Update() NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    if (!m_Handle)
    {
        return;
    }
    if (!m_Event.TryWait())
    {
        return;
    }

    m_Event.Clear();

    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Handle->GetImpl(GET_IMPL()));
}

DeliveryCacheProgressStatus DeliveryCacheProgress::GetStatus() const NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    return static_cast<DeliveryCacheProgressStatus>(CONST_GET_IMPL()->status);
}

DirectoryName DeliveryCacheProgress::GetCurrentDirectoryName() const NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    return CONST_GET_IMPL()->dirName;
}

FileName DeliveryCacheProgress::GetCurrentFileName() const NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    return CONST_GET_IMPL()->fileName;
}

int64_t DeliveryCacheProgress::GetCurrentDownloaded() const NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    return CONST_GET_IMPL()->currentDownloaded;
}

int64_t DeliveryCacheProgress::GetCurrentTotal() const NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    return CONST_GET_IMPL()->currentTotal;
}

int64_t DeliveryCacheProgress::GetWholeDownloaded() const NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    return CONST_GET_IMPL()->currentDownloaded + CONST_GET_IMPL()->wholeDownloaded;
}

int64_t DeliveryCacheProgress::GetWholeTotal() const NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    return CONST_GET_IMPL()->wholeTotal;
}

nn::Result DeliveryCacheProgress::GetResult() const NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    nn::Result result;
    std::memcpy(&result, &CONST_GET_IMPL()->result, sizeof (nn::Result));

    return detail::HandleError(result);
}

void DeliveryCacheProgress::Attach(detail::ipc::IDeliveryCacheProgressService* handle) NN_NOEXCEPT
{
    std::lock_guard<decltype (m_Mutex)> lock(m_Mutex);

    Detach();

    nn::sf::NativeHandle eventHandle;
    NN_ABORT_UNLESS_RESULT_SUCCESS(handle->GetEvent(&eventHandle));

    m_Event.AttachReadableHandle(eventHandle.GetOsHandle(), eventHandle.IsManaged(), nn::os::EventClearMode_ManualClear);
    eventHandle.Detach();

    NN_ABORT_UNLESS_RESULT_SUCCESS(handle->GetImpl(GET_IMPL()));

    m_Handle = handle;
}

void DeliveryCacheProgress::Detach() NN_NOEXCEPT
{
    if (m_Handle)
    {
        nn::sf::ReleaseSharedObject(m_Handle);
        m_Handle = nullptr;

        nn::os::DestroySystemEvent(m_Event.GetBase());
    }
}

}}
