﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/bcat_DeliveryCacheDirectory.h>
#include <nn/bcat/detail/bcat_ErrorHandler.h>
#include <nn/bcat/detail/bcat_ShimLibraryGlobal.h>
#include <nn/bcat/detail/ipc/bcat_ServiceConfig.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/util/util_StringUtil.h>

#define NN_DETAIL_BCAT_ACCESSLOG_DELIVERY_CACHE_DIRECTORY NN_DETAIL_FS_ACCESS_LOG_FORMAT_CLASSNAME(DeliveryCacheDirectory)

namespace nn { namespace bcat {

NN_STATIC_ASSERT(DeliveryCacheDirectory::HandleCountMax == detail::ipc::DeliveryCacheDirectoryServiceCountPerProcess);

DeliveryCacheDirectory::DeliveryCacheDirectory() NN_NOEXCEPT :
    m_Handle(nullptr)
{

}

DeliveryCacheDirectory::~DeliveryCacheDirectory() NN_NOEXCEPT
{
    if (m_Handle)
    {
        const auto handle = nn::fs::detail::IdentifyAccessLogHandle::MakeValue(m_Handle);
        NN_DETAIL_FS_ACCESS_LOG((Close(), nn::ResultSuccess()), handle, NN_DETAIL_BCAT_ACCESSLOG_DELIVERY_CACHE_DIRECTORY);
    }
}

nn::Result DeliveryCacheDirectory::Open(const char* dirName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(dirName);

    DirectoryName d = {};
    nn::util::Strlcpy(d.value, dirName, sizeof (d.value));

    NN_RESULT_DO(Open(d));

    NN_RESULT_SUCCESS;
}

nn::Result DeliveryCacheDirectory::Open(const DirectoryName& dirName) NN_NOEXCEPT
{
    auto open = [&]() NN_NOEXCEPT -> nn::Result
    {
        NN_RESULT_THROW_UNLESS(dirName.IsValid(), ResultInvalidArgument());

        NN_RESULT_THROW_UNLESS(!m_Handle, ResultAlreadyOpened());

        detail::ipc::IDeliveryCacheDirectoryService* handle = nullptr;

        NN_DETAIL_BCAT_IPC_RESULT_DO(detail::ShimLibraryGlobal::GetInstance().CreateDirectoryService(&handle));

        bool isSuccess = false;

        NN_UTIL_SCOPE_EXIT
        {
            if (!isSuccess)
            {
                nn::sf::ReleaseSharedObject(handle);
            }
        };

        NN_DETAIL_BCAT_IPC_RESULT_DO(handle->Open(dirName));

        m_Handle = handle;
        detail::ShimLibraryGlobal::GetInstance().IncrementDeliveryCacheDirectoryCount();

        isSuccess = true;

        NN_RESULT_SUCCESS;
    };

    NN_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG(open(), nn::fs::detail::IdentifyAccessLogHandle::MakeValue(m_Handle), NN_DETAIL_BCAT_ACCESSLOG_DELIVERY_CACHE_DIRECTORY NN_DETAIL_FS_ACCESS_LOG_FORMAT_PATH, dirName.value));
    NN_RESULT_SUCCESS;
}

void DeliveryCacheDirectory::Close() NN_NOEXCEPT
{
    auto close = [&]() NN_NOEXCEPT
    {
        if (m_Handle)
        {
            nn::sf::ReleaseSharedObject(m_Handle);

            m_Handle = nullptr;
            detail::ShimLibraryGlobal::GetInstance().DecrementDeliveryCacheDirectoryCount();
        }
    };
    const auto handle = nn::fs::detail::IdentifyAccessLogHandle::MakeValue(m_Handle);
    NN_DETAIL_FS_ACCESS_LOG((close(), nn::ResultSuccess()), handle, NN_DETAIL_BCAT_ACCESSLOG_DELIVERY_CACHE_DIRECTORY);
}

nn::Result DeliveryCacheDirectory::Read(int* outCount, DeliveryCacheDirectoryEntry* outEntries, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outEntries);
    NN_SDK_REQUIRES_GREATER(count, 0);

    auto read = [&]() NN_NOEXCEPT -> nn::Result
    {
        NN_RESULT_THROW_UNLESS(m_Handle, ResultNotOpened());

        int32_t actualCount = 0;

        NN_DETAIL_BCAT_IPC_RESULT_DO(m_Handle->Read(&actualCount, nn::sf::OutArray<DeliveryCacheDirectoryEntry>(outEntries, count)));

        *outCount = actualCount;

        NN_RESULT_SUCCESS;
    };
    NN_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG(read(), nn::fs::detail::IdentifyAccessLogHandle::MakeValue(m_Handle),
        NN_DETAIL_BCAT_ACCESSLOG_DELIVERY_CACHE_DIRECTORY));
    NN_RESULT_SUCCESS;
}

int DeliveryCacheDirectory::GetCount() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_Handle);

    int32_t count = 0;

    NN_ABORT_UNLESS_RESULT_SUCCESS(m_Handle->GetCount(&count));

    return count;
}

}}
