﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/bcat/bcat_Api.h>
#include <nn/bcat/detail/bcat_ErrorHandler.h>
#include <nn/bcat/detail/bcat_ShimLibraryGlobal.h>
#include <nn/nifm/nifm_ApiClientManagement.h>
#include <nn/fs/detail/fs_AccessLog.h>
#include <nn/util/util_StringUtil.h>

namespace nn { namespace bcat {

void Initialize() NN_NOEXCEPT
{
    detail::ShimLibraryGlobal::GetInstance().Initialize();
}

nn::Result MountDeliveryCacheStorage() NN_NOEXCEPT
{
    NN_DETAIL_BCAT_IPC_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG(detail::ShimLibraryGlobal::GetInstance().MountDeliveryCacheStorage(), nullptr, ""));

    NN_RESULT_SUCCESS;
}

void UnmountDeliveryCacheStorage() NN_NOEXCEPT
{
    NN_DETAIL_FS_ACCESS_LOG((detail::ShimLibraryGlobal::GetInstance().UnmountDeliveryCacheStorage(), nn::ResultSuccess()), nullptr, "");
}

nn::Result EnumerateDeliveryCacheDirectory(int* outCount, DirectoryName* outDirectoryNames, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outCount);
    NN_SDK_REQUIRES_NOT_NULL(outDirectoryNames);
    NN_SDK_REQUIRES_GREATER(count, 0);

    NN_DETAIL_BCAT_IPC_RESULT_DO(NN_DETAIL_FS_ACCESS_LOG(detail::ShimLibraryGlobal::GetInstance().EnumerateDeliveryCacheDirectory(outCount, outDirectoryNames, count), nullptr, ""));

    NN_RESULT_SUCCESS;
}

nn::Result RequestSyncDeliveryCache(DeliveryCacheProgress* outProgress) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outProgress);

    NN_ABORT_UNLESS(!detail::ShimLibraryGlobal::GetInstance().IsDeliveryCacheStorageMounted(),
        "[bcat] Delivery cache storage is mounted.");

    NN_RESULT_THROW_UNLESS(nn::nifm::IsAnyInternetRequestAccepted(nn::nifm::GetClientId()), ResultInternetRequestNotAccepted());

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    nn::sf::SharedPointer<detail::ipc::IDeliveryCacheProgressService> service = nullptr;

    NN_DETAIL_BCAT_IPC_RESULT_DO(session->RequestSyncDeliveryCache(&service));

    outProgress->Attach(service.Detach());

    NN_RESULT_SUCCESS;
}

nn::Result RequestSyncDeliveryCache(DeliveryCacheProgress* outProgress, const char* dirName) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(outProgress);
    NN_SDK_REQUIRES_NOT_NULL(dirName);

    DirectoryName d = {};

    NN_RESULT_THROW_UNLESS(nn::util::Strlcpy(d.value, dirName, sizeof (d.value)) < sizeof (d.value), ResultInvalidArgument());
    NN_RESULT_THROW_UNLESS(d.IsValid(), ResultInvalidArgument());

    NN_ABORT_UNLESS(!detail::ShimLibraryGlobal::GetInstance().IsDeliveryCacheStorageMounted(),
        "[bcat] Delivery cache storage is mounted.");

    NN_RESULT_THROW_UNLESS(nn::nifm::IsAnyInternetRequestAccepted(nn::nifm::GetClientId()), ResultInternetRequestNotAccepted());

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    nn::sf::SharedPointer<detail::ipc::IDeliveryCacheProgressService> service = nullptr;

    NN_DETAIL_BCAT_IPC_RESULT_DO(session->RequestSyncDeliveryCacheWithDirectoryName(&service, d));

    outProgress->Attach(service.Detach());

    NN_RESULT_SUCCESS;
}

void CancelSyncDeliveryCacheRequest() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(!detail::ShimLibraryGlobal::GetInstance().IsDeliveryCacheStorageMounted(),
        "[bcat] Delivery cache storage is mounted.");

    auto session = detail::ShimLibraryGlobal::GetInstance().GetSession();

    NN_ABORT_UNLESS_RESULT_SUCCESS(session->CancelSyncDeliveryCacheRequest());
}

}}
