﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SystemThreadDefinition.h>
#include "audioctrl_AsyncTaskController.h"

#define NN_AUDIOCTRL_TRACE(...)
#if !defined(NN_AUDIOCTRL_TRACE)
#define NN_AUDIOCTRL_TRACE(format, ...) NN_DETAIL_STRUCTURED_SDK_LOG(audioctrl, Trace, 0, "[audioctrl:%s] " format, NN_CURRENT_FUNCTION_NAME, ##__VA_ARGS__)
#endif

namespace nn { namespace audioctrl { namespace server { namespace detail {

AsyncTaskController::AsyncTaskController(UserInputHandler& userInputHandler, PlayReportController& playReportController) NN_NOEXCEPT
    : m_UserInputHandler(userInputHandler),
      m_PlayReportController(playReportController),
      m_HasCodecIc(false),
      m_HasHdmi(false),
      m_IsUacEnabled(false),
      m_IsAwake(false),
      m_IsInitialized(false),
      m_Mutex(true),
      m_ExitEvent(nn::os::EventClearMode_AutoClear),
      m_Thread{}
{

}

void AsyncTaskController::Initialize(bool hasCodecIc, bool hasHdmi, bool isUacEnabled) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(!m_IsInitialized);

    if(m_IsInitialized)
    {
        return;
    }

    m_IsInitialized = true;

    m_HasCodecIc = hasCodecIc;
    m_HasHdmi = hasHdmi;
    m_IsUacEnabled = isUacEnabled;
    Wake();
}

void AsyncTaskController::Finalize() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    if(!m_IsInitialized)
    {
        return;
    }


    Sleep();

    m_IsInitialized = false;
}

void AsyncTaskController::Sleep() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    if(m_IsAwake == false)
    {
        return;
    }

    m_IsAwake = false;

    // タスク処理スレッドを終了して破棄
    m_ExitEvent.Signal();
    nn::os::WaitThread(&m_Thread);
    nn::os::DestroyThread(&m_Thread);
}

void AsyncTaskController::Wake() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    NN_SDK_ASSERT(m_IsInitialized);

    if(m_IsAwake)
    {
        return;
    }

    m_IsAwake = true;

    // タスク処理スレッドを作成
    NN_ABORT_UNLESS_RESULT_SUCCESS(nn::os::CreateThread(&m_Thread, ThreadFunction, this, m_ThreadStack, sizeof(m_ThreadStack), NN_SYSTEM_THREAD_PRIORITY(audioctrl, AsyncTaskController)));
    nn::os::SetThreadNamePointer(&m_Thread, NN_SYSTEM_THREAD_NAME(audioctrl, AsyncTaskController));
    nn::os::StartThread(&m_Thread);
}

void AsyncTaskController::ThreadFunction(void* arg) NN_NOEXCEPT
{
    reinterpret_cast<AsyncTaskController*>(arg)->ThreadFunctionImpl();
}

void AsyncTaskController::ThreadFunctionImpl() NN_NOEXCEPT
{
    NN_SDK_ASSERT(m_IsInitialized);

    nn::os::MultiWaitType multiWait;

    nn::os::MultiWaitHolderType buttonUpHolder;
    nn::os::MultiWaitHolderType buttonDownHolder;
    nn::os::MultiWaitHolderType buttonRepeatTimerHolder;
    nn::os::MultiWaitHolderType headphoneJackDetectHolder;
    nn::os::MultiWaitHolderType headphoneJackDetectTimerHolder;
    nn::os::MultiWaitHolderType volumeSettingsTimerHolder;
    nn::os::MultiWaitHolderType playReportCommitTimerHolder;
    nn::os::MultiWaitHolderType uacAttachHolder;
    nn::os::MultiWaitHolderType uacDetachHolder;
    nn::os::MultiWaitHolderType unsupportedUacAttachHolder;
    nn::os::MultiWaitHolderType hidAudioControlStateChangeHolder;
    nn::os::MultiWaitHolderType exitHolder;

    // This enum's sequence must be same as the sequence of WaitAny argument.
    enum EventIndex
    {
        EventIndex_ButtonUp = 0,
        EventIndex_ButtonDown,
        EventIndex_ButtonRepeatTimer,
        EventIndex_HeadphoneMicJackDetect,
        EventIndex_HeadphoneMicJackDetectTimer,
        EventIndex_VolumeSettingsSaveTimer,
        EventIndex_PlayReportCommitTimer,
        EventIndex_UacSpeakerAttach,
        EventIndex_UacSpeakerDetach,
        EventIndex_UnsupportedUacSpeakerAttach,
        EventIndex_HidAudioControlStateChange,
        EventIndex_Exit,
    };

    // 多重待ちオブジェクトの初期化とリンク
    nn::os::InitializeMultiWait( &multiWait );

    if(m_HasCodecIc)
    {
        nn::os::InitializeMultiWaitHolder(&buttonUpHolder, &m_UserInputHandler.GetButtonUpEvent());
        nn::os::SetMultiWaitHolderUserData(&buttonUpHolder, static_cast<uintptr_t>(EventIndex_ButtonUp));
        nn::os::LinkMultiWaitHolder(&multiWait, &buttonUpHolder);

        nn::os::InitializeMultiWaitHolder(&buttonDownHolder, &m_UserInputHandler.GetButtonDownEvent());
        nn::os::SetMultiWaitHolderUserData(&buttonDownHolder, static_cast<uintptr_t>(EventIndex_ButtonDown));
        nn::os::LinkMultiWaitHolder(&multiWait, &buttonDownHolder);

        nn::os::InitializeMultiWaitHolder(&buttonRepeatTimerHolder, m_UserInputHandler.GetButtonRepeatTimerEvent().GetBase());
        nn::os::SetMultiWaitHolderUserData(&buttonRepeatTimerHolder, static_cast<uintptr_t>(EventIndex_ButtonRepeatTimer));
        nn::os::LinkMultiWaitHolder(&multiWait, &buttonRepeatTimerHolder);

        nn::os::InitializeMultiWaitHolder(&headphoneJackDetectHolder, m_UserInputHandler.GetHeadpphoneJackDetect().GetBase());
        nn::os::SetMultiWaitHolderUserData(&headphoneJackDetectHolder, static_cast<uintptr_t>(EventIndex_HeadphoneMicJackDetect));
        nn::os::LinkMultiWaitHolder(&multiWait, &headphoneJackDetectHolder);

        nn::os::InitializeMultiWaitHolder(&headphoneJackDetectTimerHolder, m_UserInputHandler.GetHeadphoneJackDetectTimerEvent().GetBase());
        nn::os::SetMultiWaitHolderUserData(&headphoneJackDetectTimerHolder, static_cast<uintptr_t>(EventIndex_HeadphoneMicJackDetectTimer));
        nn::os::LinkMultiWaitHolder(&multiWait, &headphoneJackDetectTimerHolder);
    }

    if(m_IsUacEnabled)
    {
        nn::os::InitializeMultiWaitHolder(&uacAttachHolder, m_UserInputHandler.GetUacSpeakerAttachEvent());
        nn::os::SetMultiWaitHolderUserData(&uacAttachHolder, static_cast<uintptr_t>(EventIndex_UacSpeakerAttach));
        nn::os::LinkMultiWaitHolder(&multiWait, &uacAttachHolder);

        nn::os::InitializeMultiWaitHolder(&uacDetachHolder, m_UserInputHandler.GetUacSpeakerDetachEvent());
        nn::os::SetMultiWaitHolderUserData(&uacDetachHolder, static_cast<uintptr_t>(EventIndex_UacSpeakerDetach));
        nn::os::LinkMultiWaitHolder(&multiWait, &uacDetachHolder);

        nn::os::InitializeMultiWaitHolder(&unsupportedUacAttachHolder, m_UserInputHandler.GetUacUnsupportedSpeakerAttachEvent());
        nn::os::SetMultiWaitHolderUserData(&unsupportedUacAttachHolder, static_cast<uintptr_t>(EventIndex_UnsupportedUacSpeakerAttach));
        nn::os::LinkMultiWaitHolder(&multiWait, &unsupportedUacAttachHolder);
    }

    nn::os::InitializeMultiWaitHolder(&volumeSettingsTimerHolder, m_UserInputHandler.GetVolumeSettingsTimerEvent().GetBase());
    nn::os::SetMultiWaitHolderUserData(&volumeSettingsTimerHolder, static_cast<uintptr_t>(EventIndex_VolumeSettingsSaveTimer));
    nn::os::LinkMultiWaitHolder(&multiWait, &volumeSettingsTimerHolder);

    nn::os::InitializeMultiWaitHolder( &playReportCommitTimerHolder, m_PlayReportController.GetCommitTimer().GetBase());
    nn::os::SetMultiWaitHolderUserData( &playReportCommitTimerHolder, static_cast<uintptr_t>(EventIndex_PlayReportCommitTimer));
    nn::os::LinkMultiWaitHolder( &multiWait, &playReportCommitTimerHolder);

    nn::os::InitializeMultiWaitHolder( &hidAudioControlStateChangeHolder, &m_UserInputHandler.GetHidAudioControlChangeEvent());
    nn::os::SetMultiWaitHolderUserData( &hidAudioControlStateChangeHolder, static_cast<uintptr_t>(EventIndex_HidAudioControlStateChange));
    nn::os::LinkMultiWaitHolder( &multiWait, &hidAudioControlStateChangeHolder);

    nn::os::InitializeMultiWaitHolder(&exitHolder, m_ExitEvent.GetBase());
    nn::os::SetMultiWaitHolderUserData(&exitHolder, static_cast<uintptr_t>(EventIndex_Exit));
    nn::os::LinkMultiWaitHolder(&multiWait, &exitHolder);

    // イベントをハンドリングするメインループ
    for(;;)
    {
        auto holder = nn::os::WaitAny(&multiWait);
        auto eventIndex = nn::os::GetMultiWaitHolderUserData(holder);

        NN_AUDIOCTRL_TRACE("eventIndex(%d)\n", eventIndex);

        if(eventIndex == EventIndex_ButtonUp || eventIndex == EventIndex_ButtonDown)
        {
            m_UserInputHandler.VolumeButtonHandler();
        }
        else if(eventIndex == EventIndex_ButtonRepeatTimer)
        {
            m_UserInputHandler.VolumeButtonTimerEventHandler();
        }
        else if(eventIndex == EventIndex_HeadphoneMicJackDetectTimer)
        {
            m_UserInputHandler.HeadphoneJackDetectTimerEventHandler();
        }
        else if (eventIndex == EventIndex_HeadphoneMicJackDetect)
        {
            m_UserInputHandler.HeadphoneJackDetectEventHandler();
        }
        else if(eventIndex == EventIndex_VolumeSettingsSaveTimer)
        {
            m_UserInputHandler.VolumeSettingsSaveTimerEventHandler();
        }
        else if(eventIndex == EventIndex_PlayReportCommitTimer)
        {
            m_PlayReportController.Commit();
        }
        else if(eventIndex == EventIndex_UacSpeakerAttach)
        {
            m_UserInputHandler.UacSpeakerAttachHandler();
        }
        else if(eventIndex == EventIndex_UacSpeakerDetach)
        {
            m_UserInputHandler.UacSpeakerDetachHandler();
        }
        else if(eventIndex == EventIndex_UnsupportedUacSpeakerAttach)
        {
            m_UserInputHandler.UacUnsupportedSpeakerAttachHandler();
        }
        else if(eventIndex == EventIndex_HidAudioControlStateChange)
        {
            m_UserInputHandler.HidAudioControlStateChangeEventHandler();
        }
        else if(eventIndex == EventIndex_Exit)
        {
            // 終了する前に未保存のボリューム設定情報を保存
            m_UserInputHandler.VolumeSettingsSaveTimerEventHandler();
            m_ExitEvent.Clear();
            break;
        }
    }

    if(m_HasCodecIc)
    {
        nn::os::UnlinkMultiWaitHolder(&buttonUpHolder);
        nn::os::UnlinkMultiWaitHolder(&buttonDownHolder);
        nn::os::UnlinkMultiWaitHolder(&buttonRepeatTimerHolder);
        nn::os::UnlinkMultiWaitHolder(&headphoneJackDetectHolder);
        nn::os::UnlinkMultiWaitHolder(&headphoneJackDetectTimerHolder);

        nn::os::FinalizeMultiWaitHolder(&buttonUpHolder);
        nn::os::FinalizeMultiWaitHolder(&buttonDownHolder);
        nn::os::FinalizeMultiWaitHolder(&buttonRepeatTimerHolder);
        nn::os::FinalizeMultiWaitHolder(&headphoneJackDetectHolder);
        nn::os::FinalizeMultiWaitHolder(&headphoneJackDetectTimerHolder);
    }

    if(m_IsUacEnabled)
    {
        nn::os::UnlinkMultiWaitHolder(&uacAttachHolder);
        nn::os::UnlinkMultiWaitHolder(&uacDetachHolder);
        nn::os::UnlinkMultiWaitHolder(&unsupportedUacAttachHolder);

        nn::os::FinalizeMultiWaitHolder(&uacAttachHolder);
        nn::os::FinalizeMultiWaitHolder(&uacDetachHolder);
        nn::os::FinalizeMultiWaitHolder(&unsupportedUacAttachHolder);
    }

    nn::os::UnlinkMultiWaitHolder(&volumeSettingsTimerHolder);
    nn::os::UnlinkMultiWaitHolder(&playReportCommitTimerHolder);
    nn::os::UnlinkMultiWaitHolder(&hidAudioControlStateChangeHolder);
    nn::os::UnlinkMultiWaitHolder(&exitHolder);

    nn::os::FinalizeMultiWaitHolder(&volumeSettingsTimerHolder);
    nn::os::FinalizeMultiWaitHolder(&playReportCommitTimerHolder);
    nn::os::FinalizeMultiWaitHolder(&hidAudioControlStateChangeHolder);
    nn::os::FinalizeMultiWaitHolder(&exitHolder);

    nn::os::FinalizeMultiWait(&multiWait);
} // NOLINT(impl/function_size)

}}}}  // namespace nn::audioctrl::server::detail
