﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkAssert.h>
#include <nn/audio/audio_Applet.h>
#include "audio_CreateAudioOutManager.h"

namespace nn {
namespace audio {

namespace {

nn::sf::SharedPointer<detail::IAudioOutManagerForApplet> CreateAudioOutManagerForApplet() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    return CreateAudioOutManagerForAppletByHipc();
#else
    return CreateAudioOutManagerForAppletByDfc();
#endif
}

} // anonymous namespace

nn::Result RequestSuspendAudioOuts(nn::applet::AppletResourceUserId id, nn::TimeSpan fadeTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(fadeTime, nn::TimeSpan::FromMilliSeconds(0), nn::TimeSpan::FromMilliSeconds(10));
    return CreateAudioOutManagerForApplet()->RequestSuspend(id, fadeTime.GetNanoSeconds());
}

nn::Result RequestResumeAudioOuts(nn::applet::AppletResourceUserId id, nn::TimeSpan fadeTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(fadeTime, nn::TimeSpan::FromMilliSeconds(0), nn::TimeSpan::FromMilliSeconds(10));
    return CreateAudioOutManagerForApplet()->RequestResume(id, fadeTime.GetNanoSeconds());
}

nn::Result GetAudioOutsProcessMasterVolume(float* pOutVolume, nn::applet::AppletResourceUserId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutVolume);

    NN_RESULT_DO(CreateAudioOutManagerForApplet()->GetProcessMasterVolume(pOutVolume, id));
    NN_RESULT_SUCCESS;
}

nn::Result SetAudioOutsProcessMasterVolume(nn::applet::AppletResourceUserId id, float volume, nn::TimeSpan fadeTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(volume, nn::audio::ProcessMasterVolumeMin, nn::audio::ProcessMasterVolumeMax);
    NN_SDK_REQUIRES_MINMAX(fadeTime, nn::TimeSpan::FromMilliSeconds(0), nn::TimeSpan::FromMilliSeconds(1000));

    NN_RESULT_DO(CreateAudioOutManagerForApplet()->SetProcessMasterVolume(id, volume, fadeTime.GetNanoSeconds()));
    NN_RESULT_SUCCESS;
}

nn::Result GetAudioOutsProcessRecordVolume(float* pOutVolume, nn::applet::AppletResourceUserId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutVolume);

    NN_RESULT_DO(CreateAudioOutManagerForApplet()->GetProcessRecordVolume(pOutVolume, id));
    NN_RESULT_SUCCESS;
}

nn::Result SetAudioOutsProcessRecordVolume(nn::applet::AppletResourceUserId id, float volume, nn::TimeSpan fadeTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(volume, nn::audio::ProcessRecordVolumeMin, nn::audio::ProcessRecordVolumeMax);
    NN_SDK_REQUIRES_MINMAX(fadeTime, nn::TimeSpan::FromMilliSeconds(0), nn::TimeSpan::FromMilliSeconds(1000));

    NN_RESULT_DO(CreateAudioOutManagerForApplet()->SetProcessRecordVolume(id, volume, fadeTime.GetNanoSeconds()));
    NN_RESULT_SUCCESS;
}

}  // namespace audio
}  // namespace nn
