﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkAssert.h>
#include <nn/audio/audio_Applet.h>
#include "audio_CreateAudioInManager.h"

namespace nn {
namespace audio {

namespace {

nn::sf::SharedPointer<detail::IAudioInManagerForApplet> CreateAudioInManagerForApplet() NN_NOEXCEPT
{
#if defined(NN_BUILD_CONFIG_OS_HORIZON)
    return CreateAudioInManagerForAppletByHipc();
#else
    return CreateAudioInManagerForAppletByDfc();
#endif
}

} // anonymous namespace

nn::Result RequestSuspendAudioIns(nn::applet::AppletResourceUserId id, nn::TimeSpan fadeTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(fadeTime, nn::TimeSpan::FromMilliSeconds(0), nn::TimeSpan::FromMilliSeconds(10));
    return CreateAudioInManagerForApplet()->RequestSuspend(id, fadeTime.GetNanoSeconds());
}

nn::Result RequestResumeAudioIns(nn::applet::AppletResourceUserId id, nn::TimeSpan fadeTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(fadeTime, nn::TimeSpan::FromMilliSeconds(0), nn::TimeSpan::FromMilliSeconds(10));
    return CreateAudioInManagerForApplet()->RequestResume(id, fadeTime.GetNanoSeconds());
}

nn::Result GetAudioInsProcessMasterVolume(float* pOutVolume, nn::applet::AppletResourceUserId id) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutVolume);

    NN_RESULT_DO(CreateAudioInManagerForApplet()->GetProcessMasterVolume(pOutVolume, id));
    NN_RESULT_SUCCESS;
}

nn::Result SetAudioInsProcessMasterVolume(nn::applet::AppletResourceUserId id, float volume, nn::TimeSpan fadeTime) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(nn::audio::ProcessMasterVolumeMin <= volume && volume <= nn::audio::ProcessMasterVolumeMax);
    NN_SDK_REQUIRES(nn::TimeSpan::FromMilliSeconds(0) <= fadeTime && fadeTime <= nn::TimeSpan::FromMilliSeconds(1000));

    NN_RESULT_DO(CreateAudioInManagerForApplet()->SetProcessMasterVolume(id, volume, fadeTime.GetNanoSeconds()));
    NN_RESULT_SUCCESS;
}

}  // namespace audio
}  // namespace nn
