﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_SystemThreadDefinition.h>
#include "audio_UacOutDevice.h"
#include "audio_UacVolume.h"
#include <nne/audio/audio.h>
#include <nn/audio/detail/audio_Log.h>
#include "../audio_AppletVolumeManager.h"
#include "../audio_PolyphaseResampler.h"

namespace nn {
namespace audio {
namespace detail {

namespace
{
    static const uint32_t g_UacSupportedSampleRates[] =
    {
        48000,
        44100,
        //32000,
        //24000,
        //22050,
        //12000,
        //11025,
        //8000,
    };
}

UacOutDevice::MixBuffer::MixBuffer()
    : m_Mutex(false)
    , m_ResamplerState()
    , m_pInFormat(nullptr)
    , m_pOutFormat(nullptr)
    , m_EstimateSrcOutputCount(0)
    , m_InputBytesToSamples(0)
    , m_OutputBytesToSamples(0)
    , m_NumAppendedBytes(0)
    , m_Volume(1.0f)
{
    memset(m_Buffer, 0, sizeof(m_Buffer));
}

int UacOutDevice::MixBuffer::AppendBytes(void* buffer, int size) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    NN_SDK_ASSERT_NOT_NULL(buffer);
    NN_SDK_ASSERT_MINMAX(m_NumAppendedBytes, 0, sizeof(m_Buffer));

    auto readBuffer = static_cast<int8_t*>(buffer);
    auto pSrc = reinterpret_cast<const int16_t*>(&readBuffer[0]);
    auto pDest = reinterpret_cast<int16_t*>(&m_Buffer[m_NumAppendedBytes]);

    if(size == 0)
    {
        return 0;
    }
    if(size >= sizeof(m_Buffer))
    {
        size = sizeof(m_Buffer);
    }

    if(m_pInFormat)
    {
        int srcInputBufferSize = size;

        auto srcOutputBufferSize = static_cast<int>(estimatePolyphaseResamplerOutputBufferSize(&m_ResamplerState, srcInputBufferSize, true));

        auto usableSize = std::min(srcOutputBufferSize, GetFreeBytes());

        if(usableSize != srcOutputBufferSize)
        {
            return 0;
        }

        auto actualSamples = polyphaseResample(&m_ResamplerState, pSrc, pDest, srcInputBufferSize / m_InputBytesToSamples);

        const auto gainSamples = actualSamples * m_pInFormat->channelCount;
        const auto sampleCount4 = nn::util::align_down(gainSamples, 4);
        const auto sampleCount4Padding = gainSamples - sampleCount4;
        const auto gain = static_cast<int32_t>((1 << 15) * m_Volume);

        nn::audio::server::detail::ApplyUniformGain1(pDest, pDest, gain, sampleCount4Padding);
        pDest += sampleCount4Padding;

        nn::audio::server::detail::ApplyUniformGain4(pDest, pDest, gain, sampleCount4);

        m_NumAppendedBytes += actualSamples * m_OutputBytesToSamples;
        return actualSamples * m_OutputBytesToSamples;
    }
    else
    {
        auto usableSize = std::min(size, GetFreeBytes());

        const auto sampleCount = usableSize / sizeof(int16_t);
        const auto sampleCount4 = nn::util::align_down(sampleCount, 4);
        const auto sampleCount4Padding = sampleCount - sampleCount4;
        const auto gain = static_cast<int32_t>((1 << 15) * m_Volume);

        nn::audio::server::detail::ApplyUniformGain1(pDest, pSrc, gain, sampleCount4Padding);
        pDest += sampleCount4Padding;
        pSrc += sampleCount4Padding;

        nn::audio::server::detail::ApplyUniformGain4(pDest, pSrc, gain, sampleCount4);

        m_NumAppendedBytes += usableSize;
        return usableSize;
    }
}

int UacOutDevice::MixBuffer::GetEstimatedNextFrameSize(int inputSizeBytes) NN_NOEXCEPT
{
    return estimatePolyphaseResamplerOutputBufferSize(&m_ResamplerState, inputSizeBytes, true);
}

int UacOutDevice::MixBuffer::GetBytes(void* outBuffer, int size) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    auto usableSize = std::min(size, GetAppendedBytes());

    if(usableSize == size)
    {
        auto writeBuffer = reinterpret_cast<int8_t*>(outBuffer);
        memcpy(&writeBuffer[0], &m_Buffer[0], usableSize);
    }
    return usableSize;
}

void UacOutDevice::MixBuffer::ReleaseBytes(int size) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    memmove(&m_Buffer[0], &m_Buffer[size], m_NumAppendedBytes - size);

    m_NumAppendedBytes -= size;
}

int UacOutDevice::MixBuffer::GetFreeBytes() const NN_NOEXCEPT
{
    return sizeof(m_Buffer) - m_NumAppendedBytes;
}

int UacOutDevice::MixBuffer::GetAppendedBytes() const NN_NOEXCEPT
{
    return m_NumAppendedBytes;
}

int UacOutDevice::MixBuffer::GetBufferCapacity() const NN_NOEXCEPT
{
    return sizeof(m_Buffer);
}

void UacOutDevice::MixBuffer::SetVolume(float volume) NN_NOEXCEPT
{
    // NN_DETAIL_AUDIO_INFO("[%s:%s:%4d] volume(%.2f)\n", __FILE__, NN_CURRENT_FUNCTION_NAME,  __LINE__, volume);
    m_Volume = volume;
}

void UacOutDevice::MixBuffer::Reset() NN_NOEXCEPT
{
    memset(m_Buffer, 0, sizeof(m_Buffer));
    m_NumAppendedBytes = 0;
}

bool UacOutDevice::MixBuffer::InitResampler(MixerFormat* inFormat, MixerFormat* outFormat) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if(inFormat->channelCount != outFormat->channelCount)
    {
        return false;
    }

    m_EstimateSrcOutputCount = (outFormat->sampleRate / (1000 / UacFrameDurationMs)) - 1;
    m_InputBytesToSamples = (inFormat->channelCount  * GetSampleByteSize(inFormat->sampleFormat));
    m_OutputBytesToSamples = (outFormat->channelCount  * GetSampleByteSize(outFormat->sampleFormat));
    m_pInFormat = inFormat;
    m_pOutFormat = outFormat;

    auto requiredBufferSize = requiredPolyphaseResamplerMemory(inFormat->channelCount, outFormat->sampleRate, inFormat->sampleRate, false);
    NN_ABORT_UNLESS(sizeof(m_ResamplerState) >= requiredBufferSize, "m_ResamplerState too small (Need %zd, have %zd)\n", requiredBufferSize, sizeof(m_ResamplerState));
    initializePolyphaseResampler(&m_ResamplerState, outFormat->sampleRate, inFormat->sampleRate, inFormat->channelCount, false, 1.0f);
    return true;
}

void UacOutDevice::MixBuffer::DisableResampler() NN_NOEXCEPT
{
    m_pInFormat = m_pOutFormat = nullptr;
}

UacOutDevice::UacOutDevice() NN_NOEXCEPT
    : m_Mutex(true)
    , m_pHost(nullptr)
    , m_pParser(nullptr)
    , m_pInterface(nullptr)
    , m_pInputTerminal(nullptr)
    , m_pFeatureUnit(nullptr)
    , m_pStateChangeEvent(nullptr)
    , m_Name(nullptr)
    , m_NextUsbFrame(0)
    , m_UacBufferIndex(0)
    , m_UacBufferReadIndex(0)
    , m_OutCount(0)
    , m_AltSetting(2)
    , m_IsStarted(false)
    , m_IsInitialized(false)
    , m_IsDeviceInitialized(false)
    , m_IsDeviceStarted(false)
    , m_NeedResync(false)
    , m_SkipSubmitBuffer(false)
    , m_MixVolume(1.0f)
    , m_Gain(1.0f)
{
    for(int i = 0; i < UacBufferMaxCount; ++i)
    {
        memset(m_UacBuffer[i], 0, UacBufferSize);
    }
    for (int i = 0; i < UacFrameDurationMs; ++i)
    {
        m_InputBytes[i] = UacMaxSamplesPerMs * UacMaxChannelCount * UacSampleSize;
    }

}

int UacOutDevice::SetEndpointSamplingRate(nn::cduac::Interface* interface, nn::cduac::Parser* parser) NN_NOEXCEPT
{
    nn::Result result;

    auto profile = interface->GetInterfaceProfile();

    if(!profile)
    {
        return 0;
    }

    auto altSetting = FindAltSetting(interface, parser);
    nn::usb::UsbEndpointDescriptor *pEndpoint = parser->GetEndpoint(interface->GetInterfaceProfile(), altSetting, 0);

    if (pEndpoint)
    {
        auto format = parser->GetAudioStreamingFormatType(profile, altSetting);
        if(format && format->bSamFreqType == 1)
        {
            return GetEndpointSamplingRateFromFormatType(format);
        }
        else if(format)
        {
            nn::cduac::AudioSamplingFrequency audioSamplingFrequencyRequest;
            auto sampleRate = GetEndpointSamplingRateFromFormatType(format);
            SetSamplingFrequency(&audioSamplingFrequencyRequest, sampleRate);

            auto result = interface->SetEndpointControl(
                                            nn::cduac::Request_SetCur,
                                            nn::cduac::Endpoint_SamplingFrequencyControl,
                                            pEndpoint->bEndpointAddress,
                                            sizeof(nn::cduac::AudioSamplingFrequency),
                                            &audioSamplingFrequencyRequest
                                            );
            if(!result.IsSuccess())
            {
                NN_DETAIL_AUDIO_ERROR("SetSampleRate failed (%d:%d)\n", result.GetModule(), result.GetDescription());
            }

            return sampleRate;
        }
    }
    return 0;
}

int UacOutDevice::GetEndpointSamplingRateFromFormatType(nn::cduac::AudioStreamingFormatType* format) NN_NOEXCEPT
{
    for (int i = 0; i < sizeof(g_UacSupportedSampleRates) / sizeof(g_UacSupportedSampleRates[0]); i++)
    {
        for(uint8_t j = 0; j < format->bSamFreqType; ++j)
        {
            auto uacFreq = format->tSamFreq[j];
            auto sampleRate = uacFreq.n1 + (uacFreq.n256 * 256) + (uacFreq.n65536 * 65536);
            if(g_UacSupportedSampleRates[i] == sampleRate)
            {
                return sampleRate;
            }
        }
    }
    return 0;
}

bool UacOutDevice::IsSupported(nn::cduac::Interface* interface, nn::cduac::Parser* parser) NN_NOEXCEPT
{
    uint8_t altSetting = FindAltSetting(interface, parser);
    auto formatType = parser->GetAudioStreamingFormatType(interface->GetInterfaceProfile(), altSetting);
    if(formatType)
    {
        const auto sampleRate = GetEndpointSamplingRateFromFormatType(formatType);
        if(sampleRate != 0)
        {
            MixerFormat outFormat;
            outFormat.sampleRate = sampleRate;
            outFormat.channelCount = formatType->bNrChannles;
            outFormat.sampleFormat = UacFormatToAudioFormat(formatType->bFormatType, formatType->bBitResolution);
            return Mixer::IsOutFormatSupported(outFormat);
        }
    }

    return false;
}

uint8_t UacOutDevice::FindAltSetting(nn::cduac::Interface* interface, nn::cduac::Parser* parser) NN_NOEXCEPT
{
    for (uint8_t i = 1; i < nn::usb::HsLimitMaxAltInterfaceSettingsCount; i++)
    {
        auto profile = interface->GetInterfaceProfile();
        nn::usb::UsbInterfaceDescriptor *pUsbInterfaceDescriptor = parser->GetInterface(profile, i);

        if (pUsbInterfaceDescriptor)
        {
            nn::cduac::AudioStreamingGeneral* general     = parser->GetAudioStreamingGeneral(profile, i);
            nn::cduac::AudioStreamingFormatType* format  = parser->GetAudioStreamingFormatType(profile, i);

            if (general && format)
            {
                if (general->wFormatTag == nn::cduac::AudioDataFormat::AudioDataFormat_Pcm && format->bBitResolution == (UacSampleSize * 8)) // PCM format
                {
                    if (format->bNrChannles == 2) // find highest?
                    {
                        return i;
                    }
                }
            }
        }
    }
    return -1;
}

bool UacOutDevice::Update() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if(IsInitialized())
    {
        CheckXferStatus();
        if(m_NeedResync)
        {
            ResyncUacDevice(false);
        }
        if(!m_SkipSubmitBuffer)
        {
            SubmitUacBuffer();
        }
        m_SkipSubmitBuffer = false;
        return !nn::os::TryWaitSystemEvent(m_pStateChangeEvent);
    }
    return true;
}

//Plug/Unplug
void UacOutDevice::InitializeDevice(const char* deviceName, nn::cduac::Interface* interface, nn::cduac::Host* host, nn::cduac::Parser* parser) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    m_pInterface = interface;
    m_pHost = host;
    m_pParser = parser;
    m_Name = deviceName;
    m_UacBufferIndex = 0;
    m_UacBufferReadIndex = 0;
    m_Gain = 1.0f;
    for(int i = 0; i < UacBufferMaxCount; ++i)
    {
        memset(m_UacBuffer[i], 0, UacBufferSize);
    }
    m_IsInitialized = true;
    m_pStateChangeEvent = m_pInterface->GetStateChangeEvent();
}

void UacOutDevice::FinalizeDevice() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    m_IsInitialized = false;
    m_Name = nullptr;
    m_pParser = nullptr;
    m_pHost = nullptr;
    m_pInterface = nullptr;
}

bool UacOutDevice::OpenSession(int channelCount) NN_NOEXCEPT
{
    m_InFormat.sampleRate = 48000; //gmix output rate is always 48k
    m_InFormat.channelCount = channelCount;
    m_InFormat.sampleFormat = SampleFormat_PcmInt16; //gmix sample format is always pcm16
    m_SessionId = 0;

    auto supported = Mixer::IsInFormatSupported(m_InFormat);
    return supported;
}

bool UacOutDevice::SetChannelCount(int channelCount) NN_NOEXCEPT
{
    auto oldChannelCount = m_InFormat.channelCount;
    m_InFormat.channelCount = channelCount;

    if(Mixer::IsInFormatSupported(m_InFormat))
    {
        m_Mixer.SetFormat(m_InFormat, m_OutFormat);
        return true;
    }
    else
    {
        m_InFormat.channelCount = oldChannelCount;
        return false;
    }
}

void UacOutDevice::CloseSession() NN_NOEXCEPT
{
    StopDevice();
}

//Game Start/Stop
void UacOutDevice::StartDevice() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if(!m_IsStarted)
    {
        m_IsStarted = true;
        m_MixBuffer.Reset();
        StartDeviceInternal();
        SubmitUacBuffer();
    }
}

void UacOutDevice::StopDevice() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if(m_IsStarted)
    {
        m_IsStarted = false;

        if(m_IsDeviceStarted)
        {
            StopDeviceInternal(true);
        }
    }
}

int UacOutDevice::AppendBytes(void* buffer, int size) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    return m_MixBuffer.AppendBytes(buffer, size);
}

bool UacOutDevice::IsStarted() NN_NOEXCEPT
{
    return m_IsStarted;
}

bool UacOutDevice::IsInitialized() NN_NOEXCEPT
{
    return m_IsInitialized;
}

const char* UacOutDevice::GetName() NN_NOEXCEPT
{
    NN_SDK_REQUIRES(m_IsInitialized);
    return m_Name;
}

int UacOutDevice::GetSessionId() NN_NOEXCEPT
{
    return m_SessionId;
}

bool UacOutDevice::IsDeviceInitialized() NN_NOEXCEPT
{
    return m_IsDeviceInitialized;
}

void UacOutDevice::StartDeviceInternal() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    if(!m_IsDeviceInitialized)
    {
        m_AltSetting = FindAltSetting(m_pInterface, m_pParser);
        auto profile = m_pInterface->GetInterfaceProfile();
        nn::usb::UsbEndpointDescriptor* pUsbEndpointDescriptor = m_pParser->GetEndpoint(profile, m_AltSetting, 0);

        m_pInputTerminal = m_pParser->GetInputTerminal(m_pInterface->GetInterfaceProfile());
        m_pFeatureUnit   = m_pParser->GetFeatureUnit(m_pInterface->GetInterfaceProfile());
        m_IsDeviceInitialized = true;
        SetUacControls();

        auto result = m_pInterface->SetAltSetting(
                                                m_AltSetting,                       // alternate setting to set
                                                pUsbEndpointDescriptor,                  // pointer to UsbEndpointDescriptor
                                                UacBufferMaxCount,  // maximum URB count
                                                (UacMaxSamplesPerFrame * UacSampleSize)        // maximum outstanding transfer size total
                                                );
        if (!result.IsSuccess())
        {
            NN_DETAIL_AUDIO_ERROR("SetAltSetting failed module %d description %d\n", result.GetModule(), result.GetDescription());
        }

        result = m_pInterface->RegisterBuffer(m_UacBuffer, sizeof(m_UacBuffer), m_ReportBuffer, sizeof(m_ReportBuffer));
        if (!result.IsSuccess())
        {
            NN_DETAIL_AUDIO_ERROR("RegisterBuffer failed module %d description %d\n", result.GetModule(), result.GetDescription());
            m_pInterface->SetAltSetting(0, pUsbEndpointDescriptor, 0, 0);
        }

        SetUacEndpoint();
    }

    m_IsDeviceStarted = true;
    m_UacBufferIndex = 0;
    m_UacBufferReadIndex = 0;
    //m_Mixer.ResetCurrentBuffer();
    m_NeedResync = false;
    m_SkipSubmitBuffer = false;
    SetNextUacFrameNumber();
    for(int i = 0; i < UacBufferMaxCount - 1; ++i)
    {
        SubmitUacBuffer();
    }
}

void UacOutDevice::SetUacControls() NN_NOEXCEPT
{
    if (m_pFeatureUnit && m_pInputTerminal)
    {
        uint8_t *pBmaControls = m_pFeatureUnit->bmaControls;

        auto numChannels = std::min(m_pInputTerminal->bNrChannels, static_cast<uint8_t>(UacMaxChannelCount));
        auto numControls = std::min(m_pFeatureUnit->bControlSize, static_cast<uint8_t>(sizeof(m_Controls[0])));
        auto overflowControls = m_pFeatureUnit->bControlSize - numControls;
        for (uint8_t i = 0; i < numChannels + 1; i++)
        {
            for(uint8_t j = 0; j < numControls; ++j)
            {
                m_Controls[i][j] = *pBmaControls;
                ++pBmaControls;
            }
            //Uac spec says bControlSize can be up to 255, but only defines controls for first 2 bytes.
            //Any other controls are reserved and can be skipped.
            pBmaControls += overflowControls;
        }

        if (m_Controls[0][0] & nn::cduac::Control_Volume)
        {
            //Use Master volume if possible.
            m_pInterface->GetAudioControl(nn::cduac::Request_GetCur,
                                        (nn::cduac::Control_Volume << 8),
                                        m_pFeatureUnit->bUnitId,
                                        sizeof(m_CurrentVolume[0]),
                                        &m_CurrentVolume[0]
                                        );
            m_pInterface->GetAudioControl(nn::cduac::Request_GetMax,
                                        (nn::cduac::Control_Volume << 8),
                                        m_pFeatureUnit->bUnitId,
                                        sizeof(m_MaxVolume[0]),
                                        &m_MaxVolume[0]
                                        );
            m_pInterface->GetAudioControl(nn::cduac::Request_GetMin,
                                        (nn::cduac::Control_Volume << 8),
                                        m_pFeatureUnit->bUnitId,
                                        sizeof(m_MinVolume[0]),
                                        &m_MinVolume[0]
                                        );
        }
        else
        {
            //Master volume not supported. Handle volume controls per channel
            for(int ch = 0; ch < numChannels; ++ch)
            {
                auto deviceChannel = ch + 1;
                if (m_Controls[deviceChannel][0] & nn::cduac::Control_Volume)
                {
                    m_pInterface->GetAudioControl(nn::cduac::Request_GetCur,
                                                (nn::cduac::Control_Volume << 8) | deviceChannel,
                                                m_pFeatureUnit->bUnitId,
                                                sizeof(m_CurrentVolume[ch]),
                                                &m_CurrentVolume[ch]
                                                );
                    m_pInterface->GetAudioControl(nn::cduac::Request_GetMax,
                                                (nn::cduac::Control_Volume << 8) | deviceChannel,
                                                m_pFeatureUnit->bUnitId,
                                                sizeof(m_MaxVolume[ch]),
                                                &m_MaxVolume[ch]
                                                );
                    m_pInterface->GetAudioControl(nn::cduac::Request_GetMin,
                                                (nn::cduac::Control_Volume << 8) | deviceChannel,
                                                m_pFeatureUnit->bUnitId,
                                                sizeof(m_MinVolume[ch]),
                                                &m_MinVolume[ch]
                                                );
                }
            }
        }

        if (m_Controls[0][0] & nn::cduac::Control_Mute)
        {
            m_pInterface->GetAudioControl(nn::cduac::Request_GetCur,
                                        (nn::cduac::Control_Mute << 8),
                                        m_pFeatureUnit->bUnitId,
                                        sizeof(m_IsMute[0]),
                                        &m_IsMute[0]
                                        );
        }
        else
        {
            for(int ch = 0; ch < numChannels; ++ch)
            {
                auto deviceChannel = ch + 1;
                if (m_Controls[deviceChannel][0] & nn::cduac::Control_Mute)
                {
                    m_pInterface->GetAudioControl(nn::cduac::Request_GetCur,
                                                (nn::cduac::Control_Mute << 8) | deviceChannel,
                                                m_pFeatureUnit->bUnitId,
                                                sizeof(m_IsMute[ch]),
                                                &m_IsMute[ch]
                                                );
                }
            }
        }
    }

    // Set default volume and mute state.
    SetCurrentGain(m_Gain);
    SetMute(false);
}

void UacOutDevice::SetUacEndpoint() NN_NOEXCEPT
{
    auto uacSamplingRate = SetEndpointSamplingRate(m_pInterface, m_pParser);

    auto formatType = m_pParser->GetAudioStreamingFormatType(m_pInterface->GetInterfaceProfile(), m_AltSetting);

    m_OutFormat.sampleRate = uacSamplingRate;
    m_OutFormat.channelCount = formatType->bNrChannles;
    m_OutFormat.sampleFormat = UacFormatToAudioFormat(formatType->bFormatType, formatType->bBitResolution);

    auto inputBytes = (uacSamplingRate / 1000) * m_OutFormat.channelCount * ((formatType->bBitResolution) / 8);
    m_RemainderSample = 0;
    m_RemainderIntervalMs = 0;
    if(uacSamplingRate % 1000 != 0)
    {
        m_RemainderSample = m_OutFormat.channelCount * ((formatType->bBitResolution) / 8);
        m_RemainderIntervalMs = 1000 / (uacSamplingRate % 1000);
    }
    for (int i = 0; i < UacFrameDurationMs; ++i)
    {
        m_InputBytes[i] = inputBytes;
    }

    if(m_InFormat.sampleRate == m_OutFormat.sampleRate)
    {
        m_MixBuffer.DisableResampler();
    }
    else
    {
        if(!m_MixBuffer.InitResampler(&m_InFormat, &m_OutFormat))
        {
            NN_DETAIL_AUDIO_ERROR("Init resampler failed! (inFormat: %d, %d, %d; outFormat: %d, %d, %d\n",
                    m_InFormat.sampleRate, m_InFormat.channelCount, m_InFormat.sampleFormat,
                    m_OutFormat.sampleRate, m_OutFormat.channelCount, m_OutFormat.sampleFormat);
            NN_ABORT_UNLESS(false);
        }
    }
    m_Mixer.SetFormat(m_InFormat, m_OutFormat);
}

nn::audio::SampleFormat UacOutDevice::UacFormatToAudioFormat(uint8_t uacFormat, uint8_t uacSampleSizeBits)
{
    if( uacFormat == nn::cduac::AudioDataFormat_Pcm8 || uacFormat == nn::cduac::AudioDataFormat_Pcm)
    {
        if(uacSampleSizeBits == 16)
        {
            return SampleFormat_PcmInt16;
        }
        if(uacSampleSizeBits == 32)
        {
            return SampleFormat_PcmInt32;
        }
        if(uacSampleSizeBits == 8)
        {
            return SampleFormat_PcmInt8;
        }
    }
    return SampleFormat_Invalid;
}

void UacOutDevice::SetSamplingFrequency(nn::cduac::AudioSamplingFrequency* outFrequency, uint32_t frequency) NN_NOEXCEPT
{
    outFrequency->n1         = frequency % 256;
    outFrequency->n256       = (frequency >> 8) & 0xFF; // (frequency / 256) % 256
    outFrequency->n65536     = (frequency >> 16) & 0xFF; // (frequency / 65536) % 256
}

void UacOutDevice::StopDeviceInternal(bool isConnected) NN_NOEXCEPT
{
    if(m_IsDeviceInitialized)
    {
        if(isConnected && m_IsInitialized)
        {
            uint8_t altSetting = 0;
            auto profile = m_pInterface->GetInterfaceProfile();
            nn::usb::UsbEndpointDescriptor* descriptor = m_pParser->GetEndpoint(profile, altSetting, 0);
            m_pInterface->SetAltSetting(altSetting, descriptor, 0, 0);
        }
        m_IsDeviceInitialized = false;
        m_IsDeviceStarted = false;
    }
}

int8_t* UacOutDevice::GetNextReadyBuffer() NN_NOEXCEPT
{
    auto uacBuffer = m_UacBuffer[m_UacBufferReadIndex];
    ++m_UacBufferReadIndex;
    if(m_UacBufferReadIndex >= UacBufferMaxCount)
    {
        m_UacBufferReadIndex = 0;
    }
    return uacBuffer;
}

void UacOutDevice::SubmitUacBuffer() NN_NOEXCEPT
{
    auto index = m_UacBufferIndex;

    int bytesNeeded = 0;

    if(m_RemainderSample > 0 && (m_NextUsbFrame % m_RemainderIntervalMs == 0))
    {
        m_InputBytes[0] += m_RemainderSample;
    }
    else
    {
        m_InputBytes[0] = m_InputBytes[1];
    }
    for(int i = 0; i < UacFrameDurationMs; ++i)
    {
        bytesNeeded += m_InputBytes[i];
    }

    nne::audio::gmix::ProcessOutDevice(nne::audio::gmix::Session::Name::Hda, m_SourceOffset);

    int actualBytes = m_MixBuffer.GetBytes(&m_UacBuffer[index][0], bytesNeeded);

    auto nextFrameBytesNeeded = bytesNeeded;
    if(m_RemainderSample && ( (m_NextUsbFrame + UacFrameDurationMs) % m_RemainderIntervalMs != 0) )
    {
        nextFrameBytesNeeded -= m_RemainderSample;
    }

    const int SamplesPerFrame = m_InFormat.sampleRate / (1000 / UacFrameDurationMs);
    const int BytesPerSample = m_InFormat.channelCount * GetSampleByteSize(m_InFormat.sampleFormat);
    auto nextGmixFrameSize = SamplesPerFrame * BytesPerSample;
    auto estimatedOutBytes = m_MixBuffer.GetEstimatedNextFrameSize(nextGmixFrameSize);
    if(m_MixBuffer.GetAppendedBytes() + estimatedOutBytes < nextFrameBytesNeeded)
    {
        //Standard gmix size is not enough for next uac frame
        //Increase size of next gmix frames so uac does not starve
        nextGmixFrameSize = (SamplesPerFrame * 1.1) * BytesPerSample;
    }

    m_SourceOffset = (m_SourceOffset + nextGmixFrameSize) % m_SourceSize;

    if(actualBytes != bytesNeeded)
    {
        //Mix buffer is starved.
        //Append empty buffers until it catches up.
        //TODO: Depop here
        memset(m_UacBuffer[index], 0, bytesNeeded);
    }

    nn::Result result = m_pInterface->PostTransferAsync(
                                                            m_UacBuffer[index],
                                                            m_InputBytes,
                                                            UacFrameDurationMs,
                                                            m_NextUsbFrame,
                                                            index
                                                            );
    if (!result.IsSuccess())
    {
        NN_DETAIL_AUDIO_ERROR("speaker input error %d %d\n", result.GetModule(), result.GetDescription());
        m_NeedResync = true;
    }
    else
    {
        if(actualBytes == bytesNeeded)
        {
            m_MixBuffer.ReleaseBytes(actualBytes);
        }
        m_NextUsbFrame += UacFrameDurationMs;
        ++m_UacBufferIndex;
        if(m_UacBufferIndex >= UacBufferMaxCount)
        {
            m_UacBufferIndex = 0;
        }
    }
}

void UacOutDevice::SetSourceBufferSize(int size) NN_NOEXCEPT
{
    m_SourceSize = size;
    m_SourceOffset = 0;
}

UacOutDevice::UacReadResult UacOutDevice::CheckXferStatus() NN_NOEXCEPT
{
    nn::Result result = m_pInterface->GetXferReport(&m_OutCount, m_XferReport, UacFrameDurationMs * UacBufferMaxCount);

    if (result.IsSuccess())
    {
        if (m_OutCount == UacFrameDurationMs)  // Most commone case, 1 buffer completed
        {
            uint32_t dataError = 0;
            for (uint32_t i = 0; i < m_OutCount; i++)
            {
                nn::usb::XferReport *pXferReport = &m_XferReport[i];

                if (!pXferReport->result.IsSuccess())
                {
                    // If we have a frame error we need to resync
                    if (pXferReport->result <= nn::usb::ResultNotInFrameWindow())
                    {
                        m_NeedResync = true;
                        return UacReadResult_Failure;
                    }
                    // For anything else treat it as a data error
                    else
                    {
                        dataError++;
                    }
                }
            }

            if (dataError == 0)
            {
                return UacReadResult_Success;
            }
            else // Error, depop input
            {
                return UacReadResult_Failure;
            }
        }
        else if (m_OutCount > UacFrameDurationMs)
        {
            if(m_OutCount == UacFrameDurationMs * UacBufferMaxCount)
            {
                SetNextUacFrameNumber();
                auto extraFrames = (m_OutCount / UacFrameDurationMs) - 1;
                for(int i = 0; i < extraFrames; ++i)
                {
                    SubmitUacBuffer();
                }
                return UacReadResult_Success;
            }
            else
            {
                for(int i = 0; i < m_OutCount; ++i)
                {
                    if(!m_XferReport[i].result.IsSuccess())
                    {
                        m_NeedResync = true;
                        return UacReadResult_Failure;
                    }
                }

                auto extraFrames = (m_OutCount / UacFrameDurationMs) - 1;
                for(int i = 0; i < extraFrames; ++i)
                {
                    SubmitUacBuffer();
                }
                return UacReadResult_Success;
            }
        }
        else // No buffers completed, this should never be the case :0
        {
            m_SkipSubmitBuffer = true;
            return UacReadResult_Failure;
        }
    }
    m_NeedResync = true;
    return UacReadResult_Failure;
}

void UacOutDevice::ResyncUacDevice(bool waitCompletions) NN_NOEXCEPT
{
    if(m_IsDeviceInitialized)
    {
        StopDeviceInternal(true);
        StartDeviceInternal();
    }
    m_NeedResync = false;
}

void UacOutDevice::SetNextUacFrameNumber() NN_NOEXCEPT
{
    auto result = m_pInterface->GetCurrentFrame(&m_NextUsbFrame);

    if (result.IsSuccess())
    {
        m_NextUsbFrame += (UacFrameDurationMs * 3) - (m_NextUsbFrame % UacFrameDurationMs);
    }
}

void UacOutDevice::SetMixVolume(float volume) NN_NOEXCEPT
{
    // NN_DETAIL_AUDIO_INFO("[%s:%s:%4d] volume(%.2f)\n", __FILE__, NN_CURRENT_FUNCTION_NAME,  __LINE__, volume);
    m_MixVolume = volume;
    m_MixBuffer.SetVolume(m_MixVolume);
}

void UacOutDevice::SetCurrentVolume(float gain) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_MINMAX(gain, 0.0f, 1.0f);
    if(!m_pFeatureUnit || !m_pInputTerminal)
    {
        return;
    }

    if(!m_IsDeviceInitialized)
    {
        return;
    }

    if (m_Controls[0][0] & nn::cduac::Control_Volume)
    {
        const auto VolumeRange = m_MaxVolume[0] - m_MinVolume[0];
        auto volume = m_MinVolume[0] + VolumeRange * gain;
        auto result = m_pInterface->SetAudioControl(
                                                nn::cduac::Request_SetCur,
                                                (nn::cduac::Control_Volume << 8),
                                                m_pFeatureUnit->bUnitId,
                                                sizeof(volume),
                                                &volume
                                                );

        if (!result.IsSuccess())
        {
            NN_DETAIL_AUDIO_ERROR("Could not set volume (%d:%d)\n", result.GetModule(), result.GetDescription());
        }
        m_CurrentVolume[0] = volume;
    }
    else
    {
        auto numChannels = std::min(m_pInputTerminal->bNrChannels, static_cast<uint8_t>(UacMaxChannelCount));
        for (uint8_t i = 0; i < numChannels; i++)
        {
            auto deviceChannel = i + 1;
            if (m_Controls[deviceChannel][0] & nn::cduac::Control_Volume)
            {
                const auto VolumeRange = m_MaxVolume[i] - m_MinVolume[i];
                auto volume = static_cast<int16_t>(m_MinVolume[i] + VolumeRange * gain);

                uint16_t command = (nn::cduac::Control_Volume << 8) | deviceChannel;
                auto result = m_pInterface->SetAudioControl(
                                                        nn::cduac::Request_SetCur,
                                                        command,
                                                        m_pFeatureUnit->bUnitId,
                                                        sizeof(volume),
                                                        &volume
                                                        );

                if (!result.IsSuccess())
                {
                    NN_DETAIL_AUDIO_ERROR("Could not set volume[%d] (%d:%d)\n", i, result.GetModule(), result.GetDescription());
                }
                else
                {
                    m_CurrentVolume[i] = volume;
                }
            }
        }
    }
}

void UacOutDevice::SetMute(bool mute) NN_NOEXCEPT
{
    if(!m_pFeatureUnit || !m_pInputTerminal)
    {
        return;
    }

    uint8_t data = mute;
    auto numChannels = std::min(m_pInputTerminal->bNrChannels, static_cast<uint8_t>(UacMaxChannelCount));

    if(m_Controls[0][0] & nn::cduac::Control_Mute)
    {
        if(m_IsMute[0] == mute)
        {
            return;
        }
        auto result = m_pInterface->SetAudioControl(
                                        nn::cduac::Request_GetCur,
                                        (nn::cduac::Control_Mute << 8),
                                        m_pFeatureUnit->bUnitId,
                                        sizeof(data),
                                        &data
                                        );
        if(!result.IsSuccess())
        {
            NN_DETAIL_AUDIO_ERROR("Could not set mute (%d:%d)\n", result.GetModule(), result.GetDescription());
        }
        else
        {
            for(int i = 0 ; i < numChannels; ++i)
            {
                m_IsMute[i] = mute;
            }
        }
    }
    else
    {
        for (uint8_t i = 0; i < numChannels; i++)
        {
            auto deviceChannel = i + 1;
            if(m_Controls[deviceChannel][0] & nn::cduac::Control_Mute)
            {
                if(m_IsMute[i] == mute)
                {
                    continue;
                }
                auto result = m_pInterface->SetAudioControl(
                                                nn::cduac::Request_GetCur,
                                                (nn::cduac::Control_Mute << 8) | deviceChannel,
                                                m_pFeatureUnit->bUnitId,
                                                sizeof(data),
                                                &data
                                                );
                if(!result.IsSuccess())
                {
                    NN_DETAIL_AUDIO_ERROR("Could not set mute (%d:%d)\n", result.GetModule(), result.GetDescription());
                }
                else
                {
                    m_IsMute[i] = mute;
                }
            }
        }
    }
}

void UacOutDevice::SetCurrentGain(float gain) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    NN_SDK_REQUIRES_MINMAX(gain, 0.0f, 1.0f);
    m_Gain = gain;
    SetCurrentVolume(gain);
}


nn::os::SystemEventType* UacOutDevice::GetBufferCompletionEvent() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if(m_pInterface)
    {
        return m_pInterface->GetCompletionEvent();
    }
    return nullptr;
}

void UacOutDevice::ClearBufferCompletionEvent() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    if(m_pInterface && m_IsDeviceInitialized)
    {
        auto event = m_pInterface->GetCompletionEvent();
        if(event)
        {
            nn::os::ClearSystemEvent(event);
        }
    }
}

}  // namespace detail
}  // namespace audio
}  // namespace nn


