﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <nn/result/result_HandlingUtility.h>
#include <nn/audio/detail/audio_Log.h>
#include <nn/nn_SystemThreadDefinition.h>
#include "audio_AudioWasapiDriver.h"
#include "audio_InterruptManager.win32.h"
#include "../audio_AudioOutManagerImpl.h"
#include "../audio_AudioInManagerImpl.h"
#include "../../common/audio_AudioInPrivate.h"

#ifndef AUDCLNT_STREAMFLAGS_AUTOCONVERTPCM
#define AUDCLNT_STREAMFLAGS_AUTOCONVERTPCM 0x80000000 // NOLINT(preprocessor/const)
#endif
#ifndef AUDCLNT_STREAMFLAGS_SRC_DEFAULT_QUALITY
#define AUDCLNT_STREAMFLAGS_SRC_DEFAULT_QUALITY 0x08000000 // NOLINT(preprocessor/const)
#endif

namespace nn {
namespace audio {
namespace detail {

namespace
{
    InterruptManager<AudioWasapiDriver, server::AudioOutManagerImpl::NumberOfSessions + server::AudioInManagerImpl::NumberOfAudioInSessions> g_InterruptManager;
}
NN_OS_ALIGNAS_THREAD_STACK char AudioWasapiDriver::s_Stack[StackSize];
IMMDeviceEnumerator* AudioWasapiDriver::s_pEnumerator = nullptr;
os::SemaphoreType* AudioWasapiDriver::s_pInputSemaphore = nullptr;
os::SemaphoreType* AudioWasapiDriver::s_pOutputSemaphore = nullptr;
os::ThreadType AudioWasapiDriver::s_Thread;
os::MutexType AudioWasapiDriver::s_MutexType = NN_OS_MUTEX_INITIALIZER(false);
std::atomic<int> AudioWasapiDriver::s_InitializeReferenceCount = 0;
std::atomic<bool> AudioWasapiDriver::s_IsThreadActive (false);

AudioWasapiDriver::AudioWasapiDriver() NN_NOEXCEPT
    : m_pDevice(nullptr)
    , m_pClient(nullptr)
    , m_pSemaphore(nullptr)
    , m_hEvent(nullptr)
    , m_pGenericClient(nullptr)
    , m_Mutex(true)
    , m_Format()
    , m_Volume(1.0f)
    , m_NumAppendedBuffers(0)
    , m_BufferReadPosition(0)
    , m_BufferSize(0)
    , m_Type()
    , m_IsOpen(false)
{
}

AudioWasapiDriver::~AudioWasapiDriver() NN_NOEXCEPT
{
}

bool AudioWasapiDriver::Initialize(server::SessionType type, nn::os::SemaphoreType* pSemaphore) NN_NOEXCEPT
{
    auto refCount = s_InitializeReferenceCount++;

    NN_SDK_ASSERT_NOT_NULL(pSemaphore);

    switch(type)
    {
        case server::SessionType_Input:
            NN_SDK_ASSERT(s_pInputSemaphore == nullptr);
            s_pInputSemaphore = pSemaphore;
            break;
        case server::SessionType_Output:
            NN_SDK_ASSERT(s_pOutputSemaphore == nullptr);
            s_pOutputSemaphore = pSemaphore;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    if(refCount != 0)
    {
        return true;
    }

    HRESULT hr = CoInitializeEx(nullptr, COINIT_MULTITHREADED);
    NN_ABORT_UNLESS(SUCCEEDED(hr) || hr == RPC_E_CHANGED_MODE);

    NN_ABORT_UNLESS(
        SUCCEEDED(
            CoCreateInstance(
                __uuidof(MMDeviceEnumerator),
                nullptr,
                CLSCTX_INPROC_SERVER,
                IID_PPV_ARGS(&s_pEnumerator))));

    g_InterruptManager.Initialize();

    s_IsThreadActive.store(true);
    auto result = nn::os::CreateThread(&s_Thread, ThreadFunc, nullptr, s_Stack, StackSize, NN_SYSTEM_THREAD_PRIORITY(audio, AudioOutSystem));
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    nn::os::SetThreadNamePointer(&s_Thread, NN_SYSTEM_THREAD_NAME(audio, AudioOutSystem));
    nn::os::StartThread(&s_Thread);

    return true;
}

void AudioWasapiDriver::Finalize(server::SessionType type) NN_NOEXCEPT
{
    auto refCount = s_InitializeReferenceCount--;

    switch(type)
    {
        case server::SessionType_Input:
            NN_SDK_ASSERT_NOT_NULL(s_pInputSemaphore);
            s_pInputSemaphore = nullptr;
            break;
        case server::SessionType_Output:
            NN_SDK_ASSERT_NOT_NULL(s_pOutputSemaphore);
            s_pOutputSemaphore = nullptr;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    NN_SDK_ASSERT(refCount >= 0);
    if(refCount == 0)
    {
        //Stop Update thread
        g_InterruptManager.Finalize();
        s_IsThreadActive.store(false);
        nn::os::WaitThread(&s_Thread);
        nn::os::DestroyThread(&s_Thread);
        if( s_pEnumerator )
        {
            s_pEnumerator->Release();
        }
        CoUninitialize();
    }
}

void AudioWasapiDriver::ThreadFunc(void* arg)
{
    NN_UNUSED(arg);

    while (s_IsThreadActive)
    {
        AudioWasapiDriver* pDriver = g_InterruptManager.GetSignaled();
        if (pDriver)
        {
            pDriver->UpdateSession();
        }
    }
}

EDataFlow AudioWasapiDriver::SessionTypeToWasapiType(server::SessionType sessionType) NN_NOEXCEPT
{
    switch(sessionType)
    {
        case server::SessionType_Output:
            return eRender;
        case server::SessionType_Input:
            return eCapture;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

GUID AudioWasapiDriver::SampleFormatToWindowsGuid(SampleFormat sampleFormat) NN_NOEXCEPT
{
    switch(sampleFormat)
    {
        case SampleFormat_PcmInt16:
            return KSDATAFORMAT_SUBTYPE_PCM;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

void AudioWasapiDriver::CreateWaveFormat(WAVEFORMATEXTENSIBLE* outFormat, const server::SessionFormat& format) NN_NOEXCEPT
{
    auto windowsFormat = SampleFormatToWindowsGuid(format.format);
    WORD formatSizeBits = static_cast<WORD>(nn::audio::GetSampleByteSize(format.format) * 8);
    outFormat->Format.wFormatTag = WAVE_FORMAT_EXTENSIBLE;
    outFormat->Format.nChannels = format.channelCount;
    outFormat->Format.nSamplesPerSec = format.sampleRate;
    outFormat->Format.nAvgBytesPerSec = format.channelCount * format.sampleRate * 2;
    outFormat->Format.wBitsPerSample = formatSizeBits;
    outFormat->Format.nBlockAlign = format.channelCount * 2;
    outFormat->Format.cbSize = 22;

    outFormat->Samples.wValidBitsPerSample = formatSizeBits;
    outFormat->SubFormat = windowsFormat;

    switch (format.channelCount)
    {
    case 1:
        outFormat->dwChannelMask = KSAUDIO_SPEAKER_MONO;
        break;
    case 2:
        outFormat->dwChannelMask = KSAUDIO_SPEAKER_STEREO;
        break;
    case 6:
        outFormat->dwChannelMask = KSAUDIO_SPEAKER_5POINT1;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

Result AudioWasapiDriver::IsSupported(server::SessionType sessionType, const char* name, const server::SessionFormat& format) NN_NOEXCEPT
{
    NN_SDK_REQUIRES(format.channelCount == 0 || format.channelCount == 2 || format.channelCount == 6);
    NN_SDK_ASSERT_NOT_NULL(name);

    EDataFlow type = SessionTypeToWasapiType(sessionType);
    IMMDeviceCollection* pDeviceCollection;
    IMMDevice* pDevice;
    bool nameFound = false;

    NN_RESULT_THROW_UNLESS(format.sampleRate == 0 || format.sampleRate == 48000, ResultInvalidSampleRate());
    NN_RESULT_THROW_UNLESS(format.channelCount == 0 || format.channelCount == 2 || format.channelCount == 6, ResultInvalidChannelCount());

    if (strncmp(name, GetDefaultName(sessionType), NameLength) == 0 || (sessionType == server::SessionType_Input && strncmp(name, nn::audio::common::DeviceAudioIn, NameLength) == 0))
    {
        NN_RESULT_THROW_UNLESS(SUCCEEDED(s_pEnumerator->GetDefaultAudioEndpoint(type, eConsole, &pDevice)), ResultOperationFailed());
        nameFound = true;
        pDevice->Release();
    }
    else
    {
        char deviceName[NameLength];
        UINT count = 0;

        NN_RESULT_THROW_UNLESS(SUCCEEDED(s_pEnumerator->EnumAudioEndpoints(type, DEVICE_STATE_ACTIVE, &pDeviceCollection)), ResultNotFound());

        NN_UTIL_SCOPE_EXIT
        {
            pDeviceCollection->Release();
        };

        NN_RESULT_THROW_UNLESS(SUCCEEDED(pDeviceCollection->GetCount(&count)), ResultOperationFailed());
        for(UINT i = 0; i < count; ++i)
        {
            NN_RESULT_THROW_UNLESS(SUCCEEDED(pDeviceCollection->Item(i, &pDevice)), ResultOperationFailed());

            GetDeviceName(pDevice, deviceName, NameLength);
            pDevice->Release();
            if (strncmp(name, deviceName, NameLength) == 0)
            {
                nameFound = true;
                break;
            }
        }
    }

    NN_RESULT_THROW_UNLESS(nameFound, ResultNotFound());
    NN_RESULT_SUCCESS;
}

Result AudioWasapiDriver::Open(server::SessionType sessionType, AppletVolumeManager::SessionType avmType, const char* name, server::SessionFormat& format, int sessionId, nn::dd::ProcessHandle processHandle, nn::applet::AppletResourceUserId aruid) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    NN_UNUSED(avmType);
    NN_UNUSED(sessionId);
    NN_UNUSED(processHandle);
    NN_UNUSED(aruid);

    NN_ABORT_UNLESS(!m_IsOpen);

    m_Type = SessionTypeToWasapiType(sessionType);

    NN_UTIL_SCOPE_EXIT
    {
        if(!m_IsOpen)
        {
            Close();
        }
    };

    if(format.sampleRate == 0)
    {
        format.sampleRate = 48000;
    }
    if(format.channelCount == 0)
    {
        format.channelCount = 2;
    }

    if (strncmp(name, GetDefaultName(sessionType), NameLength) == 0 || (m_Type == eCapture && strncmp(name, nn::audio::common::DeviceAudioIn, NameLength) == 0))
    {
        if (FAILED(s_pEnumerator->GetDefaultAudioEndpoint(m_Type, eConsole, &m_pDevice)))
        {
            m_pDevice = nullptr;
            NN_RESULT_THROW(ResultOperationFailed());
        }
    }
    else
    {
        IMMDeviceCollection* pDeviceCollection = nullptr;
        NN_RESULT_THROW_UNLESS(SUCCEEDED(s_pEnumerator->EnumAudioEndpoints(m_Type, DEVICE_STATE_ACTIVE, &pDeviceCollection)), ResultOperationFailed());

        NN_UTIL_SCOPE_EXIT
        {
            pDeviceCollection->Release();
        };

        UINT count = 0;
        NN_RESULT_THROW_UNLESS(SUCCEEDED(pDeviceCollection->GetCount(&count)), ResultOperationFailed());

        IMMDevice* pDevice;
        m_pDevice = nullptr;
        for (UINT i = 0; i < count; ++i)
        {
            pDeviceCollection->Item(i, &pDevice);
            char deviceName[NameLength];
            GetDeviceName(pDevice, deviceName, NameLength);
            if (strncmp(name, deviceName, NameLength) == 0)
            {
                m_pDevice = pDevice;
                break;
            }
            else
            {
                pDevice->Release();
            }
        }
        NN_RESULT_THROW_UNLESS(m_pDevice, ResultNotFound());
    }

    NN_SDK_ASSERT(m_pDevice);
    NN_RESULT_THROW_UNLESS(SUCCEEDED(m_pDevice->Activate(__uuidof(IAudioClient), CLSCTX_INPROC_SERVER, nullptr, reinterpret_cast<void**>(&m_pClient))), ResultOperationFailed());


    REFERENCE_TIME hnsDefaultDevicePeriod;
    REFERENCE_TIME hnsMinimumDevicePeriod;

    NN_RESULT_THROW_UNLESS(SUCCEEDED(m_pClient->GetDevicePeriod(&hnsDefaultDevicePeriod, &hnsMinimumDevicePeriod)), ResultOperationFailed());

    WAVEFORMATEXTENSIBLE waveFormat;
    CreateWaveFormat(&waveFormat, format);


    auto streamFlags = AUDCLNT_STREAMFLAGS_EVENTCALLBACK //Event-driven (Enable SetEventHandle)
        | AUDCLNT_STREAMFLAGS_NOPERSIST // Volume/Mute change do not persist across system restarts
        | AUDCLNT_STREAMFLAGS_AUTOCONVERTPCM // Apply SRC/channel matrixer to convert input rate/channel count to output rate/channel count
        | AUDCLNT_STREAMFLAGS_SRC_DEFAULT_QUALITY; // Use high-quality SRC


    NN_RESULT_THROW_UNLESS(SUCCEEDED(m_pClient->Initialize(AUDCLNT_SHAREMODE_SHARED, streamFlags,
        hnsDefaultDevicePeriod, 0, reinterpret_cast<const WAVEFORMATEX*>(&waveFormat), &GUID_NULL)), ResultOperationFailed());
    m_hEvent = CreateEvent(NULL, FALSE, FALSE, NULL);

    NN_RESULT_THROW_UNLESS(m_hEvent, ResultOperationFailed());

    NN_RESULT_THROW_UNLESS(SUCCEEDED(m_pClient->SetEventHandle(m_hEvent)), ResultOperationFailed());

    m_pGenericClient = GetClientImpl();
    m_BufferSize = 0;
    m_Format = format;
    m_BufferReadPosition = 0;

    NN_RESULT_THROW_UNLESS(SUCCEEDED((m_pClient->GetBufferSize(&m_BufferSize))), ResultOperationFailed());

    //If this fails, make g_InterruptManager capacity bigger.
    auto result = g_InterruptManager.Register(this);
    NN_ABORT_UNLESS_RESULT_SUCCESS(result);

    switch(m_Type)
    {
        case eRender:
            m_pSemaphore = s_pOutputSemaphore;
            break;
        case eCapture:
            m_pSemaphore = s_pInputSemaphore;
            break;
        default:
            NN_UNEXPECTED_DEFAULT;
    }

    m_Volume = 1.0f;
    m_IsOpen = true;
    NN_RESULT_SUCCESS;
} // NOLINT(impl/function_size)

Result AudioWasapiDriver::Close() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    NN_RESULT_THROW_UNLESS(m_IsOpen, ResultSuccess());

    g_InterruptManager.Unregister(this);
    if(m_pGenericClient)
    {
        if(m_Type == eRender)
        {
            m_pRenderClient->Release();
        }
        else if(m_Type == eCapture)
        {
            m_pCaptureClient->Release();
        }
    }

    if (m_pClient)
    {
        m_pClient->Release();
        m_pClient = nullptr;
    }

    if (m_pDevice)
    {
        m_pDevice->Release();
        m_pDevice = nullptr;
    }

    if (m_hEvent)
    {
        CloseHandle(m_hEvent);
        m_hEvent = nullptr;
    }

    m_IsOpen = false;
    NN_RESULT_SUCCESS;
}

Result AudioWasapiDriver::Start() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    NN_RESULT_THROW_UNLESS(m_pClient, ResultOperationFailed());

    m_pClient->Start();
    NN_RESULT_SUCCESS;
}

Result AudioWasapiDriver::Stop() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    if(m_pClient)
    {
        m_pClient->Stop();
    }

    NN_RESULT_SUCCESS;
}

void* AudioWasapiDriver::GetClientImpl() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    void* result = nullptr;

    if (m_Type == eRender)
    {
        if (FAILED(m_pClient->GetService(__uuidof(IAudioRenderClient), &result)))
        {
            return nullptr;
        }
    }
    else if (m_Type == eCapture)
    {
        if (FAILED(m_pClient->GetService(__uuidof(IAudioCaptureClient), &result)))
        {
            return nullptr;
        }
    }

    return result;
}

UINT32 AudioWasapiDriver::GetCurrentPadding() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    UINT32 padding;
    if (FAILED(m_pClient->GetCurrentPadding(&padding)))
    {
        return 0;
    }
    return padding;
}

UINT32 AudioWasapiDriver::GetBufferSize() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    return m_BufferSize;
}

void* AudioWasapiDriver::AcquireRenderBuffer(int numFrames) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    NN_SDK_ASSERT(m_Type == eRender);
    BYTE* result;
    if (FAILED(m_pRenderClient->GetBuffer(numFrames, &result)))
    {
        return nullptr;
    }

    return reinterpret_cast<void*>(result);
}

Result AudioWasapiDriver::ReleaseRenderBuffer(int numFrames) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    NN_SDK_ASSERT(m_Type == eRender);
    NN_RESULT_THROW_UNLESS(SUCCEEDED(m_pRenderClient->ReleaseBuffer(numFrames, 0)), ResultOperationFailed());
    NN_RESULT_SUCCESS;
}


void* AudioWasapiDriver::AcquireCaptureBuffer(int* numFrames) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    NN_SDK_ASSERT(m_Type == eCapture);
    BYTE* data;
    UINT frameCount;
    DWORD flag;
    m_pCaptureClient->GetBuffer(&data, &frameCount, &flag, NULL, NULL);

    *numFrames = static_cast<int>(frameCount);
    return data;
}

Result AudioWasapiDriver::ReleaseCaptureBuffer(int numFrames) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    NN_SDK_ASSERT(m_Type == eCapture);
    NN_RESULT_THROW_UNLESS(SUCCEEDED(m_pCaptureClient->ReleaseBuffer(numFrames)), ResultOperationFailed());
    NN_RESULT_SUCCESS;
}

Result AudioWasapiDriver::Sleep() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

Result AudioWasapiDriver::Wake() NN_NOEXCEPT
{
    NN_RESULT_SUCCESS;
}

uint64_t AudioWasapiDriver::GetSamplesProcessed() const NN_NOEXCEPT
{
    return 0;
}

void* AudioWasapiDriver::MapBuffer(server::SessionBuffer& buffer, nn::dd::ProcessHandle processHandle) NN_NOEXCEPT
{
    NN_UNUSED(processHandle);
    buffer.mappedAddr = buffer.addr;
    return buffer.addr;
}

void* AudioWasapiDriver::MapBuffer(server::SessionBuffer& buffer, nn::sf::NativeHandle&& bufferHandle) NN_NOEXCEPT
{
    NN_UNUSED(bufferHandle);
    NN_UNUSED(buffer);
    buffer.mappedAddr = buffer.addr;
    return buffer.addr;
}

bool AudioWasapiDriver::UnmapBuffer(const server::SessionBuffer& buffer) NN_NOEXCEPT
{
    NN_UNUSED(buffer);
    return true;
}

bool AudioWasapiDriver::TryCompleteBuffer(const server::SessionBuffer& buffer) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    if(m_NumAppendedBuffers == 0 || buffer.addr != m_AppendBufferList[0].addr)
    {
        return true;
    }
    return false;
}

void AudioWasapiDriver::UpdateSession() NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);

    //Session may have been closed between InterruptManager signal and call to UpdateSession
    if(!m_IsOpen)
    {
        return;
    }

    NN_SDK_ASSERT(m_Type == eRender || m_Type == eCapture);
    auto wasapiFrameSize = m_Format.channelCount * GetSampleByteSize(m_Format.format);
    if(m_Type == eRender)
    {
        auto wasapiReadyBytes = (m_BufferSize - GetCurrentPadding()) * wasapiFrameSize;
        auto bytesToWrite = wasapiReadyBytes;
        auto wasapiBuffer = reinterpret_cast<int8_t*>(AcquireRenderBuffer(static_cast<int>(wasapiReadyBytes / wasapiFrameSize)));
        if(!wasapiBuffer)
        {
            return;
        }
        while(wasapiReadyBytes && m_NumAppendedBuffers)
        {
            auto userReadyBytes = m_AppendBufferList[0].size - m_BufferReadPosition;
            auto readyBytes = std::min(wasapiReadyBytes,  userReadyBytes);
            auto userBuffer = reinterpret_cast<int8_t*>(m_AppendBufferList[0].addr) + m_BufferReadPosition;

            CopyAndApplyVolume(wasapiBuffer, userBuffer, readyBytes);

            m_BufferReadPosition += readyBytes;
            wasapiBuffer += readyBytes;
            wasapiReadyBytes -= readyBytes;
            if(m_BufferReadPosition == m_AppendBufferList[0].size)
            {
                --m_NumAppendedBuffers;
                memmove(m_AppendBufferList, m_AppendBufferList + 1, m_NumAppendedBuffers * sizeof(server::SessionBuffer));
                m_BufferReadPosition = 0;
            }
        }
        ReleaseRenderBuffer(static_cast<int>((bytesToWrite - wasapiReadyBytes) / wasapiFrameSize));
    }
    else
    {
        NN_SDK_ASSERT(m_Type == eCapture);
        int inputFrames;
        auto wasapiBuffer = reinterpret_cast<int8_t*>(AcquireCaptureBuffer(&inputFrames));
        size_t wasapiReadyBytes = inputFrames * wasapiFrameSize;
        if(!wasapiBuffer)
        {
            return;
        }
        while(wasapiReadyBytes && m_NumAppendedBuffers)
        {
            auto userReadyBytes = m_AppendBufferList[0].size - m_BufferReadPosition;
            auto readyBytes = std::min(wasapiReadyBytes,  userReadyBytes);
            auto userBuffer = reinterpret_cast<int8_t*>(m_AppendBufferList[0].addr) + m_BufferReadPosition;
            CopyAndApplyVolume(userBuffer, wasapiBuffer, readyBytes);

            m_BufferReadPosition += readyBytes;
            wasapiBuffer += readyBytes;
            wasapiReadyBytes -= readyBytes;
            if(m_BufferReadPosition == m_AppendBufferList[0].size)
            {
                --m_NumAppendedBuffers;
                memmove(m_AppendBufferList, m_AppendBufferList + 1, m_NumAppendedBuffers * sizeof(m_AppendBufferList[0]));
                m_BufferReadPosition = 0;
            }
        }

        ReleaseCaptureBuffer(inputFrames);
    }

    if (nn::os::GetCurrentSemaphoreCount(m_pSemaphore) == 0)
    {
        os::ReleaseSemaphore(m_pSemaphore);
    }
}

void AudioWasapiDriver::CopyAndApplyVolume(int8_t* pDst, int8_t* pSrc, size_t sizeBytes) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS(sizeBytes % GetSampleByteSize(m_Format.format) == 0);
    auto sizeSamples = sizeBytes / GetSampleByteSize(m_Format.format);

    auto pSampleDst = reinterpret_cast<int16_t*>(pDst);
    auto pSampleSrc = reinterpret_cast<int16_t*>(pSrc);

    for(size_t i = 0; i < sizeSamples; ++i)
    {
        pSampleDst[i] = static_cast<int16_t>(pSampleSrc[i] * m_Volume);
    }
}

Result AudioWasapiDriver::AppendBuffer(const server::SessionBuffer& buffer) NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    NN_RESULT_THROW_UNLESS(m_NumAppendedBuffers < BufferCountMax, ResultOperationFailed());

    m_AppendBufferList[m_NumAppendedBuffers] = buffer;
    ++m_NumAppendedBuffers;
    NN_RESULT_SUCCESS;
}

void* AudioWasapiDriver::GetBufferAddress() const NN_NOEXCEPT
{
    std::lock_guard<os::Mutex> lock(m_Mutex);
    if(m_NumAppendedBuffers == 0)
    {
        return nullptr;
    }
    return m_AppendBufferList[0].addr;
}

Result AudioWasapiDriver::SetDeviceGain(float gain) NN_NOEXCEPT
{
    m_Volume = gain;
    NN_RESULT_SUCCESS;
}

float AudioWasapiDriver::GetDeviceGain() const NN_NOEXCEPT
{
    return m_Volume;
}

const char* AudioWasapiDriver::GetDefaultName(server::SessionType sessionType) NN_NOEXCEPT
{
    NN_UNUSED(sessionType);
    return "";
}

void* AudioWasapiDriver::GetWindowsEventHandle() const NN_NOEXCEPT
{
    return m_hEvent;
}

}  // namespace detail
}  // namespace audio
}  // namespace nn
