﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_Mutex.h>
#include <nn/audio/audio_Result.h>
#include <nn/applet/applet_Types.h>
#include <nn/audio/audio_AudioOutTypes.h>
#include <nn/Result/result_HandlingUtility.h>
#include <nn/audio/detail/audio_Log.h>


#include "audio_AudioDriver.h"
#include "../audio_AudioSessionTypes.h"
#include "../audio_AppletVolumeManager.h"

namespace nn {
namespace audio {
namespace detail {

template <int ServerBufferCount, int DriverBufferCount>
class AudioSession
{
public:
    //Pure virtual functions - needs to be implemented for each session type.
    virtual const char* GetDefaultName() const NN_NOEXCEPT = 0;
    virtual Result IsSupported(const char* name, const server::SessionFormat& format) const NN_NOEXCEPT = 0;

public:
AudioSession(int32_t sessionId, AudioDriver* pDriver, nn::os::SystemEvent& releaseClientBufferEvent, server::SessionType sessionType) NN_NOEXCEPT
    : m_ClientProcessHandle(os::InvalidNativeHandle)
    , m_ReleaseClientBufferEvent(releaseClientBufferEvent)
    , m_pDriver(pDriver)
    , m_BufferList(DriverBufferCount)
    , m_Type(sessionType)
    , m_State(server::SessionState_Stopped)
    , m_SessionId(sessionId)
    , m_IsOpen(false)
{
}

~AudioSession() NN_NOEXCEPT
{
    Close();
}

Result Open(const char* name, const server::SessionFormat& format, nn::dd::ProcessHandle clientProcessHandle, const nn::applet::AppletResourceUserId& appletId) NN_NOEXCEPT
{
    NN_RESULT_DO(IsSupported(name, format));

    m_ClientProcessHandle = clientProcessHandle;
    m_AppletResourceUserId = appletId;

    memset(m_Name, 0, sizeof(m_Name));

    if(name == nullptr || strcmp(name, "") == 0)
    {
        strncpy(reinterpret_cast<char*>(m_Name),  GetDefaultName(), sizeof(m_Name));
    }
    else
    {
        strncpy(reinterpret_cast<char*>(m_Name), name, sizeof(m_Name));
    }

    m_Name[ sizeof(m_Name) - 1 ] = '\0';

    m_IsOpen = true;

    m_Format.format = format.format;

    m_Format.sampleRate = GetDefaultSampleRate();
    if(format.channelCount <= 2)
    {
        m_Format.channelCount = 2;
    }
    else
    {
        m_Format.channelCount = 6;
    }
    NN_RESULT_SUCCESS;
}

Result Start() NN_NOEXCEPT
{
    if(m_State == server::SessionState_Stopped)
    {
        auto avmType = SessionTypeToAppletVolumeManagerType(m_Type);
        m_pDriver->Open(m_Type, avmType, m_Name, m_Format, m_SessionId, m_ClientProcessHandle, m_AppletResourceUserId);
        m_pDriver->Start();

        m_State = server::SessionState_Started;
        AppendDriverBuffers();
        NN_RESULT_SUCCESS;
    }
    else
    {
        NN_RESULT_THROW(ResultOperationFailed());
    }
}

void Stop() NN_NOEXCEPT
{
    if (m_State == server::SessionState_Started)
    {
        m_pDriver->Stop();

        m_pDriver->Close();
        m_State = server::SessionState_Stopped;
    }
}

void Close() NN_NOEXCEPT
{
    Stop();
    UnmapAllBuffers();
    if (os::InvalidNativeHandle != m_ClientProcessHandle)
    {
        os::CloseNativeHandle(m_ClientProcessHandle);
        m_ClientProcessHandle = os::InvalidNativeHandle;
    }
    m_BufferList.Reset();
    m_ReleaseClientBufferEvent.Signal();
}

void Sleep() NN_NOEXCEPT
{
    if (m_State == server::SessionState_Started)
    {
        m_pDriver->Sleep();
    }
}

void Wake() NN_NOEXCEPT
{
    if (m_State == server::SessionState_Started)
    {
        m_pDriver->Wake();
    }
}

Result RegisterBufferEvent(nn::os::NativeHandle* outHandle) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(outHandle);
    *outHandle = m_ReleaseClientBufferEvent.GetReadableHandle();

    NN_RESULT_SUCCESS;
}

bool ContainsBuffer(const void* buffer) const NN_NOEXCEPT
{
    return m_BufferList.ContainsUserAddr(buffer);
}

bool AppendBuffer(void* buffer, size_t size, const void* userAddr) NN_NOEXCEPT
{
    server::SessionBuffer sessionBuffer (buffer, userAddr, size);
    auto mappedBuffer = m_pDriver->MapBuffer(sessionBuffer, m_ClientProcessHandle);

    if(mappedBuffer)
    {
        bool appended = m_BufferList.AppendServerBuffer(sessionBuffer);

        if(appended)
        {
            AppendDriverBuffers();
            return true;
        }
        else
        {
            m_pDriver->UnmapBuffer(sessionBuffer);
        }
    }
    return false;
}

bool AppendBuffer(void* buffer, size_t size, const void* userAddr, nn::sf::NativeHandle&& bufferHandle) NN_NOEXCEPT
{
    server::SessionBuffer sessionBuffer (buffer, userAddr, size);
    auto mappedBuffer = m_pDriver->MapBuffer(sessionBuffer, std::move(bufferHandle));

    if(mappedBuffer)
    {
        bool appended = m_BufferList.AppendServerBuffer(sessionBuffer);

        if(appended)
        {
            AppendDriverBuffers();
            return true;
        }
        else
        {
            m_pDriver->UnmapBuffer(sessionBuffer);
        }
    }
    return false;
}

int GetReleasedBufferCount() const NN_NOEXCEPT
{
    return m_BufferList.GetReleasedBufferCount();
}

int GetAppendedBufferCount() const NN_NOEXCEPT
{
    return m_BufferList.GetAppendedBufferCount();
}

SampleFormat GetSampleFormat() const NN_NOEXCEPT
{
    return m_Format.format;
}

int GetSampleRate() const NN_NOEXCEPT
{
    return m_Format.sampleRate;
}

uint16_t GetChannelCount() const NN_NOEXCEPT
{
    return m_Format.channelCount;
}

int GetDefaultSampleRate() const NN_NOEXCEPT
{
    return 48000;
}

int GetDefaultChannelCount() const NN_NOEXCEPT
{
    return 2;
}

server::SessionBuffer* GetReleasedBuffer() NN_NOEXCEPT
{
    return const_cast<server::SessionBuffer*>(m_BufferList.GetReleasedBuffer());
}

uint64_t GetSamplesProcessed() const NN_NOEXCEPT
{
    if (m_State == server::SessionState_Stopped)
    {
        return 0;
    }
    return m_pDriver->GetSamplesProcessed();
}

const char* GetName() const NN_NOEXCEPT
{
    return reinterpret_cast<const char*>(m_Name);
}

bool FlushBuffers() NN_NOEXCEPT
{
    if(m_State == server::SessionState_Started)
    {
        return false;
    }
    UnmapAllBuffers();
    return m_BufferList.FlushBuffers();
}

int GetSessionId() const NN_NOEXCEPT
{
    return m_SessionId;
}

Result SetDeviceGain(float gain) NN_NOEXCEPT
{
    return m_pDriver->SetDeviceGain(gain);
}

float GetDeviceGain() const NN_NOEXCEPT
{
    return m_pDriver->GetDeviceGain();
}

Result Update() NN_NOEXCEPT
{
    if (m_State != server::SessionState_Started)
    {
        NN_RESULT_SUCCESS;
    }

    if (m_BufferList.GetDeviceAppendedBufferCount() > 0)
    {
        // Check if the buffer being processing changed
        server::SessionBuffer buffer;
        auto bufferFound = m_BufferList.GetPlaybackBuffer(&buffer);
        bool isBufferReleased = false;
        while (bufferFound && m_pDriver->TryCompleteBuffer(buffer))
        {
            m_BufferList.NotifyBufferComplete();
            bufferFound = m_BufferList.GetPlaybackBuffer(&buffer);
            isBufferReleased = true;
        }
        if (isBufferReleased)
        {
            m_ReleaseClientBufferEvent.Signal();
        }
    }

    AppendDriverBuffers();

    NN_RESULT_SUCCESS;
}

protected:
AppletVolumeManager::SessionType SessionTypeToAppletVolumeManagerType(server::SessionType sessionType) const NN_NOEXCEPT
{
    switch (sessionType)
    {
        case server::SessionType_Output:
            return AppletVolumeManager::SessionType_AudioOut;
        case server::SessionType_Input:
            return AppletVolumeManager::SessionType_AudioIn;
        case server::SessionType_Record:
            return AppletVolumeManager::SessionType_FinalOutputRecorder;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

int AppendDriverBuffers() NN_NOEXCEPT
{
    if(m_State != server::SessionState_Started)
    {
        return 0;
    }
    server::SessionBuffer buffersToAppend[DriverBufferCount];
    auto numBuffers = m_BufferList.GetDeviceBuffersToAppend(buffersToAppend, DriverBufferCount);

    for (int i = 0; i < numBuffers; ++i)
    {
        m_pDriver->AppendBuffer(buffersToAppend[i]);
    }
    return numBuffers;
}

void UnmapAllBuffers() NN_NOEXCEPT
{
    server::SessionBuffer buffers[ServerBufferCount];
    int numBuffers = m_BufferList.GetMappedBuffers(buffers, ServerBufferCount);
    for(int i = 0; i < numBuffers; ++i)
    {
        m_pDriver->UnmapBuffer(buffers[i]);
    }
}

protected:
    nn::dd::ProcessHandle m_ClientProcessHandle;
    nn::applet::AppletResourceUserId m_AppletResourceUserId;
    nn::os::SystemEvent& m_ReleaseClientBufferEvent;
    AudioDriver* m_pDriver;

    server::SessionBufferList<ServerBufferCount> m_BufferList;
    server::SessionType m_Type;
    server::SessionFormat m_Format;
    server::SessionState  m_State;
    int32_t m_SessionId;
    char     m_Name[256];
    bool m_IsOpen;
};

}  // namespace detail
}  // namespace audio
}  // namespace nn
