﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>  // std::min
#include <memory>
#include <mutex>  // std::lock_guard

#include <nn/nn_Abort.h>
#include <nn/nn_SdkAssert.h>
#include <nn/nn_SystemThreadDefinition.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/nn_TimeSpan.h>
#include <nn/audio/detail/audio_Log.h>

#include "audio_AudioOutSession-os.win32.h"

namespace nn { namespace audio { namespace server {

namespace
{
    nn::os::SemaphoreType g_Semaphore;
}

int ListAudioOuts(AudioOutInfo* outAudioOuts, int count) NN_NOEXCEPT
{
    return detail::AudioWasapiDriver::ListDevices(server::SessionType_Output, outAudioOuts, count);
}

void AudioOutInitialize(nn::os::SemaphoreType** semaphore, nn::os::EventType* pEventSemaphore) NN_NOEXCEPT
{
    NN_UNUSED(pEventSemaphore);

    nn::os::InitializeSemaphore(&g_Semaphore, 0, 1);
    *semaphore = &g_Semaphore;
    detail::AudioWasapiDriver::Initialize(server::SessionType_Output, &g_Semaphore);
}

void AudioOutFinalize() NN_NOEXCEPT
{
    detail::AudioWasapiDriver::Finalize(server::SessionType_Output);
}

} //namespace server

namespace detail {

AudioWasapiDriver AudioOutSession::s_WindowsDriver[server::AudioOutManagerImpl::NumberOfSessions];

AudioOutSession::AudioOutSession(int32_t sessionId, nn::os::SystemEvent& releaseClientBufferEvent) NN_NOEXCEPT
    : AudioSession(sessionId, GetDriver(sessionId), releaseClientBufferEvent, server::SessionType_Output)
{
}

AudioOutSession::~AudioOutSession() NN_NOEXCEPT
{
}

const char* AudioOutSession::GetDefaultName() const NN_NOEXCEPT
{
    return s_WindowsDriver[m_SessionId].GetDefaultName(server::SessionType_Output);
}

Result AudioOutSession::IsSupported(const char* name, const server::SessionFormat& format) const NN_NOEXCEPT
{
    return detail::AudioWasapiDriver::IsSupported(server::SessionType_Output, name, format);
}

AudioOutState AudioOutSession::GetState() const NN_NOEXCEPT
{
    switch(m_State)
    {
        case server::SessionState_Started:
            return AudioOutState_Started;
        case server::SessionState_Stopped:
            return AudioOutState_Stopped;
        default:
            NN_UNEXPECTED_DEFAULT;
    }
}

AudioDriver* AudioOutSession::GetDriver(int sessionId) const NN_NOEXCEPT
{
    NN_ABORT_UNLESS_MINMAX(sessionId, 0, static_cast<int>(NN_ARRAY_SIZE(s_WindowsDriver)) - 1);
    return &s_WindowsDriver[sessionId];
}

void AudioOutSession::SetAudioOutVolume(float volume) NN_NOEXCEPT
{
    s_WindowsDriver[m_SessionId].SetDeviceGain(volume);
}

float AudioOutSession::GetAudioOutVolume() const NN_NOEXCEPT
{
    return s_WindowsDriver[m_SessionId].GetDeviceGain();
}

}}} // namespace nn::audio::detail
