﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/audio/audio_Result.h> // ResultSuccess(), ResultInvalidUpdateInfo()
#include <nn/audio/audio_SampleFormat.private.h>
#include <algorithm>
#include "audio_ServiceVoiceInfo.h"
#include "../common/audio_CommonWaveBuffer.h"
#include "../common/audio_SplitterParameters.h"

namespace nn {
namespace audio {
namespace server {

WaveBuffer::WaveBuffer() NN_NOEXCEPT
{
    Initialize();
}

void WaveBuffer::Initialize() NN_NOEXCEPT
{
    startSampleOffset = 0;
    endSampleOffset = 0;
    loop = false;
    isEndOfStream = false;
    buffer.Setup(nullptr, 0);
    pContextInfo.Setup(nullptr, 0);
    _isSentToDsp = true; // initial true.
}

VoiceInfo::VoiceInfo() NN_NOEXCEPT
{
    Initialize();
}

void VoiceInfo::Initialize() NN_NOEXCEPT
{
    // InParameter
    _inParameter._isInUse = false;
    _inParameter._id = 0;
    _inParameter._nodeId = 0;
    _inParameter._playState = PlayState::Stop;
    _inParameter._priority = nn::audio::VoiceType::PriorityLowest;
    _inParameter._sampleRate = 0;
    _inParameter._sampleFormat = nn::audio::SampleFormat_Invalid;
    _inParameter._channelCount = 0;
    _inParameter._pitch = 0.f;
    _inParameter._volume = 0.f;
    _inParameter._volumePrev = 0.f;
    memset(_inParameter._biquadFilterParameter, 0 , sizeof(_inParameter._biquadFilterParameter));
    _inParameter._waveBufferCount = 0;
    _inParameter._waveBufferHead = 0;
    _inParameter._pAdditionalParameterInfo.Setup(nullptr, 0);
    SetDestinationMixId(nn::audio::Invalid_MixId);
    SetSplitterInfoId(common::InvalidSplitterInfoId);
    for (auto i = 0; i < nn::audio::VoiceType::WaveBufferCountMax; ++i)
    {
        _inParameter._waveBuffer[i].Initialize();
    }
    _inParameter._isNew = false;

    // OutStatus
    _outStatus._playedSampleCount = 0;
    _outStatus._waveBufferConsumed = 0;

    voiceDroppedFlag = false;
    nodeIdBase = 0;
    nodeIdVariation = 0;
    noMappedParameter = false;
    noMappedWaveBuffer = false;
    for (auto i = 0; i < nn::audio::VoiceType::BiquadFilterCountMax; ++i)
    {
        isBiquadFilterEnabledPrev[i] = false;
    }
    waveBufferFlushRequestedCount = 0;
}

bool VoiceInfo::ShouldUpdateParameters(const nn::audio::VoiceInfo::InParameter* pInParameter) const NN_NOEXCEPT
{
    auto differentBufferPassed =
        (this->_inParameter._pAdditionalParameterInfo.GetCpuAddr() != pInParameter->_pAdditionalParameter) ||
        (this->_inParameter._pAdditionalParameterInfo.GetSize() != pInParameter->_additionalParameterSize);
    auto shouldRetry = noMappedParameter;
    return differentBufferPassed || shouldRetry;
}

void VoiceInfo::UpdateParameters(common::BehaviorParameter::ErrorInfo* pOutErrorInfo, const nn::audio::VoiceInfo::InParameter* pInParameter, server::PoolMapper& poolMapper, const server::BehaviorInfo& behaviorInfo) NN_NOEXCEPT
{
    this->_inParameter._isInUse                               = pInParameter->_isInUse;
    this->_inParameter._id                                    = pInParameter->_id;
    this->_inParameter._nodeId                                = pInParameter->_nodeId;
    this->UpdatePlayState(pInParameter->_playState);
    this->_inParameter._priority                              = pInParameter->_priority;
    this->_inParameter._sortingOrder                          = pInParameter->_sortingOrder;
    this->_inParameter._sampleRate                            = pInParameter->_sampleRate;
    this->_inParameter._sampleFormat                          = pInParameter->_sampleFormat;
    this->_inParameter._channelCount                          = pInParameter->_channelCount;
    this->_inParameter._pitch                                 = pInParameter->_pitch;
    this->_inParameter._volume                                = pInParameter->_volume;
    memcpy(this->_inParameter._biquadFilterParameter, pInParameter->_biquadFilterParameter, sizeof(this->_inParameter._biquadFilterParameter));
    this->_inParameter._waveBufferCount                       = pInParameter->_waveBufferCount;
    this->_inParameter._waveBufferHead                        = pInParameter->_waveBufferHead;
    this->waveBufferFlushRequestedCount                      += behaviorInfo.IsFlushVoiceWaveBuffersSupported() ? pInParameter->_waveBufferFlushRequestedCount : 0;
    this->SetDestinationMixId(pInParameter->_destinationMixId);
    this->SetSplitterInfoId((behaviorInfo.IsSplitterSupported()) ? pInParameter->_splitterInfoId : common::InvalidSplitterInfoId);
    memcpy(this->_inParameter._voiceChannelResourceIds, pInParameter->_voiceChannelResourceIds, sizeof(this->_inParameter._voiceChannelResourceIds));

    const auto& behaviorFlags = pInParameter->_behaviorFlags;
    const auto isPlayedSampleCountResetAtLoopPoint = behaviorInfo.IsVoicePlayedSampleCountResetAtLoopPointSupported() && behaviorFlags.Get<VoiceBehaviorFlag::IsPlayedSampleCountResetAtLoopPoint>();
    const auto isPitchAndSrcSkipped = behaviorInfo.IsVoicePitchAndSrcSkippedSupported() && behaviorFlags.Get<VoiceBehaviorFlag::IsPitchAndSrcSkipped>();
    this->_inParameter._behaviorFlags.Set<VoiceBehaviorFlag::IsPlayedSampleCountResetAtLoopPoint>(isPlayedSampleCountResetAtLoopPoint);
    this->_inParameter._behaviorFlags.Set<VoiceBehaviorFlag::IsPitchAndSrcSkipped>(isPitchAndSrcSkipped);

    if(pInParameter->_isVoiceDroppedFlagClearRequested)
    {
        this->voiceDroppedFlag = false;
    }

    if (ShouldUpdateParameters(pInParameter))
    {
        noMappedParameter =
            poolMapper.TryAttachBuffer(
                pOutErrorInfo,
                &this->_inParameter._pAdditionalParameterInfo,
                pInParameter->_pAdditionalParameter,
                static_cast<size_t>(pInParameter->_additionalParameterSize)) ? false : true;
    }
    else
    {
        pOutErrorInfo->SetResult(ResultSuccess(), 0);
    }
}

void VoiceInfo::UpdateWaveBuffers(common::BehaviorParameter::ErrorInfo* pOutErrorInfo, int errorInfoCount, const nn::audio::VoiceInfo::InParameter* pInParameter, VoiceState* pStates[], server::PoolMapper& poolMapper, const BehaviorInfo& behaviorInfo) NN_NOEXCEPT
{
    const auto channelCount = pInParameter->_channelCount;
    NN_SDK_ASSERT_MINMAX(channelCount, 1, ::nn::audio::VoiceType::ChannelCountMax);

    // 新規 Voice 時の初期化処理
    if(pInParameter->_isNew)
    {
        // サーバ側 WaveBuffer の初期化
        for (int i = 0; i < ::nn::audio::VoiceType::WaveBufferCountMax; ++i)
        {
            auto serviceBuffer = &this->_inParameter._waveBuffer[i];
            serviceBuffer->Initialize();
        }

        // isWaveBufferValid の初期化
        for(int channel = 0; channel < channelCount; ++channel)
        {
            for (int i = 0; i < ::nn::audio::VoiceType::WaveBufferCountMax; ++i)
            {
                pStates[channel]->isWaveBufferValid[i] = false;
            }
        }
    }

    // Client から送られてきた WaveBuffer を Server 側 WaveBuffer に反映
    NN_SDK_ASSERT(errorInfoCount >= ::nn::audio::VoiceType::WaveBufferCountMax * 2);
    NN_UNUSED(errorInfoCount);
    for (int i = 0; i < ::nn::audio::VoiceType::WaveBufferCountMax; ++i)
    {
        auto clientBuffer = &pInParameter->_waveBuffer[i];
        auto validState = pStates[0]->isWaveBufferValid[i];
        auto serviceBuffer = &this->_inParameter._waveBuffer[i];

        this->UpdateWaveBuffer(&pOutErrorInfo[2 * i], serviceBuffer, clientBuffer, pInParameter->_sampleFormat, validState, poolMapper, behaviorInfo);
    }
}

bool VoiceInfo::ShouldUpdateWaveBuffer(const ::nn::audio::VoiceInfo::WaveBufferInternal* pInWaveBuffer) const NN_NOEXCEPT
{
    auto newBuffer = (pInWaveBuffer->_isSentToServer == false);
    auto shouldRetry = noMappedWaveBuffer;
    return newBuffer || shouldRetry;
}

namespace {
bool CheckSampleOffsetError(const ::nn::audio::VoiceInfo::WaveBufferInternal* pInWaveBuffer, uint8_t format) NN_NOEXCEPT
{
    bool sizeError = false;
    switch (format)
    {
    case SampleFormat_PcmInt16:
        sizeError = (pInWaveBuffer->size < common::CalcBufferSizeOfPcmInt16(pInWaveBuffer->startSampleOffset)) ||
            (pInWaveBuffer->size < common::CalcBufferSizeOfPcmInt16(pInWaveBuffer->endSampleOffset));
        break;
    case SampleFormat_Adpcm:
        sizeError = (pInWaveBuffer->size < common::CalcBufferSizeOfAdpcm(pInWaveBuffer->startSampleOffset)) ||
            (pInWaveBuffer->size < common::CalcBufferSizeOfAdpcm(pInWaveBuffer->endSampleOffset));
        break;
    case SampleFormat_Opus:
        // TODO: error checks
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
    sizeError = sizeError || (pInWaveBuffer->startSampleOffset < 0);
    sizeError = sizeError || (pInWaveBuffer->endSampleOffset < 0);
    return sizeError;
}

}

void VoiceInfo::UpdateWaveBuffer(
    common::BehaviorParameter::ErrorInfo* pOutErrorInfo,
    server::WaveBuffer* pOutWaveBuffer,
    const ::nn::audio::VoiceInfo::WaveBufferInternal* pInWaveBuffer,
    uint8_t format,
    bool validState,
    server::PoolMapper& poolMapper,
    const BehaviorInfo& behaviorInfo) NN_NOEXCEPT
{
    if(validState == false && pOutWaveBuffer->_isSentToDsp )
    {
        if (!pOutWaveBuffer->buffer.GetCpuAddr().IsNullPtr())
        {
            poolMapper.ForceUnmapPointer(&pOutWaveBuffer->buffer);
            pOutWaveBuffer->buffer.Setup(nullptr, 0);
        }
    }

    if(ShouldUpdateWaveBuffer(pInWaveBuffer))
    {
        if (CheckSampleOffsetError(pInWaveBuffer, format))
        {
            pOutErrorInfo[0].SetResult(ResultInvalidUpdateInfo(), pInWaveBuffer->buffer.GetAddress());
            return;
        }

        NN_SDK_ASSERT(pOutWaveBuffer->_isSentToDsp == true);
        pOutWaveBuffer->_isSentToDsp           = false;
        pOutWaveBuffer->startSampleOffset = pInWaveBuffer->startSampleOffset;
        pOutWaveBuffer->endSampleOffset   = pInWaveBuffer->endSampleOffset;
        pOutWaveBuffer->loop              = pInWaveBuffer->loop;
        pOutWaveBuffer->isEndOfStream     = pInWaveBuffer->isEndOfStream;

        // Fill DspAddr.
        // buffer
        noMappedWaveBuffer =
            poolMapper.TryAttachBuffer(
                &pOutErrorInfo[0],
                &pOutWaveBuffer->buffer,
                pInWaveBuffer->buffer,
                static_cast<size_t>(pInWaveBuffer->size)) ? false : true;

        // context
        if (behaviorInfo.IsAdpcmLoopContextBugFixed() &&
            format == SampleFormat_Adpcm && pInWaveBuffer->pContext != nullptr)
        {
            noMappedWaveBuffer =
               (poolMapper.TryAttachBuffer(
                    &pOutErrorInfo[1],
                    &pOutWaveBuffer->pContextInfo,
                    pInWaveBuffer->pContext,
                    static_cast<size_t>(pInWaveBuffer->contextSize)) ? false : true) || noMappedParameter;
        }
        else
        {
            pOutWaveBuffer->pContextInfo.Setup(nullptr, 0);
        }
    }
}

void VoiceInfo::WriteOutStatus(nn::audio::VoiceInfo::OutStatus * outStatus, const nn::audio::VoiceInfo::InParameter* pInParameter, VoiceState* pStates[]) NN_NOEXCEPT
{
#if !defined(NN_SDK_BUILD_RELEASE)
    // 一度以上コマンド生成がされている && 新規に作成を要求されているボイスではない
    // (上記の条件を満たしていないと非クリア状態の VoiceState をチェックする可能性がある)
    if(pInParameter->_isNew == false && this->_inParameter._isNew == false)
    {
        // チャンネル間で VoiceState の値は一部例外を除いて同じであるはず
        const auto channelCount = pInParameter->_channelCount;
        for(int channel = 1; channel < channelCount; ++channel)
        {
            NN_SDK_ASSERT_EQUAL(pStates[0]->waveBufferConsumed, pStates[channel]->waveBufferConsumed);
            NN_SDK_ASSERT_EQUAL(pStates[0]->playedSampleCount,  pStates[channel]->playedSampleCount);
            NN_SDK_ASSERT_EQUAL(pStates[0]->offset,             pStates[channel]->offset);
            NN_SDK_ASSERT_EQUAL(pStates[0]->waveBufferIndex,    pStates[channel]->waveBufferIndex);
            NN_SDK_ASSERT_EQUAL(pStates[0]->fraction,           pStates[channel]->fraction);
            NN_SDK_ASSERT(std::memcmp(pStates[0]->isWaveBufferValid, pStates[channel]->isWaveBufferValid, sizeof(pStates[0]->isWaveBufferValid)) == 0);
        }
    }
#endif

    // pVoice が true, pInParameter が false の時にコピーしないため
    if(pInParameter->_isNew)
    {
        this->_inParameter._isNew = true;
        outStatus->_waveBufferConsumed = 0;
        outStatus->_playedSampleCount = 0;
        outStatus->_voiceDroppedFlag = false;
    }
    else if (!this->_inParameter._isNew)
    {
        outStatus->_waveBufferConsumed = pStates[0]->waveBufferConsumed;
        outStatus->_playedSampleCount = pStates[0]->playedSampleCount;
        outStatus->_voiceDroppedFlag = this->voiceDroppedFlag;
    }
    else // (pInParameter->_isNew == false && this->_inParameter._isNew)
    {
        outStatus->_waveBufferConsumed = 0;
        outStatus->_playedSampleCount = 0;
        outStatus->_voiceDroppedFlag = false;
    }
}

bool VoiceInfo::ShouldSkip() const NN_NOEXCEPT
{
    return _inParameter._isInUse == false ||  // 未使用のボイス
           _inParameter._waveBufferCount == 0 || // WaveBuffer が一つも追加されていない
           noMappedParameter || noMappedWaveBuffer ||  // pVoice にエラーが含まれる ( Unmapped な Buffer への参照など）
           voiceDroppedFlag; // Skip a voice rendering if it was a dropped voice.
}

bool VoiceInfo::HasAnyConnection() const NN_NOEXCEPT
{
    return GetDestinationMixId() != Invalid_MixId || GetSplitterInfoId() != common::InvalidSplitterInfoId;
}

namespace {

bool IsAnyWaveBufferInUsed(const VoiceState* pState) NN_NOEXCEPT
{
    for (const auto isValid : pState->isWaveBufferValid)
    {
        if (isValid)
        {
            return true;
        }
    }
    return false;
}

#if !defined(NN_SDK_BUILD_RELEASE)
bool CheckVoiceStatesForMultiChannel(const VoiceState* const pStates[], int channelCount) NN_NOEXCEPT
{
    if(pStates[0] == nullptr)
    {
        return false;
    }

    // チャンネル間で VoiceState の値は一部例外 (BiquadFilterState など) を除いて同じであるはず
    for(int channel = 1; channel < channelCount; ++channel)
    {
        auto pState = pStates[channel];

        const auto isSameState = pState != nullptr
                              && pStates[0]->waveBufferConsumed == pState->waveBufferConsumed
                              && pStates[0]->playedSampleCount  == pState->playedSampleCount
                              && pStates[0]->offset             == pState->offset
                              && pStates[0]->waveBufferIndex    == pState->waveBufferIndex
                              && pStates[0]->fraction           == pState->fraction
                              && std::memcmp(pStates[0]->isWaveBufferValid, pStates[channel]->isWaveBufferValid, sizeof(pStates[0]->isWaveBufferValid)) == 0;
        if(!isSameState)
        {
            return false;
        }
    }

    return true;
}
#endif // #if !defined(NN_SDK_BUILD_RELEASE)

} // anonymous namespace

void VoiceInfo::FlushWaveBuffers(int count, VoiceState* pStates[], int channelCount) NN_NOEXCEPT
{
    auto& param = this->_inParameter;
    const auto head = param._waveBufferHead;
    auto index = head;

    for(int i = 0; i < count; ++i)
    {
        param._waveBuffer[index]._isSentToDsp = true;

        // Update VoiceStates for each channel.
        for(int channel = 0; channel < channelCount; ++channel)
        {
            auto pState = pStates[channel];
            pState->waveBufferIndex = (pState->waveBufferIndex + 1) % VoiceType::WaveBufferCountMax;
            ++pState->waveBufferConsumed;
            pState->isWaveBufferValid[index] = false;
        }

        // Update an index for WaveBuffer
        index = (index + 1) % VoiceType::WaveBufferCountMax;
    }
}

bool VoiceInfo::UpdateParametersForCommandGeneration(VoiceState* pStates[]) NN_NOEXCEPT
{
    const auto channelCount = _inParameter._channelCount;

    NN_SDK_ASSERT(CheckVoiceStatesForMultiChannel(pStates, channelCount));

    auto& param = this->_inParameter;

    // Flush WaveBuffers if it is requested.
    const auto flushRequestedCount = this->waveBufferFlushRequestedCount;
    if(flushRequestedCount > 0)
    {
        FlushWaveBuffers(flushRequestedCount, pStates, channelCount);

        // Clear amount of requested count from app.
        this->waveBufferFlushRequestedCount = 0;
    }

    switch(param._playState)
    {
    case PlayState::Play:
        for (int j = 0; j < VoiceType::WaveBufferCountMax; ++j)
        {
            if (this->_inParameter._waveBuffer[j]._isSentToDsp == false)
            {
                for(int channel = 0; channel < channelCount; ++channel)
                {
                    auto pState = pStates[channel];
                    NN_SDK_ASSERT(pState->isWaveBufferValid[j] == false);
                    pState->isWaveBufferValid[j] = true;
                }

                this->_inParameter._waveBuffer[j]._isSentToDsp = true;
            }
        }
        param._shouldDepop = false;

        // Above for loop, update all "pState->isWaveBufferValid" states.
        // So here we can confirm if "is this voice need some command generation ?",
        // by checking "pState->isWaveBufferValid" states
        if (IsAnyWaveBufferInUsed(pStates[0]) == false)
        {
            return false;
        }
        else
        {
            return true;
        }
    case PlayState::Stop:
    case PlayState::Pause:
        // MemoryPool を「利用」している状態を明示するために、各種ポインタへの参照のみを行う。
        // 内部で SericeMemoryPoolInfo::IsUsed() を呼ぶことで、その状態を明示する形を取っている。
        // PlayState::Stop の際も「停止中に追加された WaveBuffer」を維持するために、参照を取る。
        for (int j = 0; j < VoiceType::WaveBufferCountMax; ++j)
        {
            if (param._waveBuffer[j]._isSentToDsp == false)
            {
                // check these buffer as "using"
                param._waveBuffer[j].buffer.GetReference();
                param._waveBuffer[j].pContextInfo.GetReference();
            }
        }
        if (param._sampleFormat == SampleFormat_Adpcm &&
            param._pAdditionalParameterInfo.GetCpuAddr().IsNullPtr() == false)
        {
            param._pAdditionalParameterInfo.GetReference();
        }
        param._shouldDepop = (param._playStatePrev == PlayState::Play);
        return param._shouldDepop;
    case PlayState::StopRequested:
        // すべての有効な WaveBuffer は再生済みとして扱う
        for (int j = 0; j < VoiceType::WaveBufferCountMax; ++j)
        {
            param._waveBuffer[j]._isSentToDsp = true;

            for(int channel = 0; channel < channelCount; ++channel)
            {
                auto pState = pStates[channel];
                if (pState->isWaveBufferValid[j])
                {
                    pState->waveBufferIndex = (pState->waveBufferIndex + 1) % VoiceType::WaveBufferCountMax;
                    ++pState->waveBufferConsumed;
                }
                pState->isWaveBufferValid[j] = false;
            }
        }
        // DSP でクリアされる BiquadFilterState と Depop で使われる lastSamples 以外をクリア
        // TODO: 基本 ADSP 側で閉じて利用するバッファは ADSP 側でクリアするほうがよいかもしれない。後日再検討。
        for(int channel = 0; channel < channelCount; ++channel)
        {
            auto pState = pStates[channel];
            pState->offset = 0;
            pState->playedSampleCount = 0;
            memset(pState->sampleHistory, 0, sizeof(pState->sampleHistory));
            pState->fraction = 0;
            memset(pState->context, 0, sizeof(pState->context));
        }
        param._playState = PlayState::Stop;
        param._shouldDepop = (param._playStatePrev == PlayState::Play);
        return param._shouldDepop;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

bool VoiceInfo::UpdateForCommandGeneration(VoiceContext* data) NN_NOEXCEPT
{
    VoiceState* pVoiceStates[::nn::audio::VoiceType::ChannelCountMax] = { nullptr };
    NN_SDK_ASSERT_MINMAX(_inParameter._channelCount, 1, ::nn::audio::VoiceType::ChannelCountMax);

    if (_inParameter._isNew)
    {
        ResetResources(data);
        // this avoids applying volume ramp for the first audio frame
        _inParameter._volumePrev = _inParameter._volume;
        _inParameter._isNew = false;
    }


    for (int channel = 0; channel < _inParameter._channelCount; ++channel)
    {
        const auto channelResourceId = _inParameter._voiceChannelResourceIds[channel];
        pVoiceStates[channel] = &data->GetDspSharedState(channelResourceId);
    }

    return UpdateParametersForCommandGeneration(pVoiceStates);
}

void VoiceInfo::ResetResources(VoiceContext* data) NN_NOEXCEPT
{
    const auto channelCount = _inParameter._channelCount;

    NN_SDK_ASSERT_MINMAX(channelCount, 1, ::nn::audio::VoiceType::ChannelCountMax);

    for (auto j = 0; j < channelCount; ++j)
    {
        const auto channelResourceId = _inParameter._voiceChannelResourceIds[j];
        NN_SDK_ASSERT(data->GetChannelResource(channelResourceId).IsInUse());

        memset(&data->GetDspSharedState(channelResourceId), 0, sizeof(VoiceState));
        if (_inParameter._sampleFormat == SampleFormat_Opus)
        {
            data->GetDspSharedState(channelResourceId).externalContext = data->GetState(channelResourceId).externalContext;
            data->GetDspSharedState(channelResourceId).externalContextSize = data->GetState(channelResourceId).externalContextSize;
        }
        // this avoids applying volume ramp for the first audio frame
        data->GetChannelResource(channelResourceId).UpdateInternalState();
    }
}

void VoiceInfo::UpdatePlayState(uint8_t state) NN_NOEXCEPT
{
    _inParameter._playStatePrev = _inParameter._playState;
    switch(state)
    {
    case nn::audio::VoiceType::PlayState_Pause:
        _inParameter._playState = PlayState::Pause;
        break;
    case nn::audio::VoiceType::PlayState_Play:
        _inParameter._playState = PlayState::Play;
        break;
    case nn::audio::VoiceType::PlayState_Stop:
        if (_inParameter._playState != PlayState::Stop)
        {
            _inParameter._playState = PlayState::StopRequested;
        }
        break;
    default:
        NN_UNEXPECTED_DEFAULT;
    }
}

void VoiceContext::Initialize(server::VoiceInfo** sortedInfos, server::VoiceInfo* infos, VoiceChannelResource* channelResources, VoiceState* states, VoiceState* dspSharedStates, int infoCount) NN_NOEXCEPT
{
    m_SortedInfos = sortedInfos;
    m_Infos = infos;
    m_ChannelResources = channelResources;
    m_States = states;
    m_DspSharedStates = dspSharedStates;
    m_InfoCount = infoCount;
    m_ActiveCount = 0;
}

VoiceContext::VoiceContext() NN_NOEXCEPT
    : m_SortedInfos(nullptr)
    , m_Infos(nullptr)
    , m_ChannelResources(nullptr)
    , m_States(nullptr)
    , m_DspSharedStates(nullptr)
    , m_InfoCount(0)
    , m_ActiveCount(0)
{}

nn::audio::server::VoiceInfo& VoiceContext::GetSortedInfo(int id) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(id, 0, m_InfoCount);
    return *m_SortedInfos[id];
}

nn::audio::server::VoiceInfo& VoiceContext::GetInfo(int id) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(id, 0, m_InfoCount);
    return m_Infos[id];
}

nn::audio::server::VoiceChannelResource& VoiceContext::GetChannelResource(int id) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(id, 0, m_InfoCount);
    return m_ChannelResources[id];
}

nn::audio::VoiceState& VoiceContext::GetState(int id) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(id, 0, m_InfoCount);
    return m_States[id];
}

nn::audio::VoiceState& VoiceContext::GetDspSharedState(int id) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(id, 0, m_InfoCount);
    return m_DspSharedStates[id];
}

int VoiceContext::GetCount() const NN_NOEXCEPT
{
    return m_InfoCount;
}

int VoiceContext::GetActiveCount() const NN_NOEXCEPT
{
    return m_ActiveCount;
}

void VoiceContext::SetActiveCount(int activeCount) NN_NOEXCEPT
{
    m_ActiveCount = activeCount;
}

void VoiceContext::SortInfo() NN_NOEXCEPT
{
    for (int i = 0; i < m_InfoCount; ++i)
    {
        m_SortedInfos[i] = &m_Infos[i];
        NN_SDK_ASSERT_NOT_NULL(m_SortedInfos[i]);
    }

    // Sort by reversed voice's sortingOrder (Lower to higher)
    std::sort(m_SortedInfos, m_SortedInfos + m_InfoCount,
        [](const server::VoiceInfo* pLeft, const server::VoiceInfo* pRight)
    {
        NN_SDK_ASSERT_NOT_NULL(pLeft);
        NN_SDK_ASSERT_NOT_NULL(pRight);

        if (pLeft->_inParameter._priority != pRight->_inParameter._priority)
        {
            return pLeft->_inParameter._priority > pRight->_inParameter._priority;
        }
        else
        {
            return pLeft->_inParameter._sortingOrder > pRight->_inParameter._sortingOrder;
        }
    });
}

void VoiceContext::UpdateStateByDspShared() NN_NOEXCEPT
{
    memcpy(m_States, m_DspSharedStates, sizeof(VoiceState) * m_InfoCount);
}

}  // namespace server
}  // namespace audio
}  // namespace nn
