﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <cstring> // memset, memcpy
#include "audio_ServiceMixInfo.h"
#include "audio_ServiceSplitterInfo.h"
#include "audio_ServiceEffectInfo.h"

namespace nn {
namespace audio {
namespace server {


const int32_t MixInfo::InvalidDistanceFromFinalMix = std::numeric_limits<int32_t>::min();

MixInfo::MixInfo(int32_t* processingOrder, int32_t orderCountMax, const server::BehaviorInfo& behaviorInfo) NN_NOEXCEPT
{
    CleanUp();
    this->effectProcessingOrder = processingOrder;
    this->effectProcessingOrderCountMax = orderCountMax;
    longSizePreDelaySupported = behaviorInfo.IsLongSizePreDelaySupported();

    ClearEffectProcessingOrder();
}

void MixInfo::Update(
    common::EdgeMatrix& edges,
    const nn::audio::MixInfo& inMix,
    const EffectContext& effectContext,
    const SplitterContext& splitterContext,
    const server::BehaviorInfo& behaviorInfo) NN_NOEXCEPT
{
    volume = inMix.volume;
    sampleRate = inMix.sampleRate;
    bufferCount = inMix.bufferCount;
    isInUse = inMix.isInUse;
    mixId = inMix.mixId;
    nodeId = inMix.nodeId;
    memcpy(mixVolume, inMix.mixVolume, sizeof(float) * nn::audio::MixBufferCountMax * nn::audio::MixBufferCountMax);

    if (behaviorInfo.IsSplitterSupported())
    {
        UpdateConnection(edges, inMix, splitterContext);
    }
    else
    {
        destinationMixId = inMix.destinationMixId;
        splitterInfoId = common::InvalidSplitterInfoId;
    }

    // Refresh Effect Processing order
    NN_SDK_ASSERT(effectContext.GetCount() <= effectProcessingOrderCountMax);
    auto effectOrder = reinterpret_cast<int32_t*>(effectProcessingOrder.GetPointer());
    for (auto i = 0; i < effectProcessingOrderCountMax; ++i)
    {
        NN_SDK_ASSERT_NOT_NULL(effectOrder);
        effectOrder[i] = nn::audio::EffectInfo::InParameter::InvalidProcessingOrder;
    }
    for (auto i = 0; i < effectContext.GetCount(); ++i)
    {
        NN_SDK_ASSERT(effectProcessingOrderCountMax > 0);
        if (effectContext.GetInfo(i).GetMixId() == mixId)
        {
            auto effectSlotIndex = i;
            auto processingOrder = effectContext.GetInfo(i).GetProcessingOrder();
            NN_SDK_ASSERT(processingOrder < effectProcessingOrderCountMax);
            if (processingOrder > effectProcessingOrderCountMax)
            {
                break;
            }
            NN_SDK_ASSERT_NOT_NULL(effectOrder);
            effectOrder[processingOrder] = effectSlotIndex;
        }
    }
}

void MixInfo::ClearEffectProcessingOrder() NN_NOEXCEPT
{
    auto orderArray = reinterpret_cast<int32_t*>(this->effectProcessingOrder.GetPointer());
    for (auto i = 0; i < this->effectProcessingOrderCountMax; ++i)
    {
        orderArray[i] = nn::audio::EffectInfo::InParameter::InvalidProcessingOrder;
    }
}

bool MixInfo::HasAnyConnection() const NN_NOEXCEPT
{
    return destinationMixId != Invalid_MixId || splitterInfoId != common::InvalidSplitterInfoId;
}

void MixInfo::CleanUp() NN_NOEXCEPT
{
    volume = 0;
    sampleRate = 0;
    bufferCount = 0;
    isInUse = false;
    longSizePreDelaySupported = false;
    memset(_padding, 0, sizeof(_padding));

    mixId = Invalid_MixId;
    nodeId = 0;
    bufferOffset = 0;
    destinationMixId = InvalidDistanceFromFinalMix;
    effectProcessingOrder = nullptr;
    effectProcessingOrderCountMax = 0;

    destinationMixId = Invalid_MixId;
    memset(mixVolume, 0, sizeof(mixVolume));
    splitterInfoId = common::InvalidSplitterInfoId;
    memset(reserved, 0, sizeof(reserved));
}

void MixInfo::UpdateConnection(
    common::EdgeMatrix& edges,
    const nn::audio::MixInfo& inMix,
    const SplitterContext& splitterContext) NN_NOEXCEPT
{
    // 更新の有無の確認
    {
        bool sameMixId = this->destinationMixId == inMix.destinationMixId;
        bool sameSplitter = this->splitterInfoId == inMix.splitterInfoId;
        bool sameSplitterDestinations = (this->splitterInfoId == common::InvalidSplitterInfoId) ?
            true : splitterContext.GetInfo(this->splitterInfoId).HasNewConnection() == false;
        if (sameMixId && sameSplitter && sameSplitterDestinations)
        {
            return;
        }
    }

    // Refresh all edges connected to this MixInfo
    edges.RemoveEdges(this->mixId);
    if (inMix.destinationMixId != Invalid_MixId)
    {
        edges.Connect(this->mixId, inMix.destinationMixId);
    }
    else if (inMix.splitterInfoId != common::InvalidSplitterInfoId)
    {
        auto& splitter = splitterContext.GetInfo(inMix.splitterInfoId);
        for (auto i = 0; i < splitter.GetDestinationCount(); ++i)
        {
            auto dest = splitterContext.GetDestinationData(inMix.splitterInfoId, i);
            if (dest && dest->GetMixId() != Invalid_MixId)
            {
                edges.Connect(this->mixId, dest->GetMixId());
            }
        }
    }

    // update destination info
    destinationMixId = inMix.destinationMixId;
    splitterInfoId = inMix.splitterInfoId;
}

MixContext::MixContext() NN_NOEXCEPT
    : m_SortedInfos(nullptr)
    , m_Infos(nullptr)
    , m_InfoCount(0)
    , m_EffectProcessingOrderBuffer(nullptr)
    , m_EffectProcessingOrderBufferSize(0)
{}

void MixContext::Initialize(
    server::MixInfo** sorted, server::MixInfo* info, int infoCount,
    int32_t* effectProcessingOrderBuffer, size_t effectProcessingOrderBufferSize,
    void* nodeStateBuffer, size_t nodeStateBufferSize,
    void* edgeMatrixBuffer, size_t edgeMatrixBufferSize) NN_NOEXCEPT
{
    m_SortedInfos = sorted;
    m_Infos = info;
    m_InfoCount = infoCount;
    m_EffectProcessingOrderBuffer = effectProcessingOrderBuffer;
    m_EffectProcessingOrderBufferSize = effectProcessingOrderBufferSize;

    if (nodeStateBuffer != nullptr && edgeMatrixBuffer != nullptr)
    {
        m_NodeState.Initialize(nodeStateBuffer, nodeStateBufferSize, infoCount);
        m_Edges.Initialize(edgeMatrixBuffer, edgeMatrixBufferSize, infoCount);
    }

    for (auto i = 0; i < m_InfoCount; ++i)
    {
        m_SortedInfos[i] = &m_Infos[i];
    }
}

nn::audio::server::MixInfo& MixContext::GetSortedInfo(int index) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(index, 0, m_InfoCount);
    return *m_SortedInfos[index];
}

void MixContext::SetSortedInfo(int index, server::MixInfo* pInfo) NN_NOEXCEPT
{
    m_SortedInfos[index] = pInfo;
}

nn::audio::server::MixInfo& MixContext::GetInfo(MixId id) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(id, 0, m_InfoCount);
    return m_Infos[id];
}

nn::audio::server::MixInfo& MixContext::GetFinalMixInfo() const NN_NOEXCEPT
{
    return GetInfo(MixInfo::FinalMixId);
}

int MixContext::GetCount() const NN_NOEXCEPT
{
    return m_InfoCount;
}

void MixContext::UpdateDistancesFromFinalMix() NN_NOEXCEPT
{
    // 距離変数の初期化
    for (auto i = 0; i < GetCount(); ++i)
    {
        GetInfo(i).distanceFromFinalMix = MixInfo::InvalidDistanceFromFinalMix;
    }

    // FinalMix までの距離を計算
    for (auto i = 0; i < GetCount(); ++i)
    {
        auto* pMixInfo = &GetInfo(i);
        SetSortedInfo(i, pMixInfo); // TODO: なぜここで初期化するのか？

        if (!pMixInfo->isInUse)
        {
            continue;
        }

        auto destinationMixId = pMixInfo->mixId;
        auto distance = 0;

        // FinalMix まで探索
        for (; distance < GetCount(); ++distance)
        {
            if (destinationMixId == MixInfo::FinalMixId)
            {
                break;
            }
            else if (destinationMixId == Invalid_MixId)
            {
                // FinalMix に接続されていなかった時は無効な距離を設定
                distance = MixInfo::InvalidDistanceFromFinalMix;
                break;
            }
            else
            {
                auto pDestination = &GetInfo(destinationMixId);
                // 接続先 Mix が既に FinalMix までの距離を計算済みだった時は、その結果を利用する
                if (pDestination->distanceFromFinalMix != MixInfo::InvalidDistanceFromFinalMix)
                {
                    distance = pDestination->distanceFromFinalMix + 1;
                    break;
                }
                destinationMixId = pDestination->destinationMixId;
            }
        }

        // 接続の循環を検出した時は無効な距離を設定
        // (循環していなければ、距離が Mix の個数以上になることはありえない)
        if (distance >= GetCount())
        {
            distance = MixInfo::InvalidDistanceFromFinalMix;
        }

        // 計算した FinalMix までの距離を保存
        pMixInfo->distanceFromFinalMix = distance;
    }
}

void MixContext::SortInfo() NN_NOEXCEPT
{
    // original sort logic.
    UpdateDistancesFromFinalMix();

    // FinalMix からの距離を基にソート
    // TODO: std::sort は In-Place ソートかどうか保障がないので置き換えを検討する
    std::sort(m_SortedInfos, m_SortedInfos + m_InfoCount,
        [](const server::MixInfo* pLeft, const server::MixInfo* pRight)
    {
        return pLeft->distanceFromFinalMix > pRight->distanceFromFinalMix;
    });

    CalcMixBufferOffset();
}

bool MixContext::TsortInfo(const SplitterContext& splitterContext) NN_NOEXCEPT
{
    if (splitterContext.UsingSplitter() == false)
    {
        CalcMixBufferOffset();
        return true;
    }

    if (m_NodeState.Tsort(m_Edges) == false)
    {
        return false;
    }
    auto i = 0;
    for (auto itr = m_NodeState.ResultBegin(); itr != m_NodeState.ResultEnd(); ++itr)
    {
        NN_SDK_ASSERT_LESS(i, m_InfoCount);
        if (i >= m_InfoCount)
        {
            break;
        }
        SetSortedInfo(i++, &m_Infos[*itr]);
    }

    CalcMixBufferOffset();

    return true;
}

void MixContext::CalcMixBufferOffset() NN_NOEXCEPT
{
    // Buffer ごとの mixBufferOffset を計算
    int32_t mixBufferOffset = 0;
    for (int i = 0; i < m_InfoCount; ++i)
    {
        auto p = &GetSortedInfo(i);
        if (p->isInUse)
        {
            p->bufferOffset = mixBufferOffset;
            mixBufferOffset += p->bufferCount;
        }
    }
}

}  // namespace server
}  // namespace audio
}  // namespace nn
