﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

// for Server side library header

#include <nn/nn_Macro.h>
#include <nn/audio/audio_AudioRendererTypes.h>
#include <nn/audio/detail/audio_AudioRendererTypesInternal.h>
#include <nn/audio/audio_PerformanceMetricsTypes.h>
#include <nn/audio/audio_PerformanceMetricsTypes.private.h>
#include "audio_ServiceMemoryPoolInfo.h"
#include "audio_ServiceBehaviorInfo.h"
#include "../common/audio_Util.h"
#include "../common/audio_PerformanceMetricsCommon.h"
#include "../common/audio_UpdateDataHeader.h" // RevisionInfo

namespace nn { namespace audio { namespace server {

struct PerformanceStatistics
{
    // overall statistics
    int32_t processingTimeMax;     //!< 1オーディオフレームで使用した最大処理時間
    int32_t processingTimeMin;     //!< 1オーディオフレームで使用した最少処理時間
    int32_t glitchCount;           //!< AudioRenderer が open されてから発生したグリッチの回数
};

struct PerformanceEntryAddresses
{
    DspAddr dspFrameBase;
    int32_t offsetToStartTime;
    int32_t offsetToCount;
    int32_t offsetToProcessingTime;
    int8_t _reserved[4];
};

// 実装のインターフェイスクラス
class IPerformanceManagerImpl
{
public:
    IPerformanceManagerImpl() NN_NOEXCEPT = default;
    virtual ~IPerformanceManagerImpl() NN_NOEXCEPT = default;
    virtual void Initialize(void* buffer, size_t bufferSize, const detail::AudioRendererParameterInternal& parameter, const BehaviorInfo& behavior, const server::MemoryPoolInfo& servicePool) NN_NOEXCEPT = 0;
    virtual bool IsInitialized() const NN_NOEXCEPT = 0;
    virtual size_t CopyHistories(void* buffer, size_t bufferSize) NN_NOEXCEPT = 0;
    virtual bool GetNextEntry(PerformanceEntryAddresses* block, PerformanceEntryType target, NodeId targetId = 0) NN_NOEXCEPT = 0;
    virtual bool GetNextEntry(PerformanceEntryAddresses* block, PerformanceDetailType target, PerformanceEntryType parentType, NodeId parentId) NN_NOEXCEPT = 0;
    virtual bool GetNextEntry(PerformanceEntryAddresses* block, uint32_t** ppEstimatedProcessingTime, PerformanceSysDetailType target, NodeId parentId) NN_NOEXCEPT = 0;
    virtual void SetDetailTarget(NodeId targetId) NN_NOEXCEPT = 0;
    virtual bool IsDetailTarget(NodeId id) const NN_NOEXCEPT = 0;
    virtual void TapFrame(bool isRenderingTimeLimitExceeded, int voiceDropCount, int64_t startTime) NN_NOEXCEPT = 0;
};

// タグ構造体
struct PerformanceManagerImplVersion1Tag;
struct PerformanceManagerImplVersion2Tag;

// 実装
template<typename Tag, typename HeaderType, typename EntryType, typename DetailType, typename SysDetailType>
class PerformanceManagerImpl : public IPerformanceManagerImpl
{
    NN_DISALLOW_COPY( PerformanceManagerImpl );
    NN_DISALLOW_MOVE( PerformanceManagerImpl );

private:
    struct PerformanceFrameAddresses
    {
        HeaderType* header;
        EntryType* entry;
        DetailType* detail;
    };

public:
    PerformanceManagerImpl() NN_NOEXCEPT = default;
    virtual ~PerformanceManagerImpl() NN_NOEXCEPT = default;

    static size_t GetRequiredBufferSizeForPerformanceMetricsPerFrame(const detail::AudioRendererParameterInternal& parameter) NN_NOEXCEPT
    {
        return common::GetRequiredBufferSizeForPerformanceMetricsPerFrame<HeaderType, EntryType, DetailType>(parameter);
    }

    void Initialize(void* buffer, size_t bufferSize, const detail::AudioRendererParameterInternal& parameter, const BehaviorInfo& behavior, const server::MemoryPoolInfo& servicePool) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(buffer);

        // calculate each frame size.
        m_EntryCountPerFrame = common::GetRequiredEntryCount(parameter);
        m_DetailCountPerFrame = common::GetRequiredDetailCount(parameter);
        m_OneFrameSize = sizeof(HeaderType) +
                        sizeof(EntryType) * m_EntryCountPerFrame +
                        sizeof(DetailType) * m_DetailCountPerFrame;

        int32_t totalFrameCount = static_cast<int>(bufferSize / m_OneFrameSize);

        m_DspBuffer = static_cast<int8_t*>(buffer);

        // Note: 実際に DSP からアクセスされるのは [buffer, buffer + m_OneFrameSize) の領域のみ
        m_DspMappedBuffer = servicePool.Translate(buffer, bufferSize);

        totalFrameCount--; // 1 frame 分を DSP がアクセスする領域として確保。残りが history を保管する領域となる
        m_CircularBase = static_cast<int8_t*>(buffer) + m_OneFrameSize;
        m_HistoryCountMax = totalFrameCount;

        // setup first PerformanceMetrics address.
        GetCurrentFrameAddress(&m_CurrentFrame);
        GetHistoryFrameAddress(&m_HistoryFrame, 0);

        m_DetailTarget = 0;
        m_DataFormatVersion = behavior.GetPerformanceMetricsDataFormat();

        m_HistoryIndex = 0;
        m_HistoryReadIndex = 0;

        // m_EntryOffset と m_DetailOffset は ResetEntryCount() 内部で初期化される
        ResetEntryCount();

        m_Initialized = true;
    }

    bool IsInitialized() const NN_NOEXCEPT NN_OVERRIDE
    {
        return m_Initialized;
    }

    // history 領域に書かれているパフォーマンス情報を取り出す
    size_t CopyHistories(void* buffer, size_t bufferSize) NN_NOEXCEPT NN_OVERRIDE
    {
        if ( (buffer == nullptr) || (bufferSize == 0) || (m_Initialized == false) )
        {
            return 0;
        }

        auto base = nn::util::BytePtr(buffer);
        auto leftSize = bufferSize;
        size_t totalCopySize = 0;
        HeaderType* destHeader = nullptr;

        while (m_HistoryReadIndex != m_HistoryIndex)
        {
            PerformanceFrameAddresses frame;
            GetHistoryFrameAddress(&frame, m_HistoryReadIndex);


            // check left size roughly.
            if (leftSize <  (sizeof(HeaderType) +
                            sizeof(EntryType) * frame.header->entryCount +
                            sizeof(DetailType) * frame.header->detailCount) +
                            sizeof(HeaderType) /* margin for last FrameHeader which must be filled zero */ )
            {
                break; // shortage buffer
            }

            // copy entries & details
            size_t copyOffset = sizeof(HeaderType);
            int copiedEntryCount = 0;
            int copiedDetailCount = 0;
            int totalProcessingTime = 0;
            for (auto i = 0; i < frame.header->entryCount; ++i)
            {
                if ( (frame.entry[i].processingTime == 0) && (frame.entry[i].startTime == 0))
                {
                    continue;
                }
                memcpy((base + copyOffset).Get(), &frame.entry[i], sizeof(EntryType));
                copyOffset += sizeof(EntryType);
                ++copiedEntryCount;

                totalProcessingTime += frame.entry[i].processingTime;
            }

            for (auto i = 0; i < frame.header->detailCount; ++i)
            {
                if ((frame.detail[i].processingTime == 0) && (frame.detail[i].startTime == 0))
                {
                    continue;
                }

                memcpy((base + copyOffset).Get(), &frame.detail[i], sizeof(DetailType));
                copyOffset += sizeof(DetailType);
                ++copiedDetailCount;
            }

            // update header info
            const auto copiedSize = copyOffset;
            destHeader = reinterpret_cast<HeaderType*>(base.Get());
            UpdateHeader(destHeader,
                         frame.header,
                         totalProcessingTime,
                         copiedSize,
                         copiedEntryCount,
                         copiedDetailCount);

            base += copiedSize;
            leftSize -= copiedSize;
            totalCopySize += copiedSize;

            ++m_HistoryReadIndex;
            m_HistoryReadIndex %= m_HistoryCountMax;
        }

        // destHeader が非 nullptr ならば、上記処理で margin を確保しているので、
        // leftSize > sizeof(PerformanceFrameHeader) は必ず成功する想定
        // NN_SDK_ASSERT でも十分だが、失敗を穏便に収めるために念のためチェックする
        if (destHeader && leftSize > sizeof(HeaderType))
        {
            // fill zero, next to last frame's header
            destHeader = reinterpret_cast<HeaderType*>(base.Get());
            memset(destHeader, 0, sizeof(HeaderType));
        }

        NN_SDK_ASSERT_LESS_EQUAL(totalCopySize, std::numeric_limits<size_t>::max());
        return totalCopySize;
    }

    bool GetNextEntry(PerformanceEntryAddresses* block, PerformanceEntryType target, NodeId targetId = 0) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(block);

        if (m_Initialized == false)
        {
            return false;
        }

        block->dspFrameBase = m_DspMappedBuffer;

        auto bufferBase = reinterpret_cast<uintptr_t>(m_DspBuffer);
        block->offsetToCount = static_cast<int32_t>(reinterpret_cast<uintptr_t>(&m_CurrentFrame.header->entryCount) - bufferBase);

        EntryType* entry = m_CurrentFrame.entry + m_EntryOffset;
        block->offsetToStartTime = static_cast<int32_t>(reinterpret_cast<uintptr_t>(&entry->startTime) - bufferBase);
        block->offsetToProcessingTime = static_cast<int32_t>(reinterpret_cast<uintptr_t>(&entry->processingTime) - bufferBase);

        ++m_EntryOffset;
        memset(entry, 0, sizeof(EntryType));

        // Set type & id in advance.
        entry->entryType = static_cast<int8_t>(target);
        entry->id = targetId;

        return true;
    }

    bool GetNextEntry(PerformanceEntryAddresses* block, PerformanceDetailType target, PerformanceEntryType parentType, NodeId parentId) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_SDK_ASSERT_NOT_NULL(block);

        if (m_Initialized == false || m_DetailOffset > nn::audio::common::PerformanceDetailCountMax)
        {
            return false;
        }

        block->dspFrameBase = m_DspMappedBuffer;

        auto bufferBase = reinterpret_cast<uintptr_t>(m_DspBuffer);
        block->offsetToCount = static_cast<int32_t>(reinterpret_cast<uintptr_t>(&m_CurrentFrame.header->detailCount) - bufferBase);

        DetailType* detail = m_CurrentFrame.detail + m_DetailOffset;
        block->offsetToStartTime = static_cast<int32_t>(reinterpret_cast<uintptr_t>(&detail->startTime) - bufferBase);
        block->offsetToProcessingTime = static_cast<int32_t>(reinterpret_cast<uintptr_t>(&detail->processingTime) - bufferBase);

        ++m_DetailOffset;
        memset(detail, 0, sizeof(DetailType));

        // Set type & id in advance.
        detail->detailType = static_cast<int8_t>(target);
        detail->parentEntryType = static_cast<int8_t>(parentType);
        detail->parentId = parentId;

        return true;
    }

    bool GetNextEntry(PerformanceEntryAddresses* block, uint32_t** ppEstimatedProcessingTime, PerformanceSysDetailType target, NodeId parentId) NN_NOEXCEPT NN_OVERRIDE;

    void SetDetailTarget(NodeId targetId) NN_NOEXCEPT NN_OVERRIDE
    {
        m_DetailTarget = targetId;
    }

    bool IsDetailTarget(NodeId id) const NN_NOEXCEPT NN_OVERRIDE
    {
        return id == m_DetailTarget;
    }

    void TapFrame(bool isRenderingTimeLimitExceeded, int voiceDropCount, int64_t startTime) NN_NOEXCEPT NN_OVERRIDE
    {
        if (m_Initialized == false)
        {
            return;
        }

        SaveLastFrameToHistory(isRenderingTimeLimitExceeded, voiceDropCount, startTime);
        ResetEntryCount();
    }

private:
    void UpdateHeader(HeaderType* pDestHeader, const HeaderType* pSrcHeader, int totalProcessingTime, size_t size, int entryCount, int detailCount) NN_NOEXCEPT;
    void CopyFrame(HeaderType* destFrame, const HeaderType* srcFrame, bool isRenderingTimeLimitExceeded, int voiceDropCount, int64_t startTime) NN_NOEXCEPT;

    void ResetEntryCount() NN_NOEXCEPT
    {
        // cleanup dsp frame.
        m_EntryOffset = 0;
        m_DetailOffset = 0;

        auto pHeader = m_CurrentFrame.header;
        pHeader->entryCount = 0;
        pHeader->detailCount = 0;
    }

    void SaveLastFrameToHistory(bool isRenderingTimeLimitExceeded, int voiceDropCount, int64_t startTime) NN_NOEXCEPT
    {
        // 書き出し先のインデックスを取得
        // m_DspFrame からコピー
        PerformanceFrameAddresses user, dsp;
        auto index = GetNextHistoryIndex();
        if (index >= 0)
        {
            GetHistoryFrameAddress(&user, index);
            GetCurrentFrameAddress(&dsp);

            if(user.header != nullptr && dsp.header != nullptr)
            {
                CopyFrame(user.header, dsp.header, isRenderingTimeLimitExceeded, voiceDropCount, startTime);
            }
        }
    }

    void GetCurrentFrameAddress(PerformanceFrameAddresses* block) NN_NOEXCEPT
    {
        SetFrameAddress(block, m_DspBuffer, 0);
    }

    void SetFrameAddress(PerformanceFrameAddresses* block, int8_t* baseAddr, int frameIndex) NN_NOEXCEPT
    {
        block->header = reinterpret_cast<HeaderType*>(baseAddr + (m_OneFrameSize * frameIndex));
        block->entry = reinterpret_cast<EntryType*>(block->header + 1);
        block->detail = reinterpret_cast<DetailType*>(block->entry + m_EntryCountPerFrame);
    }

    int GetNextHistoryIndex() NN_NOEXCEPT
    {
        if (m_HistoryCountMax <= 0)
        {
            return -1;
        }

        const auto index = m_HistoryIndex;
        m_HistoryIndex = (m_HistoryIndex + 1) % m_HistoryCountMax;
        return index;
    }

    uint32_t GetFrameIndex() NN_NOEXCEPT
    {
        auto rt = m_FrameIndex;
        if (m_FrameIndex >= std::numeric_limits<uint32_t>::max())
        {
            m_FrameIndex = 0;
        }
        else
        {
            ++m_FrameIndex;
        }
        return rt;
    }

    void GetHistoryFrameAddress(PerformanceFrameAddresses* block, int index) NN_NOEXCEPT
    {
        if (m_HistoryCountMax > 0)
        {
            SetFrameAddress(block, m_CircularBase, index);
        }
        else
        {
            block->header = nullptr;
            block->entry = nullptr;
            block->detail = nullptr;
        }
    }

private:
    int8_t* m_DspBuffer = nullptr; // buffer base address which allocated from AudioRenderer's workBuffer
    DspAddr m_DspMappedBuffer = 0; // ADSP mapped buffer base

    // ADSP access frame
    uint32_t m_FrameIndex = 0u;
    PerformanceFrameAddresses m_CurrentFrame = {}; // DSP alway access here.
    int32_t m_EntryOffset = 0; // EntryType offset for generating each commands.
    int32_t m_DetailOffset = 0; // EntryType offset for generating each commands.

    // User accessible buffers.
    int8_t* m_CircularBase = nullptr; // head address of ring buffers
    PerformanceFrameAddresses m_HistoryFrame = {}; // circular base is next to DSP access base.
    int32_t m_HistoryIndex = 0;
    int32_t m_HistoryReadIndex = 0;
    int32_t m_HistoryCountMax = 0;

    // each Performance frame settings.
    int32_t m_EntryCountPerFrame = 0;
    int32_t m_DetailCountPerFrame = 0;
    size_t m_OneFrameSize = 0u; // Frame size for this rendering config.

    bool m_Initialized = false;

    NodeId m_DetailTarget = 0;
    PerformanceMetricsDataFormat m_DataFormatVersion = PerformanceMetricsDataFormat::Invalid;
};

using PerformanceManagerImplVersion1 = PerformanceManagerImpl<PerformanceManagerImplVersion1Tag, common::PerformanceFrameHeaderVersion1, common::PerformanceEntryVersion1, common::PerformanceDetailVersion1, PerformanceSysDetail>;
using PerformanceManagerImplVersion2 = PerformanceManagerImpl<PerformanceManagerImplVersion2Tag, common::PerformanceFrameHeaderVersion2, common::PerformanceEntryVersion2, common::PerformanceDetailVersion2, PerformanceSysDetail>;

class PerformanceManager
{
    NN_DISALLOW_COPY( PerformanceManager );
    NN_DISALLOW_MOVE( PerformanceManager );

public:
    PerformanceManager() NN_NOEXCEPT = default;
    ~PerformanceManager() NN_NOEXCEPT = default;
    static size_t GetRequiredBufferSizeForPerformanceMetricsPerFrame(const BehaviorInfo& behavior, const detail::AudioRendererParameterInternal& parameter) NN_NOEXCEPT;
    void Initialize(void* buffer, size_t bufferSize, const detail::AudioRendererParameterInternal& parameter, const BehaviorInfo& behavior, const server::MemoryPoolInfo& servicePool) NN_NOEXCEPT;
    bool IsInitialized() NN_NOEXCEPT;
    bool GetNextEntry(PerformanceEntryAddresses* block, PerformanceEntryType target, NodeId targetId = 0) NN_NOEXCEPT;
    bool GetNextEntry(PerformanceEntryAddresses* block, PerformanceDetailType target, PerformanceEntryType parentType, NodeId parentId) NN_NOEXCEPT;
    bool GetNextEntry(PerformanceEntryAddresses* block, uint32_t** ppEstimatedProcessingTime, PerformanceSysDetailType target, NodeId parentId) NN_NOEXCEPT;
    void TapFrame(bool isRenderingTimeLimitExceeded, int voiceDropCount, int64_t startTime) NN_NOEXCEPT;
    size_t CopyHistories(void* buffer, size_t bufferSize) NN_NOEXCEPT;
    void SetDetailTarget(NodeId targetId) NN_NOEXCEPT;
    bool IsDetailTarget(NodeId id) const NN_NOEXCEPT;

private:
    IPerformanceManagerImpl* CreateImpl(PerformanceMetricsDataFormat format) NN_NOEXCEPT;

private:
    IPerformanceManagerImpl* m_Impl = nullptr;
    using BiggestImplType = common::BiggestType<PerformanceManagerImplVersion1, PerformanceManagerImplVersion2>::Result;
    using BiggestAlignImplType = common::BiggestAlignType<PerformanceManagerImplVersion1, PerformanceManagerImplVersion2>::Result;
    uint8_t m_WorkBuffer[sizeof(BiggestImplType) + NN_ALIGNOF(BiggestAlignImplType) - 1];
};

}}}  // namespace nn::audio::server

