﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <stdint.h>
#include <nn/nn_SdkAssert.h>
#include <nne/audio/audio_GlobalMix.h>
#include <nn/audio/audio_SampleFormat.h>
#include "audio_Resampler.private.h"

namespace nn {
namespace audio {

struct MixerFormat
{
    int32_t sampleRate;
    int32_t channelCount;
    SampleFormat sampleFormat;
};

class Mixer
{
    enum MixerDepopState
    {
        MixerDepopState_Normal,
        MixerDepopState_Depopping,
        MixerDepopState_NeedRamp,
        MixerDepopState_Ramping,
    };

    struct UserBuffer
    {
        UserBuffer()
            : buffer(nullptr)
            , size(0)
            , writePosition(0)
            {}
        UserBuffer(void* _buffer, size_t _size)
            : buffer(_buffer)
            , size(_size)
            , writePosition(0)
        {}
        void* buffer;
        size_t size;
        size_t writePosition;
    };

public:
    Mixer();

    static bool IsInFormatSupported(const MixerFormat& inFormat) NN_NOEXCEPT;
    static bool IsOutFormatSupported(const MixerFormat& outFormat) NN_NOEXCEPT;
    void ResetCurrentBuffer() NN_NOEXCEPT;

    bool SetFormat(const MixerFormat& inFormat, const MixerFormat& outFormat) NN_NOEXCEPT;
    void SetFrameRate(int FramesPerSecond) NN_NOEXCEPT;

    void Depop() NN_NOEXCEPT;
    void Mix(const void* inBuffer, int32_t numInBytes, float volume) NN_NOEXCEPT;
    int MixToOutBuffer(void* pOutBuffer, void* pInBuffer, int numOutBytes, int numInBytes, float volume) NN_NOEXCEPT;

    bool AppendBuffer(void* buffer, size_t bufferSize) NN_NOEXCEPT;
    void* GetBufferAddress() NN_NOEXCEPT;
    void ClearBuffers() NN_NOEXCEPT;

    void DebugReleaseBuffer() NN_NOEXCEPT;
private:
    void MixMonoToStereo(const void* inBuffer, void* outputBuffer, int32_t numSamples) NN_NOEXCEPT;
    void MixStereoToStereo(const void* inBuffer, void* outputBuffer, int32_t numSamples) NN_NOEXCEPT;
    void MixSurroundToStereo(const void* inBuffer, void* outputBuffer, int32_t numSamples) NN_NOEXCEPT;

    void ReleaseBuffer();
    void* GetUserBufferAddress();
    int32_t GetUserBufferSamplesRemaining();
    void UserBufferInfoUpdatePosition(int32_t samples);

    static const int UserBufferMaxCount = nne::audio::gmix::NumAudioInBuffers;
    static const int MaxChannelCount = 2;
    static const int MaxOutputSampleRate = 48000;
    static const int MaxInputSampleRate = 48000;
    static const int MaxOutputFrameDurationMs = 5;
    static const int MaxOutputSampleCountPerFrame = MaxOutputSampleRate / (1000 / MaxOutputFrameDurationMs);
    static const int SrcPaddingSamples = 2;
    static const int SrcMaxSampleCount = ((MaxOutputSampleCountPerFrame + SrcPaddingSamples) * MaxChannelCount);

    UserBuffer m_UserBuffer[UserBufferMaxCount];
    nn::audio::ResamplerState m_ResampleState;
    MixerFormat m_InFormat;
    MixerFormat m_OutFormat;
    MixerDepopState m_DepopState;
    int32_t m_DepopDelta[MaxChannelCount];
    int32_t m_LastGoodSample[MaxChannelCount];
    int32_t m_FrameRate;
    int32_t m_NumAppendBuffers;
    int32_t m_MaxInputBytes;
    bool m_CanAppendBuffers;
    bool    m_IsFormatValid;
};

}  // namespace audio
}  // namespace nn
