﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "audio_ICommandBuffer.h"
#include "../common/audio_Command.h" // CommandHeader
#include "audio_PerformanceMetricsManager.h"
#include "audio_ServiceVoiceInfo.h"
#include "audio_ServiceMemoryPoolInfo.h"
#include "audio_ServiceMixInfo.h"
#include "audio_ServiceSinkInfo.h"
#include "audio_ICommandProcessingTimeEstimator.h"

namespace nn { namespace audio { namespace server {

class CommandBuffer : public ICommandBuffer
{
    NN_DISALLOW_COPY( CommandBuffer );
    NN_DISALLOW_MOVE( CommandBuffer );

public:
    struct CommandBufferState
    {
        size_t offset;
        int commandCount;
        uint32_t estimatedProcessingTime;
    };

public :
    CommandBuffer(void* commandBuffer,  size_t bufferSize, size_t initialOffset, const server::MemoryPoolInfo& servicePool, ICommandProcessingTimeEstimator* pEstimator) NN_NOEXCEPT;
    void* GetBuffer() const NN_NOEXCEPT;
    size_t GetCommandTotalSize() const NN_NOEXCEPT;
    int GetCount() const NN_NOEXCEPT;
    uint32_t GetEstimatedProcessingTime() const NN_NOEXCEPT;
    CommandBufferState GetState() const NN_NOEXCEPT;
    void SetState(const CommandBufferState& state) NN_NOEXCEPT;

    virtual PcmInt16DataSourceCommand* GeneratePcmInt16DataSourceCommand(const server::VoiceInfo* pVoice, VoiceState* state, int mixBufferCount, int32_t channel, NodeId nodeId) NN_NOEXCEPT;
    virtual AdpcmDataSourceCommand* GenerateAdpcmDataSourceCommand(const server::VoiceInfo* pVoice, VoiceState* state, int mixBufferCount, NodeId nodeId) NN_NOEXCEPT;
    virtual OpusDataSourceCommand* GenerateOpusDataSourceCommand(const server::VoiceInfo* pVoice, VoiceState* state, int mixBufferCount, NodeId nodeId) NN_NOEXCEPT;
    virtual AuxCommand* GenerateAuxCommand(int mixBufferOffset, int8_t input, int8_t output, const AuxBufferAddresses* pAddresses, bool enabled, int32_t sampleCount, DspAddr sendBuffer, DspAddr returnBuffer, int updateCount, int offset, NodeId nodeId) NN_NOEXCEPT;
    virtual DelayCommand* GenerateDelayEffectCommand(int mixBufferOffset, const DelayParameter* pParameter, void* pState, bool enabled, DspAddr workBuffer, NodeId nodeId) NN_NOEXCEPT;
    virtual ReverbCommand* GenerateReverbEffectCommand(int mixBufferOffset, const ReverbParameter* pParameter, void* pState, bool enabled, bool longSizePreDelaySupported, DspAddr workBuffer, NodeId nodeId) NN_NOEXCEPT;
    virtual I3dl2ReverbCommand* GenerateI3dl2ReverbEffectCommand(int mixBufferOffset, const I3dl2ReverbParameter* pParameter, void* pState, bool enabled, DspAddr workBuffer, NodeId nodeId) NN_NOEXCEPT;
    virtual CopyMixBufferCommand* GenerateCopyMixBufferCommand(int inputBufferIndex, int outputBufferIndex, NodeId nodeId) NN_NOEXCEPT;
    virtual BiquadFilterCommand* GenerateBiquadFilterCommand(int mixBufferOffset, const BiquadFilterParameter* pFilterParameter, BiquadFilterState* pState, int8_t input, int8_t output, bool needInitialization, NodeId nodeId) NN_NOEXCEPT;
    virtual MixCommand* GenerateMixCommand(int inputBufferIndex, int outputBufferIndex, float mixVolume, NodeId nodeId) NN_NOEXCEPT;
    virtual MixRampCommand* GenerateMixRampCommand(int inputBufferIndex, int outputBufferIndex, float mixVolume0, float mixVolume1, DspAddr voiceStateDspAddr, NodeId nodeId) NN_NOEXCEPT;
    virtual MixRampGroupedCommand* GenerateMixRampGroupedCommand(int mixBufferCount, int inputBufferIndex, int outputBufferIndexStart, const float* pMixVolume, const float* pMixVolumePrev, DspAddr lastSampleAddress, NodeId nodeId) NN_NOEXCEPT;
    virtual DepopPrepareCommand* GenerateDepopPrepareCommand(VoiceState* state, int32_t* depopBuffer, int mixBufferCount, int mixBufferOffset, NodeId nodeId, bool commandEnabled) NN_NOEXCEPT;
    virtual DepopForMixBuffersCommand* GenerateDepopForMixBuffersCommand(int32_t* depopBuffer, int mixBufferOffset, int mixBufferCount, NodeId nodeId, int32_t sampleRate) NN_NOEXCEPT;
    virtual ClearMixBufferCommand* GenerateClearMixBufferCommand(NodeId nodeId) NN_NOEXCEPT;
    virtual VolumeCommand* GenerateVolumeCommand(float volume, int mixBufferCount, NodeId nodeId) NN_NOEXCEPT;
    virtual VolumeRampCommand* GenerateVolumeRampCommand(float volume0, float volume1, int mixBufferCount, NodeId nodeId) NN_NOEXCEPT;
    virtual DeviceSinkCommand* GenerateDeviceSinkCommand(int mixBufferOffset, const server::SinkInfoBase* pSinkInfo, uint32_t sessionId, DspAddr mixBufferAddr, NodeId nodeId) NN_NOEXCEPT;
    virtual CircularBufferSinkCommand* GenerateCircularBufferSinkCommand(int mixBufferOffset, const server::SinkInfoBase* pSinkInfo, NodeId nodeId) NN_NOEXCEPT;
    virtual DownMix6chTo2chCommand* GenerateDownMixCommand(int mixBufferOffset, const int8_t* input, const int8_t* output, const int32_t* downMixParam, NodeId nodeId) NN_NOEXCEPT;
    virtual UpsampleCommand* GenerateUpsampleCommand(int mixBufferOffset, UpsamplerInfo* pInfo, int inputCount, const int8_t* input, int32_t buffeCount, int32_t sampleCount, int32_t sampleRate, NodeId nodeId) NN_NOEXCEPT;
    virtual PerformanceCommand* GeneratePerformanceCommand(const PerformanceEntryAddresses* entry, const PerformanceCommandType type, NodeId nodeId) NN_NOEXCEPT;

private:
    uintptr_t m_BufferBase;
    size_t m_BufferSize;
    size_t m_Offset;
    int m_Count;
    uint32_t m_EstimatedProcessingTime;
    const server::MemoryPoolInfo& m_ServicePool;
    ICommandProcessingTimeEstimator* m_pEstimator;

private:
    CommandBuffer() NN_NOEXCEPT;
    void* GetCurrentBufferBase() const NN_NOEXCEPT;
    template<typename T>
    void EstimateProcessTime(T* pCommand)
    {
        const auto estimatedProcessingTime = m_pEstimator->Estimate(*pCommand);
        pCommand->header.estimatedProcessingTime = estimatedProcessingTime;
        m_EstimatedProcessingTime += estimatedProcessingTime;
    };
    template<typename T>
    T* CreateCommand(NodeId nodeId) NN_NOEXCEPT
    {
        NN_SDK_ASSERT_LESS_EQUAL(m_Offset + sizeof(T), m_BufferSize);

        auto pCommand = reinterpret_cast<T*>(GetCurrentBufferBase());
        auto pCommandHeader = reinterpret_cast<CommandHeader*>(pCommand);

        pCommandHeader->magic = CommandHeaderMagicNumber;
        pCommandHeader->size = sizeof(T);
        pCommandHeader->id = T::Id;
        pCommandHeader->nodeId = nodeId;
        pCommandHeader->isEnabled = true;

        ++m_Count;
        m_Offset += sizeof(T);

        return pCommand;
    };
};

}}}  // namespace nn::audio::server
