﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "../audio_DspCommon.h"
#include "../audio_Qf.h"
#include "../audio_DelayLine.h"
#include "../audio_EffectDelay.h"

namespace nn { namespace audio { namespace detail {

NN_FORCEINLINE void ApplyDelayEffect1ch(const DelayParameter* pParameter, DelayState* state, const int32_t *pInData1, int32_t *pOutData1, int sampleCount)
{
    uint32_t outSamples = sampleCount;
    uint32_t samp;

    qf sc; // Keep track of samples locally
    qf outSample;
    qf temps;
    qf s;
    qf t;

    qf gain = pParameter->_inGain;
    qf fb = pParameter->_feedbackGain; // 0.0 to 0.999
    qf lowPassFilterCoefficientB0 = state->_lowPassFilterCoefficientB0; // 0.905 to 0.05
    qf lowPassFilterCoefficientA1 = state->_lowPassFilterCoefficientA1; // 0.095 to 0.95
    qf *LowPassFilterHistory = (state->_pLowPassFilterHistory);
    qf outGain = pParameter->_outGain;
    qf dryGain = pParameter->_dryGain;

    for (samp = 0; (samp < outSamples); samp++)
    {
        // get input samples (as int32_t)
        sc = Q8ToQf(*pInData1++);

        // get output samples for each delay (as qf)
        outSample = DelayLineNextOut(&state->_delayLines[0]);

        // Calc input to each lpf and delay for channels 0 and 1
        s = MultiQfQf(outSample, fb);
        t = MultiQfQf(sc, gain);
        temps = AddQfQf(t, s);

        // low-pass filter
        s = MultiQfQf(LowPassFilterHistory[0], lowPassFilterCoefficientA1);
        t = MultiQfQf(temps, lowPassFilterCoefficientB0);
        LowPassFilterHistory[0] = t = AddQfQf(t, s);
        // tick delay lines
        DelayLineTick(&state->_delayLines[0], t);
        // Wet/Dry mix
        s = MultiQfQf(outSample, outGain);
        t = MultiQfQf(sc, dryGain);
        outSample = AddQfQf(t, s);

        *pOutData1++ = QfToQ8(outSample);
    }

    return;
}

NN_FORCEINLINE void ApplyDelayEffect2ch(const DelayParameter* pParameter, DelayState* state, const int32_t **ppInData, int32_t **ppOutData, int sampleCount)
{
    qf sc[nn::audio::DelayParameter::ChannelCountMax]; // Keep track of samples locally
    qf outSamp[nn::audio::DelayParameter::ChannelCountMax];
    qf temps[6];
    qf s;
    qf t;

    qf gain = pParameter->_inGain;
    qf feedbackCross = state->_feedbackCross;  // 0.0 to 0.5
    qf feedbackDirect = state->_feedbackDirect; // 0.0 to 0.999
    qf lowPassFilterCoefficientB0 = state->_lowPassFilterCoefficientB0; // 0.905 to 0.05
    qf lowPassFilterCoefficientA1 = state->_lowPassFilterCoefficientA1; // 0.095 to 0.95
    qf *LowPassFilterHistory = (state->_pLowPassFilterHistory);
    qf outGain = pParameter->_outGain;
    qf dryGain = pParameter->_dryGain;

    for (int samp = 0; samp < sampleCount; samp++)
    {
        // get input samples (as int32_t)
        sc[0] = Q8ToQf(ppInData[0][samp]);
        sc[1] = Q8ToQf(ppInData[1][samp]);

        // get output samples for each delay (as qf)
        outSamp[0] = DelayLineNextOut(&state->_delayLines[0]);
        outSamp[1] = DelayLineNextOut(&state->_delayLines[1]);

        // Calc input to each lpf and delay for channels 0 and 1
        s = MultiQfQf(outSamp[1], feedbackCross);
        t = MultiQfQf(outSamp[0], feedbackDirect);
        s = AddQfQf(t, s);
        t = MultiQfQf(sc[0], gain);
        temps[0] = AddQfQf(t, s);

        s = MultiQfQf(outSamp[0], feedbackCross);
        t = MultiQfQf(outSamp[1], feedbackDirect);
        s = AddQfQf(t, s);
        t = MultiQfQf(sc[1], gain);
        temps[1] = AddQfQf(t, s);

        // low-pass filter and input to delay lines channels 0 and 1
        for (auto i = 0; (i < pParameter->_numChannels); i++)
        {
            // low-pass filter
            s = MultiQfQf(LowPassFilterHistory[i], lowPassFilterCoefficientA1);
            t = MultiQfQf(temps[i], lowPassFilterCoefficientB0);
            LowPassFilterHistory[i] = t = AddQfQf(t, s);
            // tick delay lines
            DelayLineTick(&state->_delayLines[i], t);
            // Wet/Dry mix
            s = MultiQfQf(outSamp[i], outGain);
            t = MultiQfQf(sc[i], dryGain);
            outSamp[i] = AddQfQf(t, s);
        }

        ppOutData[0][samp] = QfToQ8(outSamp[0]);
        ppOutData[1][samp] = QfToQ8(outSamp[1]);
    }
}

NN_FORCEINLINE void ApplyDelayEffect4ch(const DelayParameter* pParameter, DelayState* state, const int32_t **ppInData, int32_t **ppOutData, int sampleCount)
{
    qf sc[nn::audio::DelayParameter::ChannelCountMax]; // Keep track of samples locally
    qf outSamp[nn::audio::DelayParameter::ChannelCountMax];
    qf temps[6];
    qf s;
    qf t;
    qf gain = pParameter->_inGain;
    qf feedbackCross = state->_feedbackCross;  // 0.0 to 0.5
    qf feedbackDirect = state->_feedbackDirect; // 0.0 to 0.999
    qf lowPassFilterCoefficientB0 = state->_lowPassFilterCoefficientB0; // 0.905 to 0.05
    qf lowPassFilterCoefficientA1 = state->_lowPassFilterCoefficientA1; // 0.095 to 0.95
    qf* LowPassFilterHistory = state->_pLowPassFilterHistory;
    qf OutGain = pParameter->_outGain;
    qf dryGain = pParameter->_dryGain;

    for (int samp = 0; samp < sampleCount; samp++)
    {
        // get input samples (as int32_t)

        sc[0] = Q8ToQf(ppInData[0][samp]);
        sc[1] = Q8ToQf(ppInData[1][samp]);
        sc[2] = Q8ToQf(ppInData[2][samp]);
        sc[3] = Q8ToQf(ppInData[3][samp]);

        // get output samples for each delay (as qf)
        outSamp[0] = DelayLineNextOut(&state->_delayLines[0]);
        outSamp[1] = DelayLineNextOut(&state->_delayLines[1]);
        outSamp[2] = DelayLineNextOut(&state->_delayLines[2]);
        outSamp[3] = DelayLineNextOut(&state->_delayLines[3]);

        // Calc input to each lpf and delay for channels 0 and 3
        t = AddQfQf(outSamp[2], outSamp[1]);
        s = MultiQfQf(t, feedbackCross);
        t = MultiQfQf(outSamp[0], feedbackDirect);
        s = AddQfQf(t, s);
        t = MultiQfQf(sc[0], gain);
        temps[0] = AddQfQf(t, s);

        t = AddQfQf(outSamp[0], outSamp[3]);
        s = MultiQfQf(t, feedbackCross);
        t = MultiQfQf(outSamp[1], feedbackDirect);
        s = AddQfQf(t, s);
        t = MultiQfQf(sc[1], gain);
        temps[1] = AddQfQf(t, s);

        t = AddQfQf(outSamp[3], outSamp[0]);
        s = MultiQfQf(t, feedbackCross);
        t = MultiQfQf(outSamp[2], feedbackDirect);
        s = AddQfQf(t, s);
        t = MultiQfQf(sc[2], gain);
        temps[2] = AddQfQf(t, s);

        t = AddQfQf(outSamp[1], outSamp[2]);
        s = MultiQfQf(t, feedbackCross);
        t = MultiQfQf(outSamp[3], feedbackDirect);
        s = AddQfQf(t, s);
        t = MultiQfQf(sc[3], gain);
        temps[3] = AddQfQf(t, s);

        // low-pass filter and input to delay lines channels 0 and 3
        for (auto i = 0; (i < pParameter->_numChannels); i++)
        {
            // low-pass filter
            s = MultiQfQf(LowPassFilterHistory[i], lowPassFilterCoefficientA1);
            t = MultiQfQf(temps[i], lowPassFilterCoefficientB0);
            LowPassFilterHistory[i] = t = AddQfQf(t, s);
            // tick delay lines
            DelayLineTick(&state->_delayLines[i], t);
            // Wet/Dry mix
            s = MultiQfQf(outSamp[i], OutGain);
            t = MultiQfQf(sc[i], dryGain);
            outSamp[i] = AddQfQf(t, s);
        }

        ppOutData[0][samp] = QfToQ8(outSamp[0]);
        ppOutData[1][samp] = QfToQ8(outSamp[1]);
        ppOutData[2][samp] = QfToQ8(outSamp[2]);
        ppOutData[3][samp] = QfToQ8(outSamp[3]);
    }
}

void ApplyDelayEffect6ch(const DelayParameter* pParameter, DelayState* state, const int32_t **ppInData, int32_t **ppOutData, int sampleCount)
{
    qf sc[nn::audio::DelayParameter::ChannelCountMax]; // Keep track of samples locally
    qf outSamp[nn::audio::DelayParameter::ChannelCountMax];
    qf feedback = pParameter->_feedbackGain; // 0.0 to 0.999
    qf temps[6];
    qf s;
    qf t;
    qf gain = pParameter->_inGain;
    qf feedbackCross = state->_feedbackCross;  // 0.0 to 0.5
    qf feedbackDirect = state->_feedbackDirect; // 0.0 to 0.999
    qf lowPassFilterCoefficientB0 = state->_lowPassFilterCoefficientB0; // 0.905 to 0.05
    qf lowPassFilterCoefficientA1 = state->_lowPassFilterCoefficientA1; // 0.095 to 0.95
    qf* LowPassFilterHistory = state->_pLowPassFilterHistory;
    qf OutGain = pParameter->_outGain;
    qf dryGain = pParameter->_dryGain;

    for (int samp = 0; samp < sampleCount; samp++)
    {
        // get input samples (as int32_t)

        sc[0] = Q8ToQf(ppInData[0][samp]);
        sc[1] = Q8ToQf(ppInData[1][samp]);
        sc[2] = Q8ToQf(ppInData[2][samp]);
        sc[3] = Q8ToQf(ppInData[3][samp]);
        sc[4] = Q8ToQf(ppInData[4][samp]);
        sc[5] = Q8ToQf(ppInData[5][samp]);

        // get output samples for each delay (as qf)
        outSamp[0] = DelayLineNextOut(&state->_delayLines[0]);
        outSamp[1] = DelayLineNextOut(&state->_delayLines[1]);
        outSamp[2] = DelayLineNextOut(&state->_delayLines[2]);
        outSamp[3] = DelayLineNextOut(&state->_delayLines[3]);
        outSamp[4] = DelayLineNextOut(&state->_delayLines[4]);
        outSamp[5] = DelayLineNextOut(&state->_delayLines[5]);

        // Calc input to each lpf and delay for channels 0 and 3
        t = AddQfQf(outSamp[2], outSamp[4]);
        s = MultiQfQf(t, feedbackCross);
        t = MultiQfQf(outSamp[0], feedbackDirect);
        s = AddQfQf(t, s);
        t = MultiQfQf(sc[0], gain);
        temps[0] = AddQfQf(t, s);

        t = AddQfQf(outSamp[4], outSamp[3]);
        s = MultiQfQf(t, feedbackCross);
        t = MultiQfQf(outSamp[1], feedbackDirect);
        s = AddQfQf(t, s);
        t = MultiQfQf(sc[1], gain);
        temps[1] = AddQfQf(t, s);

        t = AddQfQf(outSamp[3], outSamp[0]);
        s = MultiQfQf(t, feedbackCross);
        t = MultiQfQf(outSamp[2], feedbackDirect);
        s = AddQfQf(t, s);
        t = MultiQfQf(sc[2], gain);
        temps[2] = AddQfQf(t, s);

        t = AddQfQf(outSamp[1], outSamp[2]);
        s = MultiQfQf(t, feedbackCross);
        t = MultiQfQf(outSamp[3], feedbackDirect);
        s = AddQfQf(t, s);
        t = MultiQfQf(sc[3], gain);
        temps[3] = AddQfQf(t, s);

        t = AddQfQf(outSamp[0], outSamp[1]);
        s = MultiQfQf(t, feedbackCross);
        t = MultiQfQf(outSamp[4], feedbackDirect);
        s = AddQfQf(t, s);
        t = MultiQfQf(sc[4], gain);
        temps[4] = AddQfQf(t, s);

        s = MultiQfQf(outSamp[5], feedback);
        t = MultiQfQf(sc[5], gain);
        temps[5] = AddQfQf(t, s);

        // low-pass filter and input to delay lines channels 0 and 3

        for (auto i = 0; (i < pParameter->_numChannels); i++)
        {
            // low-pass filter
            s = MultiQfQf(LowPassFilterHistory[i], lowPassFilterCoefficientA1);
            t = MultiQfQf(temps[i], lowPassFilterCoefficientB0);
            LowPassFilterHistory[i] = t = AddQfQf(t, s);
            // tick delay lines
            DelayLineTick(&state->_delayLines[i], t);
            // Wet/Dry mix
            s = MultiQfQf(outSamp[i], OutGain);
            t = MultiQfQf(sc[i], dryGain);
            outSamp[i] = AddQfQf(t, s);
        }

        ppOutData[0][samp] = QfToQ8(outSamp[0]);
        ppOutData[1][samp] = QfToQ8(outSamp[1]);
        ppOutData[2][samp] = QfToQ8(outSamp[2]);
        ppOutData[3][samp] = QfToQ8(outSamp[3]);
        ppOutData[4][samp] = QfToQ8(outSamp[4]);
        ppOutData[5][samp] = QfToQ8(outSamp[5]);
    }
}

NN_FORCEINLINE void ApplyDelayEffectBypass(const int32_t **ppInData, int32_t **ppOutData, int channelCount, int sampleCount)
{
    for(int i = 0; i < channelCount; ++i)
    {
        if (ppOutData[i] != ppInData[i])
        {
            memcpy(ppOutData[i], ppInData[i], sampleCount* sizeof(int32_t));
        }
    }
}

}}} // namespace nn::audio::detail
