﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/*  \file  audio_Upsample32KTo48K.h
 *  \brief interface wrapper header file over Bandlimited interpolation sample rate converter
 */
#pragma once

#include <stdint.h>

#include <nn/audio/audio_Common.h>
#include "../audio_AddrTypes.h" // CpuAddr
#include "../common/audio_Util.h"

namespace nn {
namespace audio {

typedef enum SRC_BL_CONSTANTS
{
    NN_AUDIO_MAX_S16 = (32767),
    NN_AUDIO_MIN_S16 = (-32768),
    NN_AUDIO_NQRATIO  = 16,

    NN_AUDIO_SRC_BL_RATIO_32_48 = ((uint32_t)98304),
    NN_AUDIO_SRC_BL_RATIO_16_48 = ((uint32_t)196608),
    NN_AUDIO_SRC_BL_RATIO_8_48  = ((uint32_t)393216),

/*! \def Block SRC Related Constants
 */
    NN_AUDIO_SRC_BL_NZ  = 10,    /*!< number of zero crossings in the filter table, integer only */
    NN_AUDIO_SRC_BL_X_CB_SIZE = (NN_AUDIO_SRC_BL_NZ * 2),
    NN_AUDIO_SRC_BL_CB_MID   = ((NN_AUDIO_SRC_BL_X_CB_SIZE) / 2),
} SRC_BL_CONSTANTS;

/*! \struct SRC_BANDLIMITED
 *  \brief  Define format of block SRC parameters
 *
 */
typedef struct UpsamplerState
{
    int32_t     ratioRho;           /*!< Rho = FsNew/FsOld, Q15.16 format, defined by NN_AUDIO_NQRATIO */
    int32_t     input_samples[NN_AUDIO_SRC_BL_X_CB_SIZE];    /*!< Input Sample buffer for block SRC */
    uint16_t    nz_input;           /*!< how many steps through the filter will we make?  (for one side of the h filter)
                                            or how many input samples are needed to calculate an output sample (2 * nz_input + 1) */
    uint16_t    input_current;      /*!< array index to the current sample of the inputs, in input sample buffer, Q23.8 format */
    uint16_t    input_write;        /*!< array index to the next sample writing location, in input sample buffer, Q23.8 format */
    uint8_t    input_cb_lowerBound;
    uint8_t    input_cb_upperBound;
    int8_t     isInitialized;
    uint8_t    outputIndex;        /* change between 0,1,2*/

} UpsamplerState;

struct UpsamplerInfo
{
    static const int InputCountMax = 6;
    UpsamplerState srcbl[InputCountMax];
    CpuAddr pUpsamplerManager;
    CpuAddr pOutBuffers;
    int32_t sampleCount;
    int32_t inputCount;
    bool isInUse;
    int8_t input[InputCountMax];
    int8_t _padding[96 - nn::audio::BufferAlignSize];

    UpsamplerInfo() NN_NOEXCEPT;
};
#if defined(NN_BUILD_CONFIG_SPEC_NX) && !defined(NN_BUILD_CONFIG_OS_WIN)
NN_AUDIO_INFOTYPE_CHECK(UpsamplerInfo, 704 - nn::audio::BufferAlignSize);
NN_AUDIO_INFOTYPE_CHECK(UpsamplerState, 96);
NN_STATIC_ASSERT(sizeof(UpsamplerState) % common::AdspCacheLineSize == 0);
#endif

/* Do block SRC for a frame */
void SrcProcessFrame(int32_t* pOutData, int32_t* pInData, uint32_t outSampleCount, uint32_t inSampleCount, UpsamplerState* src_bl_config);


}  // namespace audio
}  // namespace nn
