﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audio_DspCommon.h"
#include "audio_EffectReverb.h"
#include "audio_DelayLine.h"
#include "audio_DspUtility.h"

// Note: NEON optimized version has not been implemented yet.
#if (defined(__ARM_NEON__) || defined(__ARM_NEON)) && 0
#include "detail/audio_EffectReverb.neon.h"
#else  // defined(__ARM_NEON__) || defined(__ARM_NEON)
#include "detail/audio_EffectReverb.generic.h"
#endif  // defined(__ARM_NEON__) || defined(__ARM_NEON)

namespace nn {
namespace audio {

namespace {

// eliminating these lookups by unrolling their respective loops
// static const int32_t __TapOutAssignments2ch[ReverbParameter::EarlyReflectionTaps] = { 0, 0, 1, 1, 0, 1, 0, 0, 1, 1 };
// static const int32_t __TapOutAssignments4ch[ReverbParameter::EarlyReflectionTaps] = { 0, 0, 1, 1, 0, 1, 2, 2, 3, 3 };
// static const int32_t __TapOutAssignments6ch[ReverbParameter::EarlyReflectionTaps] = { 0, 0, 1, 1, 4, 4, 2, 2, 3, 3 };

void ClearDelayLines(const ReverbParameter* reverb, ReverbState* state)
{
    DelayLineClear(&state->_earlyDelay);
    if (reverb->_numChannels == 6) DelayLineClear(&state->_centerDelay);

    for (auto i = 0; i < ReverbParameter::FeedbackDelayNetworkCount; i++)
    {
        DelayLineClear(&state->_fdnDelay[i]);
        DelayLineClear(&state->_allPass[i]);
    }
}

void SetAllFdnParameters(const ReverbParameter* reverb, ReverbState* state)
{
    // coef for all-passes as a function of coloration
    qf allPassCoef = MultiQfQf(QF_CONST(0.6f), QF_CONST(1.0f) - reverb->_coloration);

    // cosWHF = cos(2.0*3.14159*LateReverbHighFrequency/reverb->_sampleRate), note _sampleRate(fs) is in kHz.
    // since CosQf and SinQf range is [0..255], convert radians to mod_256:  cos(256 * LateReverbHighFreq / fs) = cos(1280/fs)
    static const qf cosWHF = CosQf(DivQfQf(QF_CONST(256 * ReverbParameter::LateReverbHighFrequency / 1000), reverb->_sampleRate));

    // fdn and ap delay lengths and fdn feedback coeffs
    for (auto i = 0; i < ReverbParameter::FeedbackDelayNetworkCount; i++)
    {
        DelayLineSetDelay(&state->_fdnDelay[i], DelayLineCalculateDelaySamples(reverb->_sampleRate, FdnDelayLen[reverb->_lateMode][i]));
        DelayLineSetDelay(&state->_allPass[i], DelayLineCalculateDelaySamples(reverb->_sampleRate, AllPassDelayLen[reverb->_lateMode][i]));

        // Set FDN Coeffs
        {
            uint32_t fdnNum = i;
            qf dcGain /*[dB]*/ = DivQfQf(S32ToQf(-3 * (int32_t)(DelayLineGetDelay(&state->_fdnDelay[fdnNum]) + AllPassGetDelay(&state->_allPass[fdnNum]))),
                                         MultiQfQf(reverb->_decayTime, reverb->_sampleRate));
            qf hfRatio = reverb->_highFrequencyDecayRatio;
            qf a1, b0;
            if (nn::audio::ReverbParameter::HighFrequencyRatioMax <= hfRatio)
            {
                a1 = QF_CONST(0.0f);
                b0 = QF_CONST(1.0f);
            }
            else
            {
                //hf_gain_db = 2.0 * g_dbk/1000.0 * ((1./hf_ratio) - 1.); // HF gain in dB
                auto hf_gain_db = MultiQfQf(DivQfQf(dcGain, QF_CONST(10.0f)),
                    DivQfQf(MultiQfS32(DivQfQf(QF_CONST(1.0f), hfRatio) - QF_CONST(1.0f), 2),
                    QF_CONST(100.0f)));
                auto hf_gain_lin = Pow10Qf(hf_gain_db); // HF gain linear

                // calculate LPF coeff using quadratic equation (a,b,c are coefs for quadratic, c=a)
                auto a = QF_CONST(1.0f) - hf_gain_lin;
                auto b = QF_CONST(2.0f) - MultiQfQf(MultiQfS32(hf_gain_lin, 2), cosWHF);
                a1 = DivQfQf(b - SqrtQf(MultiQfQf(b, b) - MultiQfS32(MultiQfQf(a, a), 4)), MultiQfS32(a, 2));// fb coef for lpf
                b0 = QF_CONST(1.0f) - a1;                    // ff coef for lpf (DC normalized)
            }

            qf dcGainLiner = Pow10Qf(DivQfQf(dcGain, QF_CONST(1000.0f)));
            // fold the feedback coef, the lpf ff coef, and the matrix normalization into one
            state->_fdnFeedback[fdnNum] = MultiQfQf(MultiQfQf(dcGainLiner, b0), QF_CONST(0.7071f));
            state->_fdnLowPassCoef[fdnNum] = a1;
        }

        AllPassSetCoef(&state->_allPass[i], allPassCoef);
        state->_fdnLowPassHistory[i] = QF_CONST(0.0f);
    }

}

} // namespace

void InitializeReverbEffect(const ReverbParameter *reverb,  ReverbState* state, void* workBuffer, bool longSizePreDelaySupported)
{
    uint32_t i;
    int32_t tempi;
    qf* buf = static_cast<qf*>(workBuffer);

    memset(state, 0, sizeof(ReverbState));

    // late reverb stuff
    for (i = 0; i < ReverbParameter::FeedbackDelayNetworkCount; i++)
    {
        tempi = DelayLineCalculateDelaySamples(reverb->_sampleRate, FdnDelayLen[ReverbParameter::LateMode_MaximumDelay][i]);
        DelayLineInit(&state->_fdnDelay[i], tempi, buf);
        buf += tempi;

        tempi = DelayLineCalculateDelaySamples(reverb->_sampleRate, AllPassDelayLen[ReverbParameter::LateMode_MaximumDelay][i]);
        AllPassInit(&state->_allPass[i], tempi, QF_CONST(0.0f), buf);
        buf += tempi;
    }

    auto delayMax = longSizePreDelaySupported ?
        ReverbParameter::EarlyReflectionDelayMax_350ms :
        ReverbParameter::EarlyReflectionDelayMax_150ms;
    // allocate early reflection delay line.  Length based on static numbers.
    tempi = DelayLineCalculateDelaySamples(reverb->_sampleRate, delayMax);
    DelayLineInit(&state->_earlyDelay, tempi, buf);
    buf += tempi;

    tempi = DelayLineCalculateDelaySamples(reverb->_sampleRate, ReverbParameter::CenterDelayMax);
    DelayLineInit(&state->_centerDelay, tempi, buf);
    buf += tempi;

    UpdateReverbEffectParameter(reverb, state);

    ClearDelayLines(reverb, state);
}

/* @TODO: the parameter calculation will be done by Audio Service, so can actually be done in floating point arithmetic */
void UpdateReverbEffectParameter(const ReverbParameter* reverb, ReverbState* state)
{
    int32_t maxDelaySampleCount = DelayLineGetMaxDelay(&state->_earlyDelay);

    // early reflections
    for (auto i = 0; (i < ReverbParameter::EarlyReflectionTaps); i++)
    {
        int32_t tempDelaySampleCount = Min(DelayLineCalculateDelaySamples(reverb->_sampleRate, reverb->_predelayTime + ErTaps[reverb->_earlyMode][i]),
                                            maxDelaySampleCount);
        state->_earlyTap[i] = tempDelaySampleCount + 1;
        state->_earlyCoef[i] = MultiQfQf(ErCoefs2[reverb->_earlyMode][i], reverb->_earlyGain);
    }

    // adjust those taps that are panned to multiple channels
    if (2 == reverb->_numChannels)
    {
        state->_earlyCoef[4] = MultiQfQf(state->_earlyCoef[4], QF_CONST(0.5f));
        state->_earlyCoef[5] = MultiQfQf(state->_earlyCoef[5], QF_CONST(0.5f));
    }

    // tap to late reverb:
    state->_lateTap = Min(DelayLineCalculateDelaySamples(reverb->_sampleRate, reverb->_predelayTime + ErTaps[reverb->_earlyMode][ReverbParameter::EarlyReflectionCount]),
                                           maxDelaySampleCount);

    SetAllFdnParameters(reverb, state);
}

void ApplyReverbEffect(const ReverbParameter *reverb, ReverbState* state, bool enabled, const int32_t **ppInData, int32_t **ppOutData, const uint32_t sampleCount)
{
    NN_AUDIO_DSP_ASSERT((reverb->_numChannels == 1) || (reverb->_numChannels == 2) || (reverb->_numChannels == 4) || (reverb->_numChannels == 6)); // for klocwork inspection check.
    NN_AUDIO_DSP_ASSERT(reverb->_earlyMode < ReverbParameter::EarlyMode_Count); // for klocwork inspection check.
    NN_AUDIO_DSP_ASSERT(reverb->_lateMode < ReverbParameter::LateMode_Count); // for klocwork inspection check.

    const auto channelCount = reverb->_numChannels;

    if (enabled)
    {
        switch (channelCount)
        {
        case 1:
            detail::ApplyReverbEffect1ch(reverb, state, sampleCount, ppInData[0], ppOutData[0]);
            break;
        case 2:
            detail::ApplyReverbEffect2ch(reverb, state, sampleCount, ppInData[0], ppInData[1], ppOutData[0], ppOutData[1]);
            break;
        case 4:
            detail::ApplyReverbEffect4ch(reverb, state, sampleCount, ppInData[0], ppInData[1], ppInData[2], ppInData[3],
                ppOutData[0], ppOutData[1], ppOutData[2], ppOutData[3]);
            break;
        case 6:
            detail::ApplyReverbEffect6ch(reverb, state, sampleCount, ppInData, ppOutData);
            break;
        default:
            detail::ApplyReverbEffectBypass(ppInData, ppOutData, channelCount, sampleCount);
            break;
        }
    }
    else // bypass
    {
        detail::ApplyReverbEffectBypass(ppInData, ppOutData, channelCount, sampleCount);
    }
}

size_t ReverbGetRequiredDelayLineBufferSize(qf qfSampleRate, int channelCount)
{
    size_t samples = 0;

    for (auto i = 0; i < ReverbParameter::FeedbackDelayNetworkCount; i++)
    {
        samples += DelayLineCalculateDelaySamples(qfSampleRate, FdnDelayLen[ReverbParameter::LateMode_MaximumDelay][i]);
        samples += DelayLineCalculateDelaySamples(qfSampleRate, AllPassDelayLen[ReverbParameter::LateMode_MaximumDelay][i]);
    }
    samples += DelayLineCalculateDelaySamples(qfSampleRate, ReverbParameter::EarlyReflectionDelayMax);

    if (channelCount == 6)
    {
        samples += DelayLineCalculateDelaySamples(qfSampleRate, ReverbParameter::CenterDelayMax);
    }

    return samples * sizeof(qf);
}

} // namespace audio
} // namespace nn
