﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <nn/nn_SdkAssert.h>
#include "audio_DspExceptionNotifier.h"

namespace nn { namespace audio { namespace dsp {

DspExceptionNotifier::DspExceptionNotifier() NN_NOEXCEPT
    : m_Mutex(false)
{
    for(auto observerHolder : m_ObserverHolders)
    {
        observerHolder.observer = nullptr;
        observerHolder.args = nullptr;
    }
}

DspExceptionNotifier::~DspExceptionNotifier() NN_NOEXCEPT
{

}

DspExceptionNotifier& DspExceptionNotifier::GetInstance() NN_NOEXCEPT
{
    NN_FUNCTION_LOCAL_STATIC(DspExceptionNotifier, instance);
    return instance;
}

bool DspExceptionNotifier::AddObserverImpl(Observer observer, void* pArgs) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    for(auto observerHolder : m_ObserverHolders)
    {
        if(observerHolder.observer == nullptr)
        {
            observerHolder.observer = observer;
            observerHolder.args = pArgs;
            return true;
        }
    }

    return false;
}

bool DspExceptionNotifier::RemoveObserverImpl(Observer observer) NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    for(auto observerHolder : m_ObserverHolders)
    {
        if(observerHolder.observer == observer)
        {
            observerHolder.observer = nullptr;
            observerHolder.args = nullptr;
            return true;
        }
    }

    return false;
}

void DspExceptionNotifier::NotifyImpl() NN_NOEXCEPT
{
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    for(const auto& observerHolder : m_ObserverHolders)
    {
        if(observerHolder.observer != nullptr)
        {
            observerHolder.observer(observerHolder.args);
        }
    }
}

bool DspExceptionNotifier::AddObserver(Observer observer, void* pArgs) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(observer);
    return GetInstance().AddObserverImpl(observer, pArgs);
}

bool DspExceptionNotifier::RemoveObserver(Observer observer) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(observer);
    return GetInstance().RemoveObserverImpl(observer);
}

void DspExceptionNotifier::Notify() NN_NOEXCEPT
{
    GetInstance().NotifyImpl();
}

}}} // namespace nn::audio::dsp
