﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audio_DspCommon.h"
#include "audio_Cache.h"
#include <nn/audio/audio_Adpcm.h> // AdpcmFrameSampleCount
#include "../common/audio_CommonWaveBuffer.h"

#include "audio_DspUtility.h"

namespace nn {
namespace audio {

int DecodePcm16(int32_t* output, const common::WaveBuffer* pWaveBuffer, int32_t channel, int32_t channelCount, int32_t offset, int sampleCount)
{
    if (pWaveBuffer->buffer == static_cast<DspAddr>(0) ||
        pWaveBuffer->size == 0 ||
        pWaveBuffer->endSampleOffset < pWaveBuffer->startSampleOffset)
    {
        return 0;
    }

    const int16_t* p = reinterpret_cast<const int16_t*>(pWaveBuffer->buffer) + (pWaveBuffer->startSampleOffset + offset) * channelCount;
    int availableSampleCount = (pWaveBuffer->endSampleOffset - pWaveBuffer->startSampleOffset) - offset;
    int processSampleCount = Min(sampleCount, availableSampleCount);

    NN_AUDIO_TRACK_BUFFER(p, processSampleCount * sizeof(int16_t) * channelCount, dsp::CachedBuffer_RequireInval);
    if (channelCount == 1)
    {
        NN_AUDIO_DSP_ASSERT(channel == 0);
        for (int i = 0; i < processSampleCount; ++i)
        {
            output[i] = p[i];
        }
    }
    else
    {
        for (int i = 0; i < processSampleCount; ++i)
        {
            output[i] = p[i * channelCount + channel];
        }
    }

    return processSampleCount;
}

namespace {

const int AdpcmFrameNibbleCount = 16;

int ConvertAdpcmSamplesToNibbles(int sampleCount)
{
    int frameCount = sampleCount / AdpcmFrameSampleCount;
    int leftSampleCount = sampleCount % AdpcmFrameSampleCount;
    return (frameCount * AdpcmFrameNibbleCount) + leftSampleCount + (leftSampleCount ? 2 : 0);
}

}

int DecodeAdpcm(int32_t* output, const common::WaveBuffer* pWaveBuffer, int32_t offset, int sampleCount, const int16_t* pCoef, void* pContext) NN_NOEXCEPT
{
    if (pWaveBuffer->buffer == static_cast<DspAddr>(0) ||
        pWaveBuffer->size == 0 ||
        pWaveBuffer->endSampleOffset < pWaveBuffer->startSampleOffset)
    {
        return 0;
    }

    NN_AUDIO_TRACK_BUFFER(pContext, sizeof(AdpcmContext), dsp::CachedBuffer_RequireInval | dsp::CachedBuffer_RequireFlush);
    uint16_t predScale = static_cast<AdpcmContext*>(pContext)->predScale;
    int pred = (predScale >> 4) & 0xf;
    int scale = predScale & 0xf;
    int16_t h0 = static_cast<AdpcmContext*>(pContext)->history[0];
    int16_t h1 = static_cast<AdpcmContext*>(pContext)->history[1];

    NN_AUDIO_TRACK_BUFFER(pCoef, sizeof(int16_t) * 2, dsp::CachedBuffer_RequireInval);
    int c0 = pCoef[2 * pred + 0];
    int c1 = pCoef[2 * pred + 1];

    int availableSampleCount = (pWaveBuffer->endSampleOffset - pWaveBuffer->startSampleOffset) - offset;
    int processSampleCount = Min(sampleCount, availableSampleCount);

    int currentNibblePosition = ConvertAdpcmSamplesToNibbles(pWaveBuffer->startSampleOffset + offset);
    const uint8_t* p = reinterpret_cast<const uint8_t*>(pWaveBuffer->buffer) + currentNibblePosition / 2;
    NN_AUDIO_TRACK_BUFFER(p, dataInvalidateSize, dsp::CachedBuffer_RequireInval);

    auto remain = processSampleCount;

    while (remain > 0)
    {
        if (currentNibblePosition % AdpcmFrameNibbleCount == 0)
        {
            NN_AUDIO_DSP_ASSERT(reinterpret_cast<uintptr_t>(p) + 1 + (remain + 1) / 2 <= reinterpret_cast<uintptr_t>(reinterpret_cast<void*>(pWaveBuffer->buffer)) + pWaveBuffer->size);
            predScale = *p++;
            pred = predScale >> 4;
            scale = predScale & 0xf;
            c0 = pCoef[2 * pred + 0];
            c1 = pCoef[2 * pred + 1];
            currentNibblePosition += 2;

            if (remain >= AdpcmFrameSampleCount)
            {
                for (auto i = 0; i < AdpcmFrameSampleCount / 2; ++i)
                {
                    int32_t n0 = *p >> 4;
                    int32_t n1 = *p++ & 0xf;

                    const int Shift = 11;
                    int mac = (n0 << 28) >> (28 - scale - Shift);
                    mac = (mac + h0 * c0 + h1 * c1 + (1 << (Shift - 1))) >> Shift;

                    // clamping
                    int16_t out0 = static_cast<int16_t>(Clamp<16>(mac));
                    *output++ = out0;

                    mac = (n1 << 28) >> (28 - scale - Shift);
                    mac = (mac + out0 * c0 + h0 * c1 + (1 << (Shift - 1))) >> Shift;

                    // clamping
                    int16_t out1 = static_cast<int16_t>(Clamp<16>(mac));
                    *output++ = out1;

                    h1 = out0;
                    h0 = out1;
                }
                remain -= AdpcmFrameSampleCount;
                currentNibblePosition += AdpcmFrameNibbleCount - 2;
                continue;
            }
        }

        int32_t nibble = *p;
        if (currentNibblePosition++ & 0x1)
        {
            nibble &= 0xf;
            ++p;
        }
        else
        {
            nibble >>= 4;
        }
        const int Shift = 11;
        int mac = (nibble << 28) >> (28 - scale - Shift);
        mac = (mac + h0 * c0 + h1 * c1 + (1 << (Shift - 1))) >> Shift;

        // clamping
        int16_t out = static_cast<int16_t>(Clamp<16>(mac));
        *output++ = out;

        h1 = h0;
        h0 = out;
        --remain;
    }

    static_cast<AdpcmContext*>(pContext)->predScale = predScale;
    static_cast<AdpcmContext*>(pContext)->history[0] = h0;
    static_cast<AdpcmContext*>(pContext)->history[1] = h1;

    return processSampleCount;
}

}  // namespace audio
}  // namespace nn
