﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "audio_DspCommon.h"

#include "../audio_AuxTypes.h"
#include "audio_AuxBufferDsp.h"
#include "audio_Cache.h"
#include <nn/audio/audio_Common.h>
#include "audio_DspUtility.h"


namespace nn {
namespace audio {
namespace dsp {

namespace {
int32_t* BufferAddressToPointer(uint64_t address)
{
#if defined(NN_BUILD_CONFIG_ADDRESS_64)
    return reinterpret_cast<int32_t*>(address);
#else
    if (0 != (address >> 32))
    {
        return nullptr;
    }
    return reinterpret_cast<int32_t*>(static_cast<uint32_t>(address));
#endif
}
} // anonymous namespace

int32_t ReadAuxBufferDsp( AuxInfoDsp* pOutBuffer, uint64_t bufferAddr, uint32_t countMax, int32_t* pOutData, uint32_t count, uint32_t readOffset, uint32_t updateCount ) NN_NOEXCEPT
{
    if (countMax == 0)
    {
        return 0;
    }

    NN_AUDIO_DSP_ASSERT( count <= countMax );
    NN_AUDIO_DSP_ASSERT( pOutBuffer != nullptr );
    NN_AUDIO_DSP_ASSERT( pOutData != nullptr );
    NN_AUDIO_DSP_ASSERT( bufferAddr != 0 );

    InvalidateDataCache(pOutBuffer, sizeof(AuxInfoDsp));
    NN_AUDIO_TRACK_BUFFER(pOutBuffer , sizeof(AuxInfoDsp), CachedBuffer_RequireInval | CachedBuffer_RequireFlush);

    // copy data.
    int32_t* dst = pOutData;
    int32_t* srcBase = BufferAddressToPointer(bufferAddr);
    uint32_t  offset = pOutBuffer->_readOffsetCount + readOffset;
    uint32_t left = count;

    if (srcBase == nullptr || offset > countMax)
    {
        return 0;
    }

    while (left > 0)
    {
        int32_t* base = srcBase + offset;
        uint32_t cnt = Min((countMax - offset), left);

        InvalidateDataCache(base, cnt * sizeof(int32_t));
        NN_AUDIO_TRACK_BUFFER(base , cnt * sizeof(int32_t), CachedBuffer_RequireInval);
        for (uint32_t i = 0; i < cnt; i++)
        {
            *dst++ = *base++;
        }
        offset += cnt;
        offset %= countMax;
        left -= cnt;
    }

    if (updateCount)
    {
        uint32_t readOffsetCount = pOutBuffer->_readOffsetCount + updateCount;
        readOffsetCount %= countMax;
        pOutBuffer->_readOffsetCount = readOffsetCount;
        FlushDataCache(pOutBuffer, sizeof(AuxInfoDsp));
    }

    return count;
}

int32_t WriteAuxBufferDsp( AuxInfoDsp* pDspInfo, uint64_t bufferAddress, uint32_t countMax, const int32_t* pData, const uint32_t writeCount, const uint32_t writeOffset, const uint32_t updateCount ) NN_NOEXCEPT
{
    NN_AUDIO_DSP_ASSERT( writeCount <= countMax );
    NN_AUDIO_DSP_ASSERT( pDspInfo != nullptr);
    NN_AUDIO_DSP_ASSERT( pData != nullptr );
    NN_AUDIO_DSP_ASSERT( bufferAddress != 0 );

    if (countMax == 0)
    {
        return 0;
    }

    InvalidateDataCache(pDspInfo, sizeof(AuxInfoDsp));
    NN_AUDIO_TRACK_BUFFER(pDspInfo , sizeof(AuxInfoDsp), CachedBuffer_RequireInval | CachedBuffer_RequireFlush);

    // write data
    const int32_t* src = pData;
    int32_t* dstBase = BufferAddressToPointer(bufferAddress);
    uint32_t  offset = pDspInfo->_writeOffsetCount + writeOffset;
    uint32_t left = writeCount;

    if (dstBase == nullptr || offset > countMax)
    {
        return 0;
    }

    while(left > 0)
    {
        int32_t* base = dstBase + offset;
        uint32_t cnt = Min((countMax - offset), left);
        NN_AUDIO_TRACK_BUFFER(base , cnt * sizeof(int32_t), CachedBuffer_RequireFlush);
        for (uint32_t i = 0; i < cnt; i++)
        {
            *base++ = *src++;
        }

        FlushDataCache(dstBase + offset, cnt * sizeof(int32_t));

        offset += cnt;
        offset %= countMax;
        left -= cnt;
    }

    if (updateCount)
    {
        uint32_t writeOffsetCount = pDspInfo->_writeOffsetCount + updateCount;
        writeOffsetCount %= countMax;
        pDspInfo->_writeOffsetCount = writeOffsetCount;

        FlushDataCache(pDspInfo, sizeof(AuxInfoDsp));
    }

    return writeCount;
}

void ResetAuxBufferDsp(AuxInfoDsp* pOutAuxBuffer) NN_NOEXCEPT
{
    NN_AUDIO_DSP_ASSERT(pOutAuxBuffer != nullptr);
    pOutAuxBuffer->_readOffsetCount = 0;
    pOutAuxBuffer->_writeOffsetCount = 0;
    pOutAuxBuffer->_unpushedCount = 0;
}

} // namespace dsp
} // namespace audio
} // namespace n

