﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/audio/audio_AudioRendererTypes.h>
#include <nn/audio/detail/audio_AudioRendererTypesInternal.h>
#include <nn/audio/audio_PerformanceMetricsTypes.h>

// for both of Client & Server side header

namespace nn { namespace audio { namespace common {

/**
 * @brief オーディオレンダラの１オーディオフレーム単位のパフォーマンス情報のヘッダ (バージョン 1) です。
 */
struct PerformanceFrameHeaderVersion1
{
    uint32_t frameMagic;               //!< PerformanceFrameHeader を識別する ID です。
    int32_t entryCount;                //!< PerformanceFrameHeader に後続する PerformanceEntry の個数
    int32_t detailCount;               //!< PerformanceEntry 群に後続する PerformanceDetail の個数
    int32_t offsetToNextHeader;        //!< 次の PerformanceFrameHeader までのオフセット [Byte]
    int32_t totalProcessingTime;       //!< この PerformanceFrame が対象とするオーディオフレームの処理にかかった時間 [MicroSeconds]
    uint32_t frameIndex;               //!< PerformanceFrame のインデックス
};

/**
 * @brief オーディオレンダラの１オーディオフレーム単位のパフォーマンス情報のヘッダ (バージョン 2) です。
 */
struct PerformanceFrameHeaderVersion2
{
    uint32_t frameMagic;               //!< PerformanceFrameHeader を識別する ID です。
    int32_t entryCount;                //!< PerformanceFrameHeader に後続する PerformanceEntry の個数
    int32_t detailCount;               //!< PerformanceEntry 群に後続する PerformanceDetail の個数
    int32_t offsetToNextHeader;        //!< 次の PerformanceFrameHeader までのオフセット [Byte]
    int32_t totalProcessingTime;       //!< この PerformanceFrame が対象とするオーディオフレームの処理にかかった時間 [MicroSeconds]
    int32_t voiceDropCount;            //!< ボイスドロップが発生したボイスの個数
    int64_t startTime;                 //!< オーディオフレームの処理開始時間 [チック数]
    uint32_t frameIndex;               //!< PerformanceFrame のインデックス
    bool isRenderingTimeLimitExceeded; //!< レンダリング時間の上限を超えたかどうか
    int8_t _padding[11];
};


#pragma pack(1)
/**
 * @brief 各 VoiceType / SubMixType / FinalMixType のパフォーマンス情報 (バージョン 1) です。
 */
struct PerformanceEntryVersion1
{
    NodeId id;                   //!< type で示された VoiceType / SubMixType / FinalMixType / DeviceSinkType / CircularBufferSinkType の NodeId
    int32_t startTime;           //!< 各オーディオフレームの処理開始を 0 とした処理開始時間 [MicroSeconds]
    int32_t processingTime;      //!< 処理にかかった時間 [MicroSeconds]
    int8_t entryType;            //!< このエントリーが保持するパフォーマンス情報の対象の種類 (nn::audio::PerformanceEntryType を保持)
    int8_t _padding[3];
};

struct PerformanceDetailVersion1
{
    NodeId parentId;             //!< 本処理が紐づく VoiceType / SubMixType / FinalMixType の ID
    int32_t startTime;           //!< 各オーディオフレームの処理開始を起点とした処理開始時間 [MicroSeconds]
    int32_t processingTime;      //!< 処理にかかった時間 [MicroSeconds]
    int8_t detailType;           //!< この詳細なパフォーマンス情報の対象の種類 (nn::audio::PerformanceDetailType を保持)
    int8_t parentEntryType;      //!< 本処理が紐づく VoiceType / SubMixType / FinalMixType の種類 (nn::audio::PerformanceEntryType を保持)
    int8_t _padding[2];
};

/**
 * @brief 各 VoiceType / SubMixType / FinalMixType のパフォーマンス情報 (バージョン 2) です。
 */
struct PerformanceEntryVersion2
{
    NodeId id;                   //!< type で示された VoiceType / SubMixType / FinalMixType / DeviceSinkType / CircularBufferSinkType の NodeId
    int32_t startTime;           //!< 各オーディオフレームの処理開始を 0 とした処理開始時間 [MicroSeconds]
    int32_t processingTime;      //!< 処理にかかった時間 [MicroSeconds]
    int8_t entryType;            //!< このエントリーが保持するパフォーマンス情報の対象の種類 (nn::audio::PerformanceEntryType を保持)
    int8_t _padding[11];
};

/**
 * @brief 各 VoiceType / SubMixType / FinalMixType に紐づく詳細なパフォーマンス情報 (バージョン 2) です。
 */
struct PerformanceDetailVersion2
{
    NodeId parentId;             //!< 本処理が紐づく VoiceType / SubMixType / FinalMixType の ID
    int32_t startTime;           //!< 各オーディオフレームの処理開始を起点とした処理開始時間 [MicroSeconds]
    int32_t processingTime;      //!< 処理にかかった時間 [MicroSeconds]
    int8_t detailType;           //!< この詳細なパフォーマンス情報の対象の種類 (nn::audio::PerformanceDetailType を保持)
    int8_t parentEntryType;      //!< 本処理が紐づく VoiceType / SubMixType / FinalMixType の種類 (nn::audio::PerformanceEntryType を保持)
    int8_t _padding[10];
};

#pragma pack()

const char* const PerformanceFrameHeaderMagic = "PERF";
const int PerformanceDetailCountMax = 100;

inline int GetRequiredEntryCount(const detail::AudioRendererParameterInternal& parameter) NN_NOEXCEPT
{
    // For whole nodes in rendering tree.
    // each tree has one entry. ( not for whole commands)
    return  (parameter.voiceCount +
             parameter.effectCount +
             parameter.subMixCount +
             parameter.sinkCount +
             1 /* for FinalMix */);
}

inline int GetRequiredDetailCount(const detail::AudioRendererParameterInternal& parameter) NN_NOEXCEPT
{
    NN_UNUSED(parameter);
    return PerformanceDetailCountMax;
}

template <typename PerformanceFrameHeaderType, typename PerformanceEntryType, typename PerformanceDetailType>
inline size_t GetRequiredBufferSizeForPerformanceMetricsPerFrame(const detail::AudioRendererParameterInternal& parameter) NN_NOEXCEPT
{
    return sizeof(PerformanceFrameHeaderType)
         + sizeof(PerformanceEntryType) * GetRequiredEntryCount(parameter)
         + sizeof(PerformanceDetailType) * GetRequiredDetailCount(parameter);
}

}}}  // namespace nn::audio::common
