﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/os/os_Mutex.h>
#include <nn/util/util_IntrusiveList.h>

#include "audio_MixManager.h"
#include "common/audio_Util.h"
#include "common/audio_SplitterParameters.h"

namespace nn {
namespace audio {

class SplitterInfo;
class SplitterDestinationData;
typedef nn::util::IntrusiveList<SplitterInfo, nn::util::IntrusiveListBaseNodeTraits<SplitterInfo>> SplitterInfoList;
typedef nn::util::IntrusiveList<SplitterDestinationData, nn::util::IntrusiveListBaseNodeTraits<SplitterDestinationData>> SplitterDestinationList;

class SplitterDestinationData : public nn::util::IntrusiveListBaseNode<SplitterDestinationData>
{
private:
    common::SplitterDestinationId m_Id;
    MixId m_DestinationId;
    float m_MixVolume[MixBufferCountMax];
    bool m_InUse;
    bool _updateRequired;

public:
    friend SplitterInfoManager;

    struct NN_AUDIO_INFOTYPE_FILED_ALIGN InParameter
    {
        uint32_t magic;
        common::SplitterDestinationId _id;
        float _mixVolume[MixBufferCountMax];
        MixId _destinationId;
        bool _inUse;
        int8_t _padding[3];
    };
    NN_AUDIO_INFOTYPE_CHECK(InParameter, 112);

private:
    void  SetInUse(bool val) NN_NOEXCEPT;
    bool  IsInUse() const NN_NOEXCEPT;
    void  ClearMixVolumes() NN_NOEXCEPT;

protected:
    static size_t GetInParameterSize(int dataCount) NN_NOEXCEPT;

public:
    explicit SplitterDestinationData(int index) NN_NOEXCEPT;
    void Initialize() NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

    common::SplitterDestinationId GetId() const NN_NOEXCEPT;
    MixId GetDestinationId() const NN_NOEXCEPT;
    void  SetDestinationId(nn::audio::MixId val) NN_NOEXCEPT;
    float GetMixVolume(int index) const NN_NOEXCEPT;
    void  SetMixVolume(float volume, int index) NN_NOEXCEPT;

    bool NeedToUpdate() const NN_NOEXCEPT;
    size_t StoreParameter(InParameter* pInParameter) NN_NOEXCEPT;
};


class SplitterInfo : public nn::util::IntrusiveListBaseNode<SplitterInfo>
{
private:
    bool _used;
    bool _updateRequired;
    int32_t _id;
    int32_t _channelCount;
    int32_t _sampleRate;
    SplitterDestinationList _sendList;

public:
    friend SplitterInfoManager;

    struct NN_AUDIO_INFOTYPE_FILED_ALIGN InParameterHeader
    {
        uint32_t magic;
        int32_t infoCount;
        int32_t dataCount;
        int32_t _reserved[5];
    };
    NN_AUDIO_INFOTYPE_CHECK(InParameterHeader, 32);

    struct NN_AUDIO_INFOTYPE_FILED_ALIGN InParameter
    {
        uint32_t magic;
        common::SplitterInfoId sendId;
        int32_t sampleRate;
        int32_t length;
        common::SplitterDestinationId resourceIds[1];
    };
    // 可変長につきサイズチェック不要

protected:
    void SetInUsed(bool used) NN_NOEXCEPT;
    void SetChannelCount(int channelCount) NN_NOEXCEPT;
    void SetSampleRate(int sampleRate) NN_NOEXCEPT;

public:
    static size_t GetInParameterSize(int destinationCount) NN_NOEXCEPT;

    explicit SplitterInfo(int32_t id) NN_NOEXCEPT;
    void Initialize(int32_t id) NN_NOEXCEPT;

    int32_t GetId() const NN_NOEXCEPT;
    bool GetInUsed() const NN_NOEXCEPT;
    int32_t GetChannelCount() const NN_NOEXCEPT;
    int32_t GetSampleRate() const NN_NOEXCEPT;
    int GetDestinationCount() const NN_NOEXCEPT;
    SplitterDestinationData* GetDestinationData(int destinationIndex, int channelIndex) NN_NOEXCEPT;

    void MarkAsUpdated() NN_NOEXCEPT;
    bool NeedToUpdate() const NN_NOEXCEPT;
    size_t StoreParameter(InParameter* pInParameter) NN_NOEXCEPT;
};

class SplitterInfoManager
{
private:
    int m_SplitterInfoCount;
    SplitterInfo* m_SplitterInfos;
    SplitterInfoList m_SplitterInfoFreeList;
    int m_SendDataCount;
    SplitterDestinationData* m_SendData;
    SplitterDestinationList m_SendDataFreeList;

    nn::os::Mutex m_Mutex;
    int m_DestinationCountMax;
    bool m_Initialized;
    SplitterInfo* Allocate() NN_NOEXCEPT;

public:
    // Initialization
    SplitterInfoManager() NN_NOEXCEPT;
    void Initialize(
        int sendInfoCount,
        int sendDataCount,
        int subMixCount,
        void* pSendInfoBuffer, size_t sendInfoBufferSize,
        void* pSendDataBuffer, size_t sendDataBufferSize) NN_NOEXCEPT;
    static size_t GetInParamterBufferSize(int sendInfoCount, int sendDataCount, int subMixCount) NN_NOEXCEPT;

    static size_t GetSplitterInfoWorkBufferSize(int sendInfoCount) NN_NOEXCEPT;
    static size_t GetSplitterDestinationWorkBufferSize(int sendDataCount) NN_NOEXCEPT;

    // operation
    SplitterInfo* AcquireSplitterInfo(int sampleRate, int channelCount, int destinationCount) NN_NOEXCEPT;
    void ReleaseSplitterInfo(SplitterInfo* pSendInfo) NN_NOEXCEPT;
    size_t UpdateInParameter(void* pParameter) NN_NOEXCEPT;
};

}  // namespace audio
}  // namespace nn
