﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <mutex>
#include <cstdlib>
#include <cstring>

#include <nn/os/os_Cache.h>
#include <nn/nn_SdkAssert.h>
#include <nn/util/util_BytePtr.h>
#include <nn/audio/audio_VoiceTypes.h>
#include <nn/audio/detail/audio_Log.h>
#include "common/audio_Util.h"
#include "common/audio_SplitterParameters.h"
#include "common/audio_WorkBufferAllocator.h"

#include "audio_VoiceInfoManager.h"

namespace nn {
namespace audio {

namespace
{

int GetDestinationCountMaxPerSingleSplitterInfo(int subMixCount) NN_NOEXCEPT
{
    return VoiceType::ChannelCountMax * (subMixCount + 1 /* finalMix */);
}

}

SplitterInfoManager::SplitterInfoManager() NN_NOEXCEPT
    : m_SplitterInfoCount(0)
    , m_SplitterInfos(nullptr)
    , m_Mutex(false)
    , m_DestinationCountMax(0)
    , m_Initialized(false)
{
    m_SplitterInfoFreeList.clear();
    m_SendDataFreeList.clear();
}

size_t SplitterInfoManager::GetInParamterBufferSize(int splitterInfoCount, int splitterSendChannelCount, int subMixCount) NN_NOEXCEPT
{
    auto headerSize = sizeof(SplitterInfo::InParameterHeader);
    auto infoSize = splitterInfoCount * SplitterInfo::GetInParameterSize(GetDestinationCountMaxPerSingleSplitterInfo(subMixCount));
    auto dataSize = splitterSendChannelCount * sizeof(SplitterDestinationData::InParameter);
    return nn::util::align_up(headerSize + infoSize + dataSize, nn::audio::common::InfoTypeFieldAlignSize);
}

size_t SplitterInfoManager::GetSplitterInfoWorkBufferSize(int splitterInfoCount) NN_NOEXCEPT
{
    return sizeof(SplitterInfo) * splitterInfoCount;
}

size_t SplitterInfoManager::GetSplitterDestinationWorkBufferSize(int splitterSendChannelCount) NN_NOEXCEPT
{
    return sizeof(SplitterDestinationData) * splitterSendChannelCount;
}

void SplitterInfoManager::Initialize(
    int splitterInfoCount,
    int splitterSendChannelCount,
    int subMixCount,
    void* pSplitterInfoBuffer, size_t splitterInfoBufferSize,
    void* pSendDataBuffer, size_t splitterDestinationDataBufferSize) NN_NOEXCEPT
{
    NN_SDK_ASSERT_GREATER_EQUAL(splitterInfoBufferSize, GetSplitterInfoWorkBufferSize(splitterInfoCount));
    NN_SDK_ASSERT_GREATER_EQUAL(splitterDestinationDataBufferSize, GetSplitterDestinationWorkBufferSize(splitterSendChannelCount));
    NN_UNUSED(splitterInfoBufferSize);
    NN_UNUSED(splitterDestinationDataBufferSize);

    m_DestinationCountMax = subMixCount + 1;
    m_SplitterInfoCount = splitterInfoCount;
    m_SplitterInfos = reinterpret_cast<SplitterInfo*>(pSplitterInfoBuffer);
    m_SplitterInfoFreeList.clear();
    m_SendDataCount = splitterSendChannelCount;
    m_SendData = reinterpret_cast<SplitterDestinationData*>(pSendDataBuffer);
    m_SendDataFreeList.clear();

    if (m_SplitterInfos == nullptr || m_SendData == nullptr)
    {
        return;
    }

    for (auto i = 0; i < m_SplitterInfoCount; ++i)
    {
        m_SplitterInfos[i].Initialize(i);
        m_SplitterInfoFreeList.push_back(m_SplitterInfos[i]);
    }

    for (auto i = 0; i < m_SendDataCount; ++i)
    {
        m_SendData[i].Initialize();
        m_SendDataFreeList.push_back(m_SendData[i]);
    }
    m_Initialized = true;
}

SplitterInfo* SplitterInfoManager::Allocate() NN_NOEXCEPT
{
    if (m_SplitterInfoFreeList.size() > 0)
    {
        auto p = &m_SplitterInfoFreeList.front();
        m_SplitterInfoFreeList.pop_front();
        return p;
    }
    return nullptr;
}

nn::audio::SplitterInfo* SplitterInfoManager::AcquireSplitterInfo(int sampleRate, int channelCount, int destinationCount) NN_NOEXCEPT
{
    NN_SDK_ASSERT(destinationCount <= m_DestinationCountMax, "destinationCount must be less than total number of SubMix & FinalMix.");
    if (destinationCount > m_DestinationCountMax || m_Initialized == false)
    {
        return nullptr;
    }

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);

    auto dataCount = channelCount * destinationCount;
    SplitterInfo* p = nullptr;
    if ((m_SendDataFreeList.size() < dataCount) ||
        (p = Allocate()) == nullptr)
    {
        return nullptr;
    }

    for (auto i = 0; i < dataCount; ++i)
    {
        auto& d = m_SendDataFreeList.front();
        m_SendDataFreeList.pop_front();
        d.Initialize();
        p->_sendList.push_back(d);
    }
    p->SetInUsed(true);
    p->SetChannelCount(channelCount);
    p->SetSampleRate(sampleRate);
    return p;
}

void SplitterInfoManager::ReleaseSplitterInfo(SplitterInfo* pSplitterInfo) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pSplitterInfo);
    NN_SDK_ASSERT_GREATER(pSplitterInfo->_sendList.size(), 0);
    NN_SDK_ASSERT(pSplitterInfo->GetInUsed());

    if (m_Initialized == false)
    {
        return;
    }

    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    pSplitterInfo->SetInUsed(false);

    auto dataCount = pSplitterInfo->_sendList.size();
    for (auto i = 0; i < dataCount; ++i)
    {
        auto& d = pSplitterInfo->_sendList.front();
        pSplitterInfo->_sendList.pop_front();
        d.Finalize();
        m_SendDataFreeList.push_back(d);
    }
    m_SplitterInfoFreeList.push_back(*pSplitterInfo);
}

size_t SplitterInfoManager::UpdateInParameter(void* pParameter) NN_NOEXCEPT
{
    if (m_Initialized == false)
    {
        return 0;
    }

    // SplitterInfo
    std::lock_guard<nn::os::Mutex> lock(m_Mutex);
    auto base = nn::util::BytePtr(pParameter);
    auto pInParamHeader = reinterpret_cast<SplitterInfo::InParameterHeader*>(base.Get());
    size_t consumedSize = sizeof(SplitterInfo::InParameterHeader);
    auto infoCount = 0;
    auto dataCount = 0;
    for (auto i = 0; i < m_SplitterInfoCount; ++i)
    {
        auto pInParam = reinterpret_cast<SplitterInfo::InParameter*>((base + consumedSize).Get());
        consumedSize += m_SplitterInfos[i].StoreParameter(pInParam);
        ++infoCount;
    }

    // SplitterDestinationData
    for (auto i = 0; i < m_SendDataCount; ++i)
    {
        auto pInParam = reinterpret_cast<SplitterDestinationData::InParameter*>((base + consumedSize).Get());
        auto size = m_SendData[i].StoreParameter(pInParam);
        if (size > 0)
        {
            consumedSize += size;
            ++dataCount;
        }
    }

    pInParamHeader->magic = common::GetSplitterInParamHeaderMagic();
    pInParamHeader->infoCount = infoCount;
    pInParamHeader->dataCount = dataCount;

    // NN_DETAIL_AUDIO_TRACE("consumed:%llu\n", nn::util::align_up(consumedSize, nn::audio::common::InfoTypeFieldAlignSize));
    return nn::util::align_up(consumedSize, nn::audio::common::InfoTypeFieldAlignSize);
}

//////////////////////////////////////////////////////////////////////////

SplitterInfo::SplitterInfo(int32_t id) NN_NOEXCEPT
{
    Initialize(id);
}

void SplitterInfo::Initialize(int32_t id) NN_NOEXCEPT
{
    _used = false;
    _id = id;
    _sendList.clear();
}

int32_t SplitterInfo::GetId() const NN_NOEXCEPT
{
    return _id;
}

void SplitterInfo::SetInUsed(bool used) NN_NOEXCEPT
{
    _updateRequired = true;
    _used = used;
}

void SplitterInfo::SetChannelCount(int channelCount) NN_NOEXCEPT
{
    _updateRequired = true;
    _channelCount = channelCount;
}

int32_t SplitterInfo::GetSampleRate() const NN_NOEXCEPT
{
    return _sampleRate;
}

void SplitterInfo::SetSampleRate(int sampleRate) NN_NOEXCEPT
{
    _updateRequired = true;
    _sampleRate = sampleRate;
}

bool SplitterInfo::NeedToUpdate() const NN_NOEXCEPT
{
    return _updateRequired;
}

bool SplitterInfo::GetInUsed() const NN_NOEXCEPT
{
    return _used;
}

int32_t SplitterInfo::GetChannelCount() const NN_NOEXCEPT
{
    return _channelCount;
}

int SplitterInfo::GetDestinationCount() const NN_NOEXCEPT
{
    return _sendList.size() / _channelCount;
}

nn::audio::SplitterDestinationData* SplitterInfo::GetDestinationData(int destinationIndex, int channelIndex) NN_NOEXCEPT
{
    auto position = destinationIndex * _channelCount + channelIndex % _channelCount;
    NN_SDK_ASSERT_LESS(position, _sendList.size());
    auto itr = _sendList.begin();
    int index = 0;
    for (; index < position; ++index)
    {
        ++itr;
    }
    return &(*itr);
}

void SplitterInfo::MarkAsUpdated() NN_NOEXCEPT
{
    _updateRequired = true;
}

size_t SplitterInfo::GetInParameterSize(int destinationCount) NN_NOEXCEPT
{
    return sizeof(SplitterInfo::InParameter) + sizeof(common::SplitterDestinationId) * (destinationCount - 1);
}

size_t SplitterInfo::StoreParameter(InParameter* pInParameter) NN_NOEXCEPT
{
    if (NeedToUpdate() == false)
    {
        return 0;
    }
    pInParameter->magic = common::GetSplitterInfoMagic();
    pInParameter->sendId = GetId();
    pInParameter->sampleRate = GetSampleRate();
    pInParameter->length = _sendList.size();
    common::SplitterDestinationId* dst = pInParameter->resourceIds;
    for (auto src = _sendList.begin(); src != _sendList.end(); ++src)
    {
        *dst = src->GetId();
        ++dst;
    }
    _updateRequired = false;
    return SplitterInfo::GetInParameterSize(pInParameter->length);
}

//////////////////////////////////////////////////////////////////////////

SplitterDestinationData::SplitterDestinationData(int index) NN_NOEXCEPT
    : m_Id(index)
{
    Initialize();
}

void SplitterDestinationData::Initialize() NN_NOEXCEPT
{
    SetInUse(false);
    SetDestinationId(Invalid_MixId);
    ClearMixVolumes();
}

void SplitterDestinationData::Finalize() NN_NOEXCEPT
{
    // Clean up.
    Initialize();
}

size_t SplitterDestinationData::GetInParameterSize(int dataCount) NN_NOEXCEPT
{
    return dataCount * sizeof(InParameter);
}

nn::audio::common::SplitterDestinationId SplitterDestinationData::GetId() const NN_NOEXCEPT
{
    return m_Id;
}

bool SplitterDestinationData::IsInUse() const NN_NOEXCEPT
{
    return m_InUse;
}

void SplitterDestinationData::SetInUse(bool val) NN_NOEXCEPT
{
    m_InUse = val;
}

void SplitterDestinationData::SetDestinationId(nn::audio::MixId val) NN_NOEXCEPT
{
    SetInUse(val != Invalid_MixId);
    _updateRequired = true;
    m_DestinationId = val;
}

nn::audio::MixId SplitterDestinationData::GetDestinationId() const NN_NOEXCEPT
{
    return m_DestinationId;
}

void SplitterDestinationData::ClearMixVolumes() NN_NOEXCEPT
{
    memset(m_MixVolume, 0, sizeof(m_MixVolume));
}

float SplitterDestinationData::GetMixVolume(int index) const NN_NOEXCEPT
{
    NN_SDK_ASSERT_RANGE(index, 0, MixBufferCountMax);
    return m_MixVolume[index];
}

void SplitterDestinationData::SetMixVolume(float volume, int index) NN_NOEXCEPT
{
    NN_SDK_ASSERT_MINMAX(volume, VoiceType::GetVolumeMin(), VoiceType::GetVolumeMax());
    NN_SDK_ASSERT_RANGE(index, 0, MixBufferCountMax);
    _updateRequired = true;
    m_MixVolume[index] = volume;
}

bool SplitterDestinationData::NeedToUpdate() const NN_NOEXCEPT
{
    return _updateRequired;
}

size_t SplitterDestinationData::StoreParameter(InParameter* pInParameter) NN_NOEXCEPT
{
    if (NeedToUpdate() == false)
    {
        return 0;
    }

    pInParameter->magic = common::GetSplitterSendDataMagic();
    pInParameter->_id = GetId();
    pInParameter->_inUse = IsInUse();
    pInParameter->_destinationId = GetDestinationId();
    memcpy(pInParameter->_mixVolume, m_MixVolume, sizeof(m_MixVolume));
    _updateRequired = false;
    return sizeof(InParameter);
}

}  // namespace audio
}  // namespace nn

