﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <cstdint>
#include "audio_AddrTypes.h"
#include "common/audio_Util.h"

namespace nn { namespace audio {

class MemoryPoolManager;

class MemoryPoolInfo
{
public:
    friend MemoryPoolManager;
    enum State
    {
        State_Invalid = 0,
        State_Acquired,
        State_RequestDetach,
        State_Detached,
        State_RequestAttach,
        State_Attached,
        State_Released,
    };

    struct NN_AUDIO_INFOTYPE_FILED_ALIGN InParameter
    {
        CpuAddr  _address;
        uint64_t _size;
        State _state;
        int8_t _padding[12];
    };
    NN_AUDIO_INFOTYPE_CHECK(InParameter, 32);

    struct NN_AUDIO_INFOTYPE_FILED_ALIGN OutStatus
    {
        State _state;
        int8_t _padding[12];
    };
    NN_AUDIO_INFOTYPE_CHECK(OutStatus, 16);

public:
    MemoryPoolInfo() NN_NOEXCEPT;
    ~MemoryPoolInfo() NN_NOEXCEPT;

    // For public API
    bool Attach() NN_NOEXCEPT;
    bool Detach() NN_NOEXCEPT;
    bool IsAttached() const NN_NOEXCEPT;
    bool IsInTransition() const NN_NOEXCEPT;

    void* GetAddress() const NN_NOEXCEPT;
    size_t GetSize() const NN_NOEXCEPT;
    State GetState() const NN_NOEXCEPT;

protected:
    // For MemoryPoolManager
    void Initialize(void* address, size_t size) NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;
    bool IsUsed() const NN_NOEXCEPT;
    bool Contains(const void* address, size_t size) const NN_NOEXCEPT;
    bool IsOverlapped(const void* address, size_t size) const NN_NOEXCEPT;
    size_t Prepare(InParameter* pInParameter) NN_NOEXCEPT;
    size_t Store(const OutStatus* pOutStatus) NN_NOEXCEPT;

    static size_t GetInParameterSize() { return sizeof(InParameter); }
    static size_t GetOutStatusSize() { return sizeof(OutStatus); }

private:
    bool SetState(State state) NN_NOEXCEPT;

    CpuAddr m_Address;
    uint64_t m_Size;
    State m_State;
    bool m_InUse;
    int8_t _reserved[3];
};

}} // namespace nn::audio
