﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Result.h>
#include <nn/os/os_Mutex.h>
#include <nn/audio/audio_EffectTypes.h>
#include "audio_EffectPrivateTypes.h"
#include "audio_MixManager.h"
#include "audio_AddrTypes.h"
#include "common/audio_Util.h"

namespace nn {
namespace audio {


enum EffectType : uint8_t
{
    EffectType_Invalid,
    EffectType_BufferMixer,
    EffectType_Aux,
    EffectType_Delay,
    EffectType_Reverb,
    EffectType_I3dl2Reverb,
    EffectType_BiquadFilter
};

struct EffectInfo
{
    struct NN_AUDIO_INFOTYPE_FILED_ALIGN InParameter
    {
        EffectType _type;
        bool _isNew;
        bool _enabled;
        int8_t _padding0[1];
        MixId _attachMixId;
        CpuAddr _bufferBase;  // buffer address which user passed.
        uint64_t _bufferSize; // size of "bufferBase" buffer.
        int32_t _processingOrder;
        int8_t _padding1[4];
        static const int32_t InvalidProcessingOrder = -1;

        union
        {
            BufferMixerParameter _bufferMixerParameter;
            DelayParameter _delayParameter;
            ReverbParameter _reverbParameter;
            I3dl2ReverbParameter _i3dl2ReverbParameter;
            AuxParameter _auxParameter;
            BiquadFilterEffectParameter _biquadFilterParameter;
        }; // sizeof(this union) == 160
        int8_t _padding3[8];
    };
    NN_AUDIO_INFOTYPE_CHECK(InParameter, 192);

    struct NN_AUDIO_INFOTYPE_FILED_ALIGN OutStatus
    {
        enum UsageStatus : uint8_t
        {
            UsageStatus_Invalid = 0,
            UsageStatus_New,
            UsageStatus_Initialized,
            UsageStatus_Used,
            UsageStatus_Removable,
        };

        UsageStatus _usageStatus;
        int8_t _padding[15];
    };
    NN_AUDIO_INFOTYPE_CHECK(OutStatus, 16);

    InParameter _inParameter;
    OutStatus _outStatus;
    int8_t _reserved[8];

    EffectInfo();
};

class EffectManager
{
private:
    int32_t m_InfoCount;
    EffectInfo* m_Infos;
    nn::os::Mutex m_Mutex;
public:
    EffectManager(int count, EffectInfo* info) NN_NOEXCEPT;

    Result Add(EffectInfo** ppOutEffectInfo, EffectType type, void* buffer, size_t bufferSize, MixInfo* attachMixInfo) NN_NOEXCEPT;
    bool IsInitialized(const EffectInfo* pEffectInfo) const NN_NOEXCEPT;
    void* Remove(EffectInfo* pEffectInfo, MixInfo* attachedMixInfo) NN_NOEXCEPT;
    size_t UpdateEffectsInParameter(void* pInParameter) NN_NOEXCEPT;
    size_t UpdateEffectOutStatus(EffectInfo::OutStatus* pOutStatus) NN_NOEXCEPT;
};

}  // namespace audio
}  // namespace nn
