﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>

#include <nn/nn_SdkAssert.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_BytePtr.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Cache.h>

#include <nn/audio/audio_Common.h>
#include <nn/audio/audio_Result.h>
#include <nn/audio/audio_AudioRendererTypes.h>
#include <nn/audio/audio_EffectTypes.h>
#include <nn/audio/audio_EffectApi.h>
#include <nn/audio/audio_SubMixTypes.h>
#include <nn/audio/audio_FinalMixTypes.h>
#include <nn/audio/detail/audio_Log.h>

#include "audio_EffectManager.h"
#include "audio_EffectPrivateTypes.h"
#include "audio_MixManager.h"

#include "audio_AuxTypes.h"
#include "audio_AuxBufferApi.h"
#include "audio_EffectUtil.h"
#include "dsp/audio_EffectCommon.h"
#include "dsp/audio_EffectReverb.h"
#include "dsp/audio_EffectI3dl2Reverb.h"
#include "common/audio_Command.h"

#include "audio_ResourceExclusionChecker.h"
#define NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pEffect) \
    detail::ScopedConfigInstanceAccessChecker scopedConfigInstanceAccessChecker(detail::FindResourceExclusionCheckerFromRegionInConfig(pEffect->_pEffectInfo))

namespace nn {
namespace audio {

NN_DEFINE_STATIC_CONSTANT(const int DelayParameter::ChannelCountMax);
NN_DEFINE_STATIC_CONSTANT(const int DelayParameter::SupportedChannelCountMax);
NN_DEFINE_STATIC_CONSTANT(const int ReverbParameter::ChannelCountMax);
NN_DEFINE_STATIC_CONSTANT(const int ReverbParameter::SupportedChannelCountMax);
NN_DEFINE_STATIC_CONSTANT(const int I3dl2ReverbParameter::ChannelCountMax);
NN_DEFINE_STATIC_CONSTANT(const int I3dl2ReverbParameter::SupportedChannelCountMax);
NN_DEFINE_STATIC_CONSTANT(const int BiquadFilterType::ChannelCountMax);

/////////////////////////////////////////////////////////////////////////////
// General purpose
/////////////////////////////////////////////////////////////////////////////

#define NN_AUDIO_REQUIRES_EFFECT_TYPE(effect, type) \
    NN_SDK_REQUIRES_NOT_NULL((effect)); \
    NN_SDK_REQUIRES_NOT_NULL((effect)->_pEffectInfo); \
    NN_SDK_REQUIRES_EQUAL((effect)->_pEffectInfo->_inParameter._type, (type));

#define NN_AUDIO_CHECK_IF_INITIALIZED(manager, effect)   \
if ((manager)->IsInitialized((effect)->_pEffectInfo)) \
{ \
    NN_DETAIL_AUDIO_WARN("[nn::audio] Warning: Passed effect is already initialized. Please confirm if %p is valid effect object.|\n", (effect)); \
} \

template<typename Effect> bool IsRemovable(const Effect* pEffect, EffectType type) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pEffect);
    return ( pEffect->_pEffectInfo != nullptr ) &&
           ( pEffect->_pEffectInfo->_inParameter._type == type ) &&
           ( (pEffect->_pEffectInfo->_outStatus._usageStatus == EffectInfo::OutStatus::UsageStatus_New ) || // 一度も update されていなければ即 Remove 可
             (pEffect->_pEffectInfo->_outStatus._usageStatus == EffectInfo::OutStatus::UsageStatus_Removable ));
}

// TODO: update delay parameter setter/getter with bellows
template< typename Config, typename Field >
void SetQfConfig(Config* c, Field f, float value, float rangeFrom, float rangeTo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(c);
    NN_SDK_REQUIRES_MINMAX(value, rangeFrom, rangeTo);
    NN_UNUSED(rangeFrom);
    NN_UNUSED(rangeTo);
    (*c).*f = qf(value * ( 1 << QF_FRACTIONAL_BIT_COUNT));
};

template< typename Config, typename Field >
float GetQfConfig(const Config* c, Field f) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(c);
    return float((*c).*f) / float(1 << QF_FRACTIONAL_BIT_COUNT);
};

template< typename Config, typename Field, typename ValueType >
void SetConfig(Config* c, Field f, ValueType value, ValueType rangeFrom, ValueType rangeTo) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(c);
    NN_SDK_REQUIRES_MINMAX(value, rangeFrom,rangeTo);
    NN_UNUSED(rangeFrom);
    NN_UNUSED(rangeTo);
    (*c).*f = value;
};

template< typename Config, typename Field, typename ValueType >
ValueType GetConfig(const Config* c, Field f) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(c);
    return ((*c).*f);
};


/////////////////////////////////////////////////////////////////////////////
// BufferMixer
/////////////////////////////////////////////////////////////////////////////

template <typename T> Result AddBufferMixerImpl(AudioRendererConfig* pConfig, BufferMixerType* pBufferMixer, T* pMixNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pBufferMixer);
    NN_SDK_REQUIRES_NOT_NULL(pMixNode);
    NN_SDK_REQUIRES_NOT_NULL(pMixNode->_pMixInfo);

    auto pEffectManager = pConfig->_pEffectManager;
    NN_AUDIO_CHECK_IF_INITIALIZED(pEffectManager, pBufferMixer);

    NN_RESULT_DO(pEffectManager->Add(&pBufferMixer->_pEffectInfo, EffectType_BufferMixer, nullptr, 0, pMixNode->_pMixInfo));
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pBufferMixer);
    memset(&pBufferMixer->_pEffectInfo->_inParameter._bufferMixerParameter, 0, sizeof(pBufferMixer->_pEffectInfo->_inParameter._bufferMixerParameter));
    pBufferMixer->_pEffectInfo->_inParameter._enabled = true;

    NN_RESULT_SUCCESS;
}

Result AddBufferMixer(AudioRendererConfig* pConfig, BufferMixerType* pBufferMixer, FinalMixType* pFinalMix) NN_NOEXCEPT
{
    return AddBufferMixerImpl(pConfig, pBufferMixer, pFinalMix);
}

Result AddBufferMixer(AudioRendererConfig* pConfig, BufferMixerType* pBufferMixer, SubMixType* pSubMix) NN_NOEXCEPT
{
    return AddBufferMixerImpl(pConfig, pBufferMixer, pSubMix);
}

template<typename T> void RemoveBufferMixerImpl(AudioRendererConfig* pConfig, BufferMixerType* pBufferMixer, T* pMixNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pBufferMixer);
    NN_SDK_REQUIRES_NOT_NULL(pBufferMixer->_pEffectInfo);
    NN_SDK_REQUIRES_EQUAL(pBufferMixer->_pEffectInfo->_inParameter._type, EffectType_BufferMixer);
    NN_SDK_REQUIRES(IsBufferMixerRemovable(pBufferMixer), "pBufferMixer is being used. Check state with \"IsBufferMixerRemovable()\"");
    NN_SDK_REQUIRES_NOT_NULL(pMixNode);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pBufferMixer);

    NN_UNUSED(pMixNode);

    auto pEffectManager = pConfig->_pEffectManager;
    pEffectManager->Remove(pBufferMixer->_pEffectInfo, pMixNode->_pMixInfo);
    pBufferMixer->_pEffectInfo = nullptr;
}

void RemoveBufferMixer(AudioRendererConfig* pConfig, BufferMixerType* pBufferMixer, FinalMixType* pFinalMix) NN_NOEXCEPT
{
    RemoveBufferMixerImpl(pConfig, pBufferMixer, pFinalMix);
}

void RemoveBufferMixer(AudioRendererConfig* pConfig, BufferMixerType* pBufferMixer, SubMixType* pSubMix) NN_NOEXCEPT
{
    RemoveBufferMixerImpl(pConfig, pBufferMixer, pSubMix);
}

void GetBufferMixerInputOutput(const BufferMixerType* pBufferMixer, int8_t* outInput, int8_t* outOutput, int* pOutCount, int count) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pBufferMixer, EffectType_BufferMixer);
    NN_SDK_REQUIRES_NOT_NULL(outInput);
    NN_SDK_REQUIRES_NOT_NULL(outOutput);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_RANGE(count, 0, MixBufferCountMax + 1);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pBufferMixer);


    auto c = std::min(count, pBufferMixer->_pEffectInfo->_inParameter._bufferMixerParameter._channelCountMax);
    for (int i = 0; i < c; ++i)
    {
        outInput[i] = pBufferMixer->_pEffectInfo->_inParameter._bufferMixerParameter._input[i];
        outOutput[i] = pBufferMixer->_pEffectInfo->_inParameter._bufferMixerParameter._output[i];
    }
    *pOutCount = c;
}

void SetBufferMixerInputOutput(BufferMixerType* pBufferMixer, const int8_t* input, const int8_t* output, int count) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pBufferMixer, EffectType_BufferMixer);
    NN_SDK_REQUIRES_NOT_NULL(input);
    NN_SDK_REQUIRES_NOT_NULL(output);
    NN_SDK_REQUIRES_RANGE(count, 0, MixBufferCountMax + 1);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pBufferMixer);


    auto& p = pBufferMixer->_pEffectInfo->_inParameter._bufferMixerParameter;
    auto c = std::min(count, MixBufferCountMax);
    for (int i = 0; i < c; ++i)
    {
        p._input[i] = input[i];
        p._output[i] = output[i];
    }
    p._channelCountMax = c;
}

float GetBufferMixerVolume(const BufferMixerType* pBufferMixer, int index) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pBufferMixer, EffectType_BufferMixer);
    NN_SDK_REQUIRES_RANGE(index, 0, MixBufferCountMax);

    return pBufferMixer->_pEffectInfo->_inParameter._bufferMixerParameter._volume[index];
}

void SetBufferMixerVolume(BufferMixerType* pBufferMixer, int index, float volume) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pBufferMixer, EffectType_BufferMixer);
    NN_SDK_REQUIRES_RANGE(index, 0, MixBufferCountMax);
    NN_SDK_REQUIRES_MINMAX(volume, 0.0f, 2.0f);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pBufferMixer);

    pBufferMixer->_pEffectInfo->_inParameter._bufferMixerParameter._volume[index] = volume;
}

bool IsBufferMixerEnabled(const BufferMixerType* pBufferMixer) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pBufferMixer, EffectType_BufferMixer);

    return pBufferMixer->_pEffectInfo->_inParameter._enabled;
}

void SetBufferMixerEnabled(BufferMixerType* pBufferMixer, bool enable) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pBufferMixer, EffectType_BufferMixer);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pBufferMixer);

    pBufferMixer->_pEffectInfo->_inParameter._enabled = enable;
}

bool IsBufferMixerRemovable(const BufferMixerType* pBufferMixer) NN_NOEXCEPT
{
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pBufferMixer);

    return IsRemovable(pBufferMixer, EffectType_BufferMixer)
            && (IsBufferMixerEnabled(pBufferMixer) == false);
}

/////////////////////////////////////////////////////////////////////////////
// Delay
/////////////////////////////////////////////////////////////////////////////
size_t GetRequiredBufferSizeForDelay(nn::TimeSpan delayTimeMax, int sampleRate, int channelCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES((sampleRate == 32000) || (sampleRate == 48000));
    NN_SDK_REQUIRES((channelCount == 1) || (channelCount == 2) || (channelCount == 4) || (channelCount == 6));
    NN_SDK_REQUIRES_LESS(delayTimeMax.GetMilliSeconds(), std::numeric_limits<int32_t>::max());

    size_t sampleSize = sizeof(qf) * static_cast<size_t>(channelCount * sampleRate * delayTimeMax.GetMilliSeconds() / 1000);
    return nn::util::align_up(sampleSize, nn::audio::BufferAlignSize);
}

template<typename T> Result AddDelayImpl(AudioRendererConfig* pConfig, DelayType* pDelay, void* buffer, size_t bufferSize,  T* pMixNode, nn::TimeSpan delayTimeMax, int channelCountMax) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pDelay);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_ASSERT_ALIGNED(reinterpret_cast<uintptr_t>(buffer), nn::audio::BufferAlignSize);
    NN_SDK_ASSERT_GREATER_EQUAL(bufferSize, GetRequiredBufferSizeForDelay(delayTimeMax, pMixNode->_pMixInfo->sampleRate, channelCountMax));
    NN_SDK_REQUIRES_NOT_NULL(pMixNode);

    NN_SDK_REQUIRES((channelCountMax == 1) || (channelCountMax == 2) || (channelCountMax == 4) || (channelCountMax == 6), "Delay effect supports only 1, 2, 4, or 6 channels.");
    NN_SDK_REQUIRES_LESS(delayTimeMax.GetMilliSeconds(), std::numeric_limits<int32_t>::max());

    auto pEffectManager = pConfig->_pEffectManager;
    NN_AUDIO_CHECK_IF_INITIALIZED(pEffectManager, pDelay);

    NN_RESULT_DO(pEffectManager->Add(&pDelay->_pEffectInfo, EffectType_Delay, buffer, bufferSize, pMixNode->_pMixInfo));
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pDelay);

    memset(&pDelay->_pEffectInfo->_inParameter._delayParameter, 0, sizeof(DelayParameter));
    pDelay->_pEffectInfo->_inParameter._delayParameter._delayTimeMax = static_cast<int32_t>(delayTimeMax.GetMilliSeconds());
    pDelay->_pEffectInfo->_inParameter._delayParameter._numChannelCountMax = static_cast<uint16_t>(channelCountMax);
    pDelay->_pEffectInfo->_inParameter._delayParameter._numChannels = static_cast<uint16_t>(channelCountMax);
    pDelay->_pEffectInfo->_inParameter._delayParameter._delayTime = 0;
    pDelay->_pEffectInfo->_inParameter._delayParameter._inGain = 0;
    pDelay->_pEffectInfo->_inParameter._delayParameter._feedbackGain = 0;
    pDelay->_pEffectInfo->_inParameter._delayParameter._outGain = qf(1.0f * (1 << QF_FRACTIONAL_BIT_COUNT)); // outGain is fixed to 1.0f
    pDelay->_pEffectInfo->_inParameter._delayParameter._dryGain = qf(1.0f * (1 << QF_FRACTIONAL_BIT_COUNT)); // pass through by default.
    pDelay->_pEffectInfo->_inParameter._delayParameter._channelSpread = 0;
    pDelay->_pEffectInfo->_inParameter._delayParameter._lowPassAmount = 0;
    pDelay->_pEffectInfo->_inParameter._delayParameter._sampleRate = pMixNode->_pMixInfo->sampleRate * (1 << QF_FRACTIONAL_BIT_COUNT);
    pDelay->_pEffectInfo->_inParameter._enabled = true;

    NN_RESULT_SUCCESS;
}

Result AddDelay(AudioRendererConfig* pConfig, DelayType* pDelay, void* buffer, size_t bufferSize, FinalMixType* pFinalMix, nn::TimeSpan delayTimeMax, int channelCountMax) NN_NOEXCEPT
{
    return AddDelayImpl(pConfig, pDelay, buffer, bufferSize, pFinalMix, delayTimeMax, channelCountMax);
}

Result AddDelay(AudioRendererConfig* pConfig, DelayType* pDelay, void* buffer, size_t bufferSize, SubMixType* pSubMix, nn::TimeSpan delayTimeMax, int channelCountMax) NN_NOEXCEPT
{
    return AddDelayImpl(pConfig, pDelay, buffer, bufferSize, pSubMix, delayTimeMax, channelCountMax);
}

template<typename T> void* RemoveDelayImpl(AudioRendererConfig* pConfig, DelayType* pDelay, T* pMixNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pDelay, EffectType_Delay);
    NN_SDK_REQUIRES_EQUAL(IsDelayRemovable(pDelay), true);
    NN_SDK_REQUIRES_NOT_NULL(pMixNode);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pDelay);

    NN_UNUSED(pMixNode);

    auto pEffectManager = pConfig->_pEffectManager;
    void* returnAddress = pEffectManager->Remove(pDelay->_pEffectInfo, pMixNode->_pMixInfo);
    pDelay->_pEffectInfo = nullptr;
    return returnAddress;
}

void* RemoveDelay(AudioRendererConfig* pConfig, DelayType* pDelay, FinalMixType* pFinalMix) NN_NOEXCEPT
{
    return RemoveDelayImpl(pConfig, pDelay, pFinalMix);
}

void* RemoveDelay(AudioRendererConfig* pConfig, DelayType* pDelay, SubMixType* pSubMix) NN_NOEXCEPT
{
    return RemoveDelayImpl(pConfig, pDelay, pSubMix);
}

bool IsDelayRemovable(const DelayType* pDelay) NN_NOEXCEPT
{
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pDelay);

    return IsRemovable(pDelay, EffectType_Delay)
            && (IsDelayEnabled(pDelay) == false);
}

void GetDelayInputOutput(const DelayType* pDelay, int8_t* outInput, int8_t* outOutput, int* pOutCount, int count) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pDelay, EffectType_Delay);
    NN_SDK_REQUIRES_NOT_NULL(outInput);
    NN_SDK_REQUIRES_NOT_NULL(outOutput);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pDelay);

    auto& p = pDelay->_pEffectInfo->_inParameter._delayParameter;
    NN_SDK_REQUIRES_RANGE(count, 0, p._numChannels + 1);

    auto c = std::min(static_cast<uint16_t>(count), p._numChannels);
    for (int i = 0; i < c; ++i)
    {
        outInput[i] = p._input[i];
        outOutput[i] = p._output[i];
    }
    *pOutCount = p._numChannels;
}

void SetDelayInputOutput(DelayType* pDelay, const int8_t* input, const int8_t* output, int count) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pDelay, EffectType_Delay);
    NN_SDK_REQUIRES_NOT_NULL(input);
    NN_SDK_REQUIRES_NOT_NULL(output);
    NN_SDK_REQUIRES_MINMAX(count, 0, nn::audio::DelayParameter::SupportedChannelCountMax );
    NN_SDK_REQUIRES_MINMAX(count, 0, nn::audio::GetDelayChannelCountMax(pDelay));
    NN_SDK_REQUIRES( (count == 1) || (count == 2) || (count == 4) || (count == 6) );
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pDelay);

    auto& p = pDelay->_pEffectInfo->_inParameter._delayParameter;
    p._parameterStatus = EffectParameterStatus_Init;

    auto c = std::min(count,
                      std::min(static_cast<int>(pDelay->_pEffectInfo->_inParameter._delayParameter._numChannelCountMax),
                               nn::audio::DelayParameter::SupportedChannelCountMax));
    for (int i = 0; i < c; ++i)
    {
        p._input[i] = input[i];
        p._output[i] = output[i];
    }
    p._numChannels = static_cast<uint16_t>(c);
}

nn::TimeSpan GetDelayTimeMax(const DelayType* pDelay) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pDelay, EffectType_Delay);
    nn::TimeSpan ret = nn::TimeSpan::FromMilliSeconds(pDelay->_pEffectInfo->_inParameter._delayParameter._delayTimeMax);
    return ret;
}

int GetDelayChannelCountMax(const DelayType* pDelay) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pDelay, EffectType_Delay);
    return pDelay->_pEffectInfo->_inParameter._delayParameter._numChannelCountMax;
}

nn::TimeSpan GetDelayTime(const DelayType* pDelay) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pDelay, EffectType_Delay);
    nn::TimeSpan ret = nn::TimeSpan::FromMilliSeconds(pDelay->_pEffectInfo->_inParameter._delayParameter._delayTime);
    return ret;
}

void SetDelayTime(DelayType* pDelay, nn::TimeSpan time) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pDelay, EffectType_Delay);
    NN_SDK_REQUIRES_LESS(time.GetMilliSeconds(), std::numeric_limits<int32_t>::max());
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pDelay);
    int32_t msec = static_cast<int32_t>(time.GetMilliSeconds());
    NN_SDK_REQUIRES_LESS_EQUAL(msec, pDelay->_pEffectInfo->_inParameter._delayParameter._delayTimeMax);
    // Delay Time 変更時は Init 。
    pDelay->_pEffectInfo->_inParameter._delayParameter._parameterStatus = EffectParameterStatus_Init;
    pDelay->_pEffectInfo->_inParameter._delayParameter._delayTime = msec;
}

template<typename Field>
void SetQfConfig(DelayType* pDelay, Field f, float value, float rangeFrom, float rangeTo) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pDelay, EffectType_Delay);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pDelay);
    auto p = &pDelay->_pEffectInfo->_inParameter._delayParameter;
    p->_parameterStatus = (p->_parameterStatus == EffectParameterStatus_Init) ? p->_parameterStatus : EffectParameterStatus_UpdateParam;
    SetQfConfig(p, f, value, rangeFrom, rangeTo);
}

template<typename Field>
float GetQfConfig(const DelayType* pDelay, Field f) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pDelay, EffectType_Delay);
    auto p = &pDelay->_pEffectInfo->_inParameter._delayParameter;
    return GetQfConfig(p, f);
}

float GetDelayFeedbackGain(const DelayType* pDelay) NN_NOEXCEPT
{
    return GetQfConfig(pDelay, &DelayParameter::_feedbackGain);
}

void SetDelayFeedbackGain(DelayType* pDelay, float feedbackGain) NN_NOEXCEPT
{
    SetQfConfig(pDelay, &DelayParameter::_feedbackGain, feedbackGain, 0.0f, 1.0f);
}

float GetDelayDryGain(const DelayType* pDelay) NN_NOEXCEPT
{
    return GetQfConfig(pDelay, &DelayParameter::_dryGain);
}

void SetDelayDryGain(DelayType* pDelay, float dryGain) NN_NOEXCEPT
{
    SetQfConfig(pDelay, &DelayParameter::_dryGain, dryGain, 0.0f, 1.0f);
}

float GetDelayInGain(const DelayType* pDelay) NN_NOEXCEPT
{
    return GetQfConfig(pDelay, &DelayParameter::_inGain);
}

void SetDelayInGain(DelayType* pDelay, float inGain) NN_NOEXCEPT
{
    SetQfConfig(pDelay, &DelayParameter::_inGain, inGain, 0.0f, 1.0f);
}

float GetDelayChannelSpread(const DelayType* pDelay) NN_NOEXCEPT
{
    return GetQfConfig(pDelay, &DelayParameter::_channelSpread);
}

void SetDelayChannelSpread(DelayType* pDelay, float channelSpread) NN_NOEXCEPT
{
    SetQfConfig(pDelay, &DelayParameter::_channelSpread, channelSpread, 0.0f, 1.0f);
}

float GetDelayLowPassAmount(const DelayType* pDelay) NN_NOEXCEPT
{
    return GetQfConfig(pDelay, &DelayParameter::_lowPassAmount);
}

void SetDelayLowPassAmount(DelayType* pDelay, float lowPassAmount) NN_NOEXCEPT
{
    SetQfConfig(pDelay, &DelayParameter::_lowPassAmount, lowPassAmount, 0.0f, 1.0f);
}

bool IsDelayEnabled(const DelayType* pDelay) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pDelay, EffectType_Delay);
    return pDelay->_pEffectInfo->_inParameter._enabled;
}

void SetDelayEnabled(DelayType* pDelay, bool enable) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pDelay, EffectType_Delay);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pDelay);

    auto p = &pDelay->_pEffectInfo->_inParameter;
    p->_enabled = enable;
    p->_delayParameter._parameterStatus = enable ? EffectParameterStatus_Init : EffectParameterStatus_UpdateParam;
}

DelayParameterSet GetDelayParameters(const DelayType* pDelay) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDelay);
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pDelay, EffectType_Delay);
    auto p = &pDelay->_pEffectInfo->_inParameter;
    DelayParameterSet ret;
    ret.delayTime = nn::TimeSpan::FromMilliSeconds(p->_delayParameter._delayTime);
    ret.inGain = GetQfConfig(pDelay, &DelayParameter::_inGain);
    ret.feedbackGain = GetQfConfig(pDelay, &DelayParameter::_feedbackGain);
    ret.dryGain = GetQfConfig(pDelay, &DelayParameter::_dryGain);
    ret.channelSpread = GetQfConfig(pDelay, &DelayParameter::_channelSpread);
    ret.lowPassAmount = GetQfConfig(pDelay, &DelayParameter::_lowPassAmount);
    return ret;
}

void SetDelayParameters(DelayType* pOutDelay, const DelayParameterSet* pParameterSet) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutDelay);
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pOutDelay, EffectType_Delay);
    SetQfConfig(pOutDelay, &DelayParameter::_dryGain, pParameterSet->dryGain, 0.0f, 1.0f);
    SetQfConfig(pOutDelay, &DelayParameter::_inGain, pParameterSet->inGain, 0.0f, 1.0f);
    SetQfConfig(pOutDelay, &DelayParameter::_feedbackGain, pParameterSet->feedbackGain, 0.0f, 1.0f);
    SetQfConfig(pOutDelay, &DelayParameter::_channelSpread, pParameterSet->channelSpread, 0.0f, 1.0f);
    SetQfConfig(pOutDelay, &DelayParameter::_lowPassAmount, pParameterSet->lowPassAmount, 0.0f, 1.0f);
    SetQfConfig(pOutDelay, &DelayParameter::_lowPassAmount, pParameterSet->lowPassAmount, 0.0f, 1.0f);
    NN_SDK_REQUIRES_LESS(pParameterSet->delayTime.GetMilliSeconds(), std::numeric_limits<int32_t>::max());
    int32_t msec = static_cast<int32_t>(pParameterSet->delayTime.GetMilliSeconds());
    NN_SDK_REQUIRES_LESS_EQUAL(msec, pOutDelay->_pEffectInfo->_inParameter._delayParameter._delayTimeMax);
    pOutDelay->_pEffectInfo->_inParameter._delayParameter._parameterStatus = EffectParameterStatus_Init;
    pOutDelay->_pEffectInfo->_inParameter._delayParameter._delayTime = msec;
}

/////////////////////////////////////////////////////////////////////////////
// Aux
/////////////////////////////////////////////////////////////////////////////

size_t GetRequiredBufferSizeForAuxSendReturnBuffer( const AudioRendererParameter* pParameter, int mixBufferFrameCount, int channelCount ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pParameter);
    NN_SDK_REQUIRES_GREATER(mixBufferFrameCount, 1);
    NN_SDK_REQUIRES_GREATER(channelCount, 0);

    size_t auxSize = nn::util::align_up(pParameter->sampleCount * mixBufferFrameCount * channelCount * sizeof(int32_t), nn::audio::BufferAlignSize); // sample buffer 分
    auxSize += nn::util::align_up(sizeof(AuxBufferInfo), nn::audio::BufferAlignSize);

    return auxSize;
}

template<typename T> Result AddAuxImpl(AudioRendererConfig* pConfig, AuxType* pAux, T* pMixNode, void* pSendBuffer, void* pReturnBuffer, const size_t auxBufferSize) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pAux);
    NN_SDK_REQUIRES_NOT_NULL(pMixNode);

    NN_SDK_REQUIRES_NOT_NULL(pSendBuffer);
    NN_SDK_REQUIRES_ALIGNED(pSendBuffer, nn::audio::BufferAlignSize);
    NN_SDK_REQUIRES_NOT_NULL(pReturnBuffer);
    NN_SDK_REQUIRES_ALIGNED(pReturnBuffer, nn::audio::BufferAlignSize);
    NN_SDK_REQUIRES_NOT_EQUAL(pSendBuffer, pReturnBuffer);
    NN_SDK_REQUIRES_GREATER(auxBufferSize, static_cast<uint32_t>(0));
    NN_SDK_REQUIRES_LESS(auxBufferSize, static_cast<uint32_t>(std::numeric_limits<int32_t>::max()));

    auto pEffectManager = pConfig->_pEffectManager;
    NN_AUDIO_CHECK_IF_INITIALIZED(pEffectManager, pAux);

    NN_RESULT_DO(pEffectManager->Add(&pAux->_pEffectInfo, EffectType_Aux, nullptr, 0, pMixNode->_pMixInfo));
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pAux);

    memset(&pAux->_pEffectInfo->_inParameter._auxParameter, 0, sizeof(pAux->_pEffectInfo->_inParameter._auxParameter));
    AuxBufferInfo* sendBuffer = nullptr;
    AuxBufferInfo* returnBuffer = nullptr;
    auto sendBufferBase = InitializeAuxBufferInfo(&sendBuffer, pSendBuffer, static_cast<int32_t>(auxBufferSize));
    auto returnBufferBase = InitializeAuxBufferInfo(&returnBuffer, pReturnBuffer, static_cast<int32_t>(auxBufferSize));
    NN_SDK_REQUIRES_NOT_NULL(sendBuffer);
    NN_SDK_REQUIRES_NOT_NULL(returnBuffer);
    auto& param = pAux->_pEffectInfo->_inParameter._auxParameter;

    const size_t infoSize = nn::util::align_up(sizeof(nn::audio::AuxBufferInfo), nn::audio::BufferAlignSize);
    param._sampleCount     = 0; // filled at SetInputOutput
    param._sampleRate      = pMixNode->_pMixInfo->sampleRate;
    param._mixBufferCount  = pMixNode->_pMixInfo->bufferCount;
    param._mixBufferSampleCount = pConfig->_pMixManager->GetMixBufferSampleCount();

    param._sendAuxBufferInfo = reinterpret_cast<uintptr_t>(sendBuffer);
    param._sendBufferBase = reinterpret_cast<uintptr_t>(sendBufferBase);
    param._returnAuxBufferInfo = reinterpret_cast<uintptr_t>(returnBuffer);
    param._returnBufferBase = reinterpret_cast<uintptr_t>(returnBufferBase);

    param._channelCountMax = 0;
    param._frameBlockCountMax = static_cast<int32_t>((auxBufferSize - infoSize) / (pConfig->_pMixManager->GetMixBufferSampleCount() * sizeof(int32_t)));
    param._availableSampleCountMax = static_cast<uint32_t>((auxBufferSize - infoSize) / sizeof(int32_t)); // change "byte size" to "sample count"

    pAux->_pEffectInfo->_inParameter._enabled = true;

    NN_RESULT_SUCCESS;
}

Result AddAux(AudioRendererConfig* pConfig, AuxType* pAux, FinalMixType* pFinalMix, void* pSendBuffer, void* pReturnBuffer, const size_t auxBufferSize) NN_NOEXCEPT
{
    return AddAuxImpl(pConfig, pAux, pFinalMix, pSendBuffer, pReturnBuffer, auxBufferSize);
}

Result AddAux(AudioRendererConfig* pConfig, AuxType* pAux, SubMixType* pSubMix, void* pSendBuffer, void* pReturnBuffer, size_t auxBufferSize) NN_NOEXCEPT
{
    return AddAuxImpl(pConfig, pAux, pSubMix, pSendBuffer, pReturnBuffer, auxBufferSize);
}

template<typename T> void RemoveAuxImpl( AudioRendererConfig* pConfig, AuxType* pAux, T* pMixNode ) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pAux, EffectType_Aux);
    NN_SDK_REQUIRES_NOT_NULL(pMixNode);
    NN_SDK_REQUIRES_EQUAL(IsAuxRemovable(pAux), true);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pAux);
    NN_UNUSED(pMixNode);

    auto pEffectManager = pConfig->_pEffectManager;
    pEffectManager->Remove(pAux->_pEffectInfo, pMixNode->_pMixInfo);
    pAux->_pEffectInfo = nullptr;
}

void RemoveAux( AudioRendererConfig* pConfig, AuxType* pAux, FinalMixType* pFinalMix ) NN_NOEXCEPT
{
    RemoveAuxImpl( pConfig, pAux, pFinalMix );
}

void RemoveAux(AudioRendererConfig* pConfig, AuxType* pAux, SubMixType* pSubMix) NN_NOEXCEPT
{
    RemoveAuxImpl( pConfig, pAux, pSubMix );
}

int GetAuxChannelCountMax(const AuxType* pAux) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pAux);
    const auto& parameter = pAux->_pEffectInfo->_inParameter._auxParameter;
    // MEMO: channelCount は SetAuxInputOutput() で設定されたチャンネル数を指定する方に使われている
    return parameter._mixBufferCount;
}

bool IsAuxRemovable(const AuxType* pAux) NN_NOEXCEPT
{
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pAux);

    return IsRemovable(pAux, EffectType_Aux)
            && (IsAuxEnabled(pAux) == false);
}

void SetAuxEnabled( AuxType* pAux, bool enable ) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pAux, EffectType_Aux);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pAux);
    pAux->_pEffectInfo->_inParameter._enabled = enable;
    auto sendInfo = reinterpret_cast<AuxBufferInfo*>(pAux->_pEffectInfo->_inParameter._auxParameter._sendAuxBufferInfo);
    auto returnInfo = reinterpret_cast<AuxBufferInfo*>(pAux->_pEffectInfo->_inParameter._auxParameter._returnAuxBufferInfo);
    ResetAuxBuffer(sendInfo);
    ResetAuxBuffer(returnInfo);
    nn::os::FlushDataCache(returnInfo, sizeof(AuxBufferInfo));
}

bool IsAuxEnabled( const AuxType* pAux ) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pAux, EffectType_Aux);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pAux);
    return pAux->_pEffectInfo->_inParameter._enabled;
}

void GetAuxInputOutput(const AuxType* pAux, int8_t* pOutInput, int8_t* pOutOutput, int* pOutCount, int count) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pAux, EffectType_Aux);
    NN_SDK_REQUIRES_NOT_NULL(pOutInput);
    NN_SDK_REQUIRES_NOT_NULL(pOutOutput);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_MINMAX(count, 0, MixBufferCountMax);

    auto c = std::min(count,pAux->_pEffectInfo->_inParameter._auxParameter._channelCountMax);
    for (int i = 0; i < c; ++i)
    {
        pOutInput[i] = pAux->_pEffectInfo->_inParameter._auxParameter._input[i];
        pOutOutput[i] = pAux->_pEffectInfo->_inParameter._auxParameter._output[i];
    }
    *pOutCount = c;
}

void SetAuxInputOutput( AuxType* pAux, const int8_t* input, const int8_t* output, const int count ) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pAux, EffectType_Aux);
    NN_SDK_REQUIRES_NOT_NULL(input);
    NN_SDK_REQUIRES_NOT_NULL(output);
    NN_SDK_REQUIRES_MINMAX( count, 0, pAux->_pEffectInfo->_inParameter._auxParameter._mixBufferCount);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pAux);

    auto& p = pAux->_pEffectInfo->_inParameter._auxParameter;
    auto c = std::min(count, pAux->_pEffectInfo->_inParameter._auxParameter._mixBufferCount);
    for (int i = 0; i < c; ++i)
    {
        p._input[i] = input[i];
        p._output[i] = output[i];
    }
    p._channelCountMax = c;
    auto frameCount = p._frameBlockCountMax / c;
    p._sampleCount = frameCount * c * p._mixBufferSampleCount;
}

int ReadAuxSendBuffer( AuxType* pAux, int32_t* pOutData, int count ) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pAux, EffectType_Aux);
    NN_SDK_REQUIRES_NOT_NULL( pOutData );
    NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);

    AuxParameter& param = pAux->_pEffectInfo->_inParameter._auxParameter;
    AuxBufferInfo *sendBuffer = reinterpret_cast<AuxBufferInfo*>(param._sendAuxBufferInfo);
    auto sendCircular = reinterpret_cast<int32_t*>(param._sendBufferBase);
    return ReadAuxBuffer( sendBuffer, sendCircular, param._sampleCount, pOutData, count );
}

int WriteAuxReturnBuffer( AuxType* pAux, const int32_t* pData, int count ) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pAux, EffectType_Aux);
    NN_SDK_REQUIRES_NOT_NULL( pData );
    NN_SDK_REQUIRES_GREATER_EQUAL(count, 0);

    AuxParameter& param = pAux->_pEffectInfo->_inParameter._auxParameter;
    AuxBufferInfo* returnBuffer = reinterpret_cast<AuxBufferInfo*>(param._returnAuxBufferInfo);
    auto returnCircular = reinterpret_cast<int32_t*>(param._returnBufferBase);
    auto countMax = static_cast<int32_t>(param._sampleCount);
    return WriteAuxBuffer( returnBuffer, returnCircular, countMax, pData, count );
}

int GetAuxSampleRate( const AuxType* pAux ) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pAux, EffectType_Aux);
    return pAux->_pEffectInfo->_inParameter._auxParameter._sampleRate;
}

int GetAuxSampleCount( const AuxType* pAux ) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pAux, EffectType_Aux);
    return pAux->_pEffectInfo->_inParameter._auxParameter._availableSampleCountMax;
}

/////////////////////////////////////////////////////////////////////////////
// Reverb
/////////////////////////////////////////////////////////////////////////////
size_t GetRequiredBufferSizeForReverb(int sampleRate, int channelCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES( (sampleRate == 32000) || (sampleRate == 48000) );
    NN_SDK_REQUIRES( (channelCount == 1) || (channelCount == 2) || (channelCount == 4) || (channelCount == 6) );

    qf fs = sampleRate / 1000 * ( 1 << QF_FRACTIONAL_BIT_COUNT ); // [kHz]
    size_t delayLineSize = nn::util::align_up(ReverbGetRequiredDelayLineBufferSize(fs, channelCount), nn::audio::BufferAlignSize);
    return delayLineSize;
}

template<typename T> Result AddReverbImpl(AudioRendererConfig* pConfig, ReverbType* pOutReverb, void* buffer, size_t bufferSize, T* pMixNode, int channelCountMax) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pOutReverb);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_ASSERT_ALIGNED(reinterpret_cast<uintptr_t>(buffer), nn::audio::BufferAlignSize);
    NN_SDK_ASSERT_GREATER_EQUAL(bufferSize, GetRequiredBufferSizeForReverb(pMixNode->_pMixInfo->sampleRate, channelCountMax));
    NN_SDK_REQUIRES_NOT_NULL(pMixNode);
    NN_SDK_REQUIRES(channelCountMax == 1 || channelCountMax == 2 || channelCountMax == 4 || channelCountMax == 6);

    //TODO: 事前条件チェック追加

    auto pEffectManager = pConfig->_pEffectManager;
    NN_AUDIO_CHECK_IF_INITIALIZED(pEffectManager, pOutReverb);

    NN_RESULT_DO(pEffectManager->Add(&pOutReverb->_pEffectInfo, EffectType_Reverb, buffer, bufferSize, pMixNode->_pMixInfo));
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pOutReverb);

    memset(&pOutReverb->_pEffectInfo->_inParameter._reverbParameter, 0, sizeof(ReverbParameter));
    pOutReverb->_pEffectInfo->_inParameter._reverbParameter._coloration = 0;
    pOutReverb->_pEffectInfo->_inParameter._reverbParameter._decayTime = qf(0.1f * (1 << QF_FRACTIONAL_BIT_COUNT));
    pOutReverb->_pEffectInfo->_inParameter._reverbParameter._earlyGain = 0;
    pOutReverb->_pEffectInfo->_inParameter._reverbParameter._dryGain = qf(1.0f * (1 << QF_FRACTIONAL_BIT_COUNT));
    pOutReverb->_pEffectInfo->_inParameter._reverbParameter._earlyMode = ConvertToDspType(ReverbType::EarlyMode_NoEarlyReflection);
    pOutReverb->_pEffectInfo->_inParameter._reverbParameter._lateMode = ConvertToDspType(ReverbType::LateMode_Room);
    pOutReverb->_pEffectInfo->_inParameter._reverbParameter._highFrequencyDecayRatio = qf(0.1f * (1 << QF_FRACTIONAL_BIT_COUNT));
    pOutReverb->_pEffectInfo->_inParameter._reverbParameter._numChannelCountMax = static_cast<uint16_t>(channelCountMax);
    pOutReverb->_pEffectInfo->_inParameter._reverbParameter._numChannels = static_cast<uint16_t>(channelCountMax);
    pOutReverb->_pEffectInfo->_inParameter._reverbParameter._sampleRate = pMixNode->_pMixInfo->sampleRate / 1000 * (1 << QF_FRACTIONAL_BIT_COUNT); // [kHz]
    pOutReverb->_pEffectInfo->_inParameter._enabled = true;

    NN_RESULT_SUCCESS;
}

Result AddReverb(AudioRendererConfig* pConfig, ReverbType* pOutReverb, void* buffer, size_t bufferSize, FinalMixType* pFinalMix, int channelCountMax) NN_NOEXCEPT
{
    return AddReverbImpl(pConfig, pOutReverb, buffer, bufferSize, pFinalMix, channelCountMax);
}

Result AddReverb(AudioRendererConfig* pConfig, ReverbType* pOutReverb, void* buffer, size_t bufferSize, SubMixType* pSubMix, int channelCountMax) NN_NOEXCEPT
{
    return AddReverbImpl(pConfig, pOutReverb, buffer, bufferSize, pSubMix, channelCountMax);
}

template<typename T>void* RemoveReverbImpl(AudioRendererConfig* pConfig, ReverbType* pReverb, T* pMixNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL( pConfig );
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_Reverb);
    NN_SDK_REQUIRES_NOT_NULL( pMixNode );
    NN_SDK_REQUIRES_EQUAL(IsReverbRemovable(pReverb), true);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pReverb);
    NN_UNUSED(pMixNode);

    auto pEffectManager = pConfig->_pEffectManager;
    void* returnAddress = pEffectManager->Remove(pReverb->_pEffectInfo, pMixNode->_pMixInfo);
    pReverb->_pEffectInfo = nullptr;
    return returnAddress;
}

void* RemoveReverb(AudioRendererConfig* pConfig, ReverbType* pReverb, FinalMixType* pFinalMix) NN_NOEXCEPT
{
    return RemoveReverbImpl(pConfig, pReverb, pFinalMix);
}

void* RemoveReverb(AudioRendererConfig* pConfig, ReverbType* pReverb, SubMixType* pSubMix) NN_NOEXCEPT
{
    return RemoveReverbImpl(pConfig, pReverb, pSubMix);
}

int GetReverbChannelCountMax(const ReverbType* pReverb) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pReverb);
    const auto& parameter = pReverb->_pEffectInfo->_inParameter._reverbParameter;
    return parameter._numChannelCountMax;
}

bool IsReverbRemovable(const ReverbType* pReverb) NN_NOEXCEPT
{
    return IsRemovable(pReverb, EffectType_Reverb)
            && (IsReverbEnabled(pReverb) == false);
}

void GetReverbInputOutput(const ReverbType* pReverb, int8_t* outInput, int8_t* outOutput, int* pOutCount, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pReverb);
    NN_SDK_REQUIRES_NOT_NULL(outInput);
    NN_SDK_REQUIRES_NOT_NULL(outOutput);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);

    auto& p = pReverb->_pEffectInfo->_inParameter._reverbParameter;
    auto c = std::min(static_cast<uint16_t>(count), p._numChannels);
    for (int i = 0; i < c; ++i)
    {
        outInput[i] = p._input[i];
        outOutput[i] = p._output[i];
    }
    *pOutCount = p._numChannels;
}

void SetReverbInputOutput(ReverbType* pReverb, const int8_t* input, const int8_t* output, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pReverb);
    NN_SDK_REQUIRES_NOT_NULL(input);
    NN_SDK_REQUIRES_NOT_NULL(output);
    NN_SDK_REQUIRES( (count == 1) || (count == 2) || (count == 4) || (count == 6) );
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pReverb);

    auto& p = pReverb->_pEffectInfo->_inParameter._reverbParameter;
    p._parameterStatus = EffectParameterStatus_Init;
    auto c = std::min(count,
                      std::min(static_cast<int>(pReverb->_pEffectInfo->_inParameter._reverbParameter._numChannelCountMax),
                               nn::audio::ReverbParameter::SupportedChannelCountMax));
    for (int i = 0; i < c; ++i)
    {
        p._input[i] = input[i];
        p._output[i] = output[i];
    }
    p._numChannels = static_cast<uint16_t>(c);
}

template<typename Field>
void SetQfConfig(ReverbType* pReverb, Field f, float value, float rangeFrom, float rangeTo) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_Reverb);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pReverb);
    ReverbParameter* p = &pReverb->_pEffectInfo->_inParameter._reverbParameter;
    if ( p->_parameterStatus != EffectParameterStatus_Init )
    {
         p->_parameterStatus = EffectParameterStatus_UpdateParam;
    }
    SetQfConfig(p, f, value, rangeFrom, rangeTo);
}

template<typename Field>
float GetQfConfig(const ReverbType* pReverb, Field f) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_Reverb);
    ReverbParameter* p = &pReverb->_pEffectInfo->_inParameter._reverbParameter;
    return GetQfConfig(p, f);
}

template<typename Field, typename ValueType>
void SetConfig(ReverbType* pReverb, Field f, ValueType value, ValueType rangeFrom, ValueType rangeTo) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_Reverb);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pReverb);
    ReverbParameter* p = &pReverb->_pEffectInfo->_inParameter._reverbParameter;
    if ( p->_parameterStatus != EffectParameterStatus_Init )
    {
         p->_parameterStatus = EffectParameterStatus_UpdateParam;
    }
    SetConfig(p, f, value, rangeFrom, rangeTo);
}

template<typename Field, typename ValueType>
ValueType GetConfig(const ReverbType* pReverb, Field f) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_Reverb);
    ReverbParameter* p = &pReverb->_pEffectInfo->_inParameter._reverbParameter;
    return GetConfig(p, f);
}

void SetReverbReverbGain(ReverbType* pReverb, float gain) NN_NOEXCEPT
{
    SetQfConfig(pReverb, &ReverbParameter::_reverbGain, gain, 0.0f, 1.0f);
}

float GetReverbReverbGain(const ReverbType* pReverb) NN_NOEXCEPT
{
    return GetQfConfig(pReverb, &ReverbParameter::_reverbGain);
}

void SetReverbPredelayTime(ReverbType* pReverb, nn::TimeSpan time) NN_NOEXCEPT
{
    // _predelayTime need to be passed in [MilliSeconds]
    SetQfConfig(pReverb, &ReverbParameter::_predelayTime, static_cast<float>(time.GetMicroSeconds()) / 1000.f, 0.0f, 300.0f);
}

nn::TimeSpan GetReverbPredelayTime(const ReverbType* pReverb) NN_NOEXCEPT
{
    float milliSeconds = GetQfConfig(pReverb, &ReverbParameter::_predelayTime);
    return nn::TimeSpan::FromMicroSeconds(static_cast<int64_t>(milliSeconds * 1000));
}

void SetReverbDecayTime(ReverbType* pReverb, nn::TimeSpan time) NN_NOEXCEPT
{
    // _decayTime need to be passed in [Seconds]
    SetQfConfig(pReverb, &ReverbParameter::_decayTime, static_cast<float>(time.GetMilliSeconds()) / 1000.f, 0.1f, 20.0f);
}

nn::TimeSpan GetReverbDecayTime(const ReverbType* pReverb) NN_NOEXCEPT
{
    float seconds = GetQfConfig(pReverb, &ReverbParameter::_decayTime);
    return nn::TimeSpan::FromMilliSeconds(static_cast<int64_t>(seconds * 1000));
}

void SetReverbColoration(ReverbType* pReverb, float value) NN_NOEXCEPT
{
    SetQfConfig(pReverb, &ReverbParameter::_coloration, value, 0.0f, 1.0f);
}

float GetReverbColoration(const ReverbType* pReverb) NN_NOEXCEPT
{
    return GetQfConfig(pReverb, &ReverbParameter::_coloration);
}

void SetReverbEarlyGain(ReverbType* pReverb, float gain) NN_NOEXCEPT
{
    SetQfConfig(pReverb, &ReverbParameter::_earlyGain, gain, 0.0f, 1.0f);
}

float GetReverbEarlyGain(const ReverbType* pReverb) NN_NOEXCEPT
{
    return GetQfConfig(pReverb, &ReverbParameter::_earlyGain);
}

void SetReverbLateGain(ReverbType* pReverb, float gain) NN_NOEXCEPT
{
    SetQfConfig(pReverb, &ReverbParameter::_lateGain, gain, 0.0f, 1.0f);
}

float GetReverbLateGain(const ReverbType* pReverb) NN_NOEXCEPT
{
    return GetQfConfig(pReverb, &ReverbParameter::_lateGain);
}

void SetReverbOutGain(ReverbType* pReverb, float gain) NN_NOEXCEPT
{
    SetQfConfig(pReverb, &ReverbParameter::_outGain, gain, 0.0f, 1.0f);
}

float GetReverbOutGain(const ReverbType* pReverb) NN_NOEXCEPT
{
    return GetQfConfig(pReverb, &ReverbParameter::_outGain);
}

void SetReverbDryGain(ReverbType* pReverb, float gain) NN_NOEXCEPT
{
    SetQfConfig(pReverb, &ReverbParameter::_dryGain, gain, 0.0f, 1.0f);
}

float GetReverbDryGain(const ReverbType* pReverb) NN_NOEXCEPT
{
    return GetQfConfig(pReverb, &ReverbParameter::_dryGain);
}

void SetReverbHighFrequencyDecayRatio(ReverbType* pReverb, float ratio) NN_NOEXCEPT
{
    SetQfConfig(pReverb, &ReverbParameter::_highFrequencyDecayRatio, ratio, 0.1f, 1.0f);
}

float GetReverbHighFrequencyDecayRatio(const ReverbType* pReverb) NN_NOEXCEPT
{
    return GetQfConfig(pReverb, &ReverbParameter::_highFrequencyDecayRatio);
}

void SetReverbEarlyMode(ReverbType* pReverb, ReverbType::EarlyMode mode) NN_NOEXCEPT
{
    SetConfig(pReverb, &ReverbParameter::_earlyMode, ConvertToDspType(mode), ConvertToDspType(ReverbType::EarlyMode_SmallRoom), ConvertToDspType(ReverbType::EarlyMode_NoEarlyReflection));
}

ReverbType::EarlyMode GetReverbEarlyMode(const ReverbType* pReverb) NN_NOEXCEPT
{
    // TODO: GetConfig() 利用
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_Reverb);
    return ConvertFromDspType(pReverb->_pEffectInfo->_inParameter._reverbParameter._earlyMode);
}

void SetReverbLateMode(ReverbType* pReverb, ReverbType::LateMode mode) NN_NOEXCEPT
{
    SetConfig(pReverb, &ReverbParameter::_lateMode, ConvertToDspType(mode), ConvertToDspType(ReverbType::LateMode_Room), ConvertToDspType(ReverbType::LateMode_MaximumDelay));
}

ReverbType::LateMode GetReverbLateMode(const ReverbType* pReverb) NN_NOEXCEPT
{
    // TODO: GetConfig() 利用
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_Reverb);
    return ConvertFromDspType(pReverb->_pEffectInfo->_inParameter._reverbParameter._lateMode);
}

void SetReverbParameters(ReverbType* pOutReverb, const ReverbParameterSet* pParameterSet) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pOutReverb, EffectType_Reverb);

    ReverbParameter* p = &pOutReverb->_pEffectInfo->_inParameter._reverbParameter;
    if (p->_parameterStatus != EffectParameterStatus_Init)
    {
        p->_parameterStatus = EffectParameterStatus_UpdateParam;
    }
    NN_SDK_REQUIRES_NOT_NULL(p);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pOutReverb);

    NN_SDK_REQUIRES_MINMAX(pParameterSet->earlyGain, 0.0f, 1.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->lateGain, 0.0f, 1.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->decayTimeSeconds, 0.1f, 20.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->highFreqDecayRatio, 0.1f, 1.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->coloration, 0.0f, 1.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->reverbGain, 0.0f, 1.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->outGain, 0.0f, 1.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->dryGain, 0.0f, 1.0f);

    p->_earlyMode = ConvertToDspType(pParameterSet->earlyMode);
    p->_earlyGain = qf(pParameterSet->earlyGain * ( 1 << QF_FRACTIONAL_BIT_COUNT));
    p->_predelayTime = qf(pParameterSet->predelayTimeMilliSeconds * ( 1 << QF_FRACTIONAL_BIT_COUNT));;
    p->_lateMode = ConvertToDspType(pParameterSet->lateMode);
    p->_lateGain = qf(pParameterSet->lateGain * ( 1 << QF_FRACTIONAL_BIT_COUNT));
    p->_decayTime = qf(pParameterSet->decayTimeSeconds * ( 1 << QF_FRACTIONAL_BIT_COUNT)); //seconds conversion
    p->_highFrequencyDecayRatio = qf(pParameterSet->highFreqDecayRatio * ( 1 << QF_FRACTIONAL_BIT_COUNT));
    p->_coloration = qf(pParameterSet->coloration * ( 1 << QF_FRACTIONAL_BIT_COUNT));
    p->_reverbGain = qf(pParameterSet->reverbGain * ( 1 << QF_FRACTIONAL_BIT_COUNT));
    p->_outGain = qf(pParameterSet->outGain * ( 1 << QF_FRACTIONAL_BIT_COUNT));
    p->_dryGain = qf(pParameterSet->dryGain * ( 1 << QF_FRACTIONAL_BIT_COUNT));
}

ReverbParameterSet GetReverbParameters(const ReverbType* pReverb) NN_NOEXCEPT
{
    ReverbParameterSet outParameterSet;
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_Reverb);
    ReverbParameter* pReverbParameter = &pReverb->_pEffectInfo->_inParameter._reverbParameter;
    NN_SDK_REQUIRES_NOT_NULL(pReverbParameter);

    outParameterSet.earlyMode = ConvertFromDspType(pReverbParameter->_earlyMode);
    outParameterSet.earlyGain = static_cast<float>(pReverbParameter->_earlyGain) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    outParameterSet.predelayTimeMilliSeconds = static_cast<float>(pReverbParameter->_predelayTime) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    outParameterSet.lateMode = ConvertFromDspType(pReverbParameter->_lateMode);
    outParameterSet.lateGain = static_cast<float>(pReverbParameter->_lateGain) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    outParameterSet.decayTimeSeconds = static_cast<float>(pReverbParameter->_decayTime) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    outParameterSet.highFreqDecayRatio = static_cast<float>(pReverbParameter->_highFrequencyDecayRatio) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    outParameterSet.coloration = static_cast<float>(pReverbParameter->_coloration) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    outParameterSet.reverbGain = static_cast<float>(pReverbParameter->_reverbGain) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    outParameterSet.outGain = static_cast<float>(pReverbParameter->_outGain) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    outParameterSet.dryGain = static_cast<float>(pReverbParameter->_dryGain) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);

    return outParameterSet;
}

void SetReverbEnabled(ReverbType* pReverb, bool enable) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_Reverb);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pReverb);
    pReverb->_pEffectInfo->_inParameter._enabled = enable;
    pReverb->_pEffectInfo->_inParameter._reverbParameter._parameterStatus = enable ? EffectParameterStatus_Init : EffectParameterStatus_UpdateParam;
}

bool IsReverbEnabled(const ReverbType* pReverb) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_Reverb);
    return pReverb->_pEffectInfo->_inParameter._enabled;
}

size_t GetRequiredBufferSizeForI3dl2Reverb(int sampleRate, int channelCountMax) NN_NOEXCEPT
{
    NN_SDK_REQUIRES((sampleRate == 32000) || (sampleRate == 48000));
    NN_SDK_REQUIRES((channelCountMax == 1) || (channelCountMax == 2) || (channelCountMax == 4) || (channelCountMax == 6));

    size_t delayLineSize = nn::util::align_up(I3DL2ReverbGetRequiredDelayLineBufferSize(sampleRate, channelCountMax), nn::audio::BufferAlignSize);
    return delayLineSize;
}

template<typename T> Result AddI3dl2ReverbImpl(AudioRendererConfig* pConfig, I3dl2ReverbType* pOutReverb, void* buffer, size_t bufferSize, T* pMixNode, int channelCountMax) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_SDK_REQUIRES_NOT_NULL(pOutReverb);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_ASSERT(nn::util::is_aligned(reinterpret_cast<uintptr_t>(buffer), nn::audio::BufferAlignSize));
    NN_SDK_ASSERT_GREATER_EQUAL(bufferSize, GetRequiredBufferSizeForI3dl2Reverb(pMixNode->_pMixInfo->sampleRate, channelCountMax));
    NN_SDK_REQUIRES_NOT_NULL(pMixNode);
    NN_SDK_REQUIRES(channelCountMax == 1 || channelCountMax == 2 || channelCountMax == 4 || channelCountMax == 6);
    //TODO: 事前条件チェック追加

    auto pEffectManager = pConfig->_pEffectManager;
    NN_AUDIO_CHECK_IF_INITIALIZED(pEffectManager, pOutReverb);

    NN_RESULT_DO(pEffectManager->Add(&pOutReverb->_pEffectInfo, EffectType_I3dl2Reverb, buffer, bufferSize, pMixNode->_pMixInfo));
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pOutReverb);

    memset(&pOutReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter, 0, sizeof(I3dl2ReverbParameter));
    pOutReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter._sampleRate = pMixNode->_pMixInfo->sampleRate; // [Hz]

    I3dl2ReverbParameterSet i3dl2Parameter;
    LoadI3dl2ReverbPreset(&i3dl2Parameter, I3dl2ReverbType::Preset_SmallRoom);
    SetI3dl2ReverbParameters(pOutReverb, &i3dl2Parameter);

    pOutReverb->_pEffectInfo->_inParameter._enabled = true;
    pOutReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter._numChannels = static_cast<uint16_t>(channelCountMax);
    pOutReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter._numChannelCountMax = static_cast<uint16_t>(channelCountMax);

    NN_RESULT_SUCCESS;
}

Result AddI3dl2Reverb(AudioRendererConfig* pConfig, I3dl2ReverbType* pOutReverb, void* buffer, size_t bufferSize, FinalMixType* pFinalMix, int channelCountMax) NN_NOEXCEPT
{
    return AddI3dl2ReverbImpl(pConfig, pOutReverb, buffer, bufferSize, pFinalMix, channelCountMax);
}

Result AddI3dl2Reverb(AudioRendererConfig* pConfig, I3dl2ReverbType* pOutReverb, void* buffer, size_t bufferSize, SubMixType* pSubMix, int channelCountMax) NN_NOEXCEPT
{
    return AddI3dl2ReverbImpl(pConfig, pOutReverb, buffer, bufferSize, pSubMix, channelCountMax);
}

template<typename T>void* RemoveI3dl2ReverbImpl(AudioRendererConfig* pConfig, I3dl2ReverbType* pReverb, T* pMixNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pConfig);
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_I3dl2Reverb);
    NN_SDK_REQUIRES_NOT_NULL(pMixNode);
    NN_SDK_REQUIRES_EQUAL(IsI3dl2ReverbRemovable(pReverb), true);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pReverb);
    NN_UNUSED(pMixNode);

    auto pEffectManager = pConfig->_pEffectManager;
    void* returnAddress = pEffectManager->Remove(pReverb->_pEffectInfo, pMixNode->_pMixInfo);
    pReverb->_pEffectInfo = nullptr;
    return returnAddress;
}

void* RemoveI3dl2Reverb(AudioRendererConfig* pConfig, I3dl2ReverbType* pReverb, FinalMixType* pFinalMix) NN_NOEXCEPT
{
    return RemoveI3dl2ReverbImpl(pConfig, pReverb, pFinalMix);
}

void* RemoveI3dl2Reverb(AudioRendererConfig* pConfig, I3dl2ReverbType* pReverb, SubMixType* pSubMix) NN_NOEXCEPT
{
    return RemoveI3dl2ReverbImpl(pConfig, pReverb, pSubMix);
}

int GetI3dl2ReverbChannelCountMax(const I3dl2ReverbType* pReverb) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pReverb);
    const auto& parameter = pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter;
    return parameter._numChannelCountMax;
}

bool IsI3dl2ReverbRemovable(const I3dl2ReverbType* pReverb) NN_NOEXCEPT
{
    return IsRemovable(pReverb, EffectType_I3dl2Reverb)
        && (IsI3dl2ReverbEnabled(pReverb) == false);
}

void GetI3dl2ReverbInputOutput(const I3dl2ReverbType* pReverb, int8_t* outInput, int8_t* outOutput, int* pOutCount, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pReverb);
    NN_SDK_REQUIRES_NOT_NULL(outInput);
    NN_SDK_REQUIRES_NOT_NULL(outOutput);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);

    auto& p = pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter;
    auto c = std::min(static_cast<uint16_t>(count), p._numChannels);
    for (int i = 0; i < c; ++i)
    {
        outInput[i] = p._input[i];
        outOutput[i] = p._output[i];
    }
    *pOutCount = p._numChannels;
}

void SetI3dl2ReverbInputOutput(I3dl2ReverbType* pReverb, const int8_t* input, const int8_t* output, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pReverb);
    NN_SDK_REQUIRES_NOT_NULL(input);
    NN_SDK_REQUIRES_NOT_NULL(output);
    NN_SDK_REQUIRES((count == 1) || (count == 2) || (count == 4) || (count == 6));
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pReverb);

    auto& p = pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter;
    p._parameterStatus = EffectParameterStatus_Init;

    auto c = std::min(count,
        std::min(static_cast<int>(pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter._numChannelCountMax),
            nn::audio::I3dl2ReverbParameter::SupportedChannelCountMax));
    for (int i = 0; i < c; ++i)
    {
        p._input[i] = input[i];
        p._output[i] = output[i];
    }
    p._numChannels = static_cast<uint16_t>(c);
}

template<typename Field, typename ValueType>
void SetConfig(I3dl2ReverbType* pReverb, Field f, ValueType value, ValueType rangeFrom, ValueType rangeTo) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_I3dl2Reverb);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pReverb);
    I3dl2ReverbParameter* p = &pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter;
    if (p->_parameterStatus != EffectParameterStatus_Init)
    {
        p->_parameterStatus = EffectParameterStatus_UpdateParam;
    }
    SetConfig(p, f, value, rangeFrom, rangeTo);
}

void SetI3dl2ReverbReverbGain(I3dl2ReverbType* pReverb, float reverb) NN_NOEXCEPT
{
    SetConfig(pReverb, &I3dl2ReverbParameter::_reverb, reverb, -10000.0f, 2000.0f);
}
float GetI3dl2ReverbReverbGain(const I3dl2ReverbType* pReverb) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_I3dl2Reverb);
    I3dl2ReverbParameter* p = &pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter;
    NN_SDK_REQUIRES_NOT_NULL(p);
    return p->_reverb;
}

void SetI3dl2ReverbReflectionsGain(I3dl2ReverbType* pReverb, float reflections) NN_NOEXCEPT
{
    SetConfig(pReverb, &I3dl2ReverbParameter::_reflections, reflections, -10000.0f, 1000.0f);
}

float GetI3dl2ReverbReflectionsGain(const I3dl2ReverbType* pReverb) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_I3dl2Reverb);
    I3dl2ReverbParameter* p = &pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter;
    NN_SDK_REQUIRES_NOT_NULL(p);
    return p->_reflections;
}

void SetI3dl2ReverbRoomGain(I3dl2ReverbType* pReverb, float room) NN_NOEXCEPT
{
    SetConfig(pReverb, &I3dl2ReverbParameter::_room, room, -10000.0f, 0.0f);
}

float GetI3dl2ReverbRoomGain(const I3dl2ReverbType* pReverb) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_I3dl2Reverb);
    I3dl2ReverbParameter* p = &pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter;
    NN_SDK_REQUIRES_NOT_NULL(p);
    return p->_room;
}

void SetI3dl2ReverbReflectionDelayTime(I3dl2ReverbType* pReverb, nn::TimeSpan time) NN_NOEXCEPT
{
    SetConfig(pReverb, &I3dl2ReverbParameter::_reflectionsDelay, static_cast<float>(time.GetMilliSeconds()) / 1000.0f, 0.0f, 0.3f);
}

nn::TimeSpan GetI3dl2ReverbReflectionDelayTime(const I3dl2ReverbType* pReverb) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_I3dl2Reverb);
    I3dl2ReverbParameter* p = &pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter;
    NN_SDK_REQUIRES_NOT_NULL(p);
    float Seconds = p->_reflectionsDelay;
    return nn::TimeSpan::FromMilliSeconds(static_cast<int64_t>(1000 * Seconds));
}

void SetI3dl2ReverbLateReverbDelayTime(I3dl2ReverbType* pReverb, nn::TimeSpan time) NN_NOEXCEPT
{
    SetConfig(pReverb, &I3dl2ReverbParameter::_reverbDelay, static_cast<float>(time.GetMilliSeconds()) / 1000.0f, 0.0f, 0.1f);
}

nn::TimeSpan GetI3dl2ReverbLateReverbDelayTime(const I3dl2ReverbType* pReverb) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_I3dl2Reverb);
    I3dl2ReverbParameter* p = &pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter;
    NN_SDK_REQUIRES_NOT_NULL(p);
    float Seconds = p->_reverbDelay;
    return nn::TimeSpan::FromMilliSeconds(static_cast<int64_t>(1000 * Seconds));
}

void SetI3dl2ReverbRoomHfGain(I3dl2ReverbType* pReverb, float roomHf) NN_NOEXCEPT
{
    SetConfig(pReverb, &I3dl2ReverbParameter::_roomHf, roomHf, -10000.0f, 0.0f);
}

float GetI3dl2ReverbRoomHfGain(const I3dl2ReverbType* pReverb) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_I3dl2Reverb);
    I3dl2ReverbParameter* p = &pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter;
    NN_SDK_REQUIRES_NOT_NULL(p);
    return p->_roomHf;
}

void SetI3dl2ReverbHfReference(I3dl2ReverbType* pReverb, float hfReference) NN_NOEXCEPT
{
    SetConfig(pReverb, &I3dl2ReverbParameter::_hfReference, hfReference, 20.0f, 20000.0f);
}

float GetI3dl2ReverbHfReference(const I3dl2ReverbType* pReverb) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_I3dl2Reverb);
    I3dl2ReverbParameter* p = &pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter;
    NN_SDK_REQUIRES_NOT_NULL(p);
    return p->_hfReference;
}

void SetI3dl2ReverbLateReverbDecayTime(I3dl2ReverbType* pReverb, nn::TimeSpan time) NN_NOEXCEPT
{
    SetConfig(pReverb, &I3dl2ReverbParameter::_decayTime, static_cast<float>(time.GetMilliSeconds()) / 1000.0f, 0.1f, 20.0f);
}

nn::TimeSpan GetI3dl2ReverbLateReverbDecayTime(const I3dl2ReverbType* pReverb) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_I3dl2Reverb);
    I3dl2ReverbParameter* p = &pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter;
    NN_SDK_REQUIRES_NOT_NULL(p);
    float Seconds = p->_decayTime;
    return nn::TimeSpan::FromMilliSeconds(static_cast<int64_t>(1000 * Seconds));
}

void SetI3dl2ReverbLateReverbHfDecayRatio(I3dl2ReverbType* pReverb, float hfDecayRatio) NN_NOEXCEPT
{
    SetConfig(pReverb, &I3dl2ReverbParameter::_hfDecayRatio, hfDecayRatio, 0.1f, 2.0f);
}

float GetI3dl2ReverbLateReverbHfDecayRatio(const I3dl2ReverbType* pReverb) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_I3dl2Reverb);
    I3dl2ReverbParameter* p = &pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter;
    NN_SDK_REQUIRES_NOT_NULL(p);
    return p->_hfDecayRatio;
}

void SetI3dl2ReverbLateReverbDiffusion(I3dl2ReverbType* pReverb, float diffusion) NN_NOEXCEPT
{
    SetConfig(pReverb, &I3dl2ReverbParameter::_diffusion, diffusion, 0.0f, 100.0f);
}

float GetI3dl2ReverbLateReverbDiffusion(const I3dl2ReverbType* pReverb) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_I3dl2Reverb);
    I3dl2ReverbParameter* p = &pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter;
    NN_SDK_REQUIRES_NOT_NULL(p);
    return p->_diffusion;
}

void SetI3dl2ReverbLateReverbDensity(I3dl2ReverbType* pReverb, float density) NN_NOEXCEPT
{
    SetConfig(pReverb, &I3dl2ReverbParameter::_density, density, 0.0f, 100.0f);
}

float GetI3dl2ReverbLateReverbDensity(const I3dl2ReverbType* pReverb) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_I3dl2Reverb);
    I3dl2ReverbParameter* p = &pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter;
    NN_SDK_REQUIRES_NOT_NULL(p);
    return p->_density;
}

void SetI3dl2ReverbDryGain(I3dl2ReverbType* pReverb, float dryGain) NN_NOEXCEPT
{
    SetConfig(pReverb, &I3dl2ReverbParameter::_dryGain, dryGain, 0.0f, 1.0f);
}

float GetI3dl2ReverbDryGain(const I3dl2ReverbType* pReverb) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_I3dl2Reverb);
    I3dl2ReverbParameter* p = &pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter;
    NN_SDK_REQUIRES_NOT_NULL(p);
    return p->_dryGain;
}

NN_DEPRECATED void LoadI3dl2ReverbPreset(I3dl2ReverbType* pReverb, I3dl2ReverbType::Preset preset) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_I3dl2Reverb);
    I3dl2ReverbParameter* p = &pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter;
    NN_SDK_REQUIRES_NOT_NULL(p);
    NN_SDK_REQUIRES_NOT_NULL(p);
    NN_SDK_REQUIRES_MINMAX(preset, I3dl2ReverbType::Preset_SmallRoom, I3dl2ReverbType::Preset_MetalCorridor);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pReverb);

    switch (preset)
    {
    case I3dl2ReverbType::Preset_SmallRoom:
        p->_room = -1045.f; //in milliBels
        p->_reflections = -310.f; //in milliBels
        p->_reverb = -602.f;  //in milliBels
        p->_diffusion = 100.f;
        p->_reflectionsDelay = 0.0046f; //in sec
        p->_reverbDelay = 0.012f;         //in sec
        p->_density = 61.0f;
        p->_decayTime = 1.0f;          // seconds 0.1 to 20.0
        p->_hfDecayRatio = 0.2f;      // 0.1 to 1.0
        p->_dryGain = 0.24f; // 0.4;            // 0.0 to 1.0
        p->_roomHf = -1000.f; // 0.0;
        p->_hfReference = 5000.0f;
        break;

    case I3dl2ReverbType::Preset_LargeRoom:
        p->_room = -1045.f; //in milliBels
        p->_reflections = -310.f; //in milliBels
        p->_reverb = -310.f;  //in milliBels
        p->_diffusion = 100.f;
        p->_reflectionsDelay = 0.0092f; //in sec
        p->_reverbDelay = 0.040f;         //in sec
        p->_density = 61.0f;
        p->_decayTime = 1.7f;          // seconds 0.1 to 20.0
        p->_hfDecayRatio = 0.3f;      // 0.1 to 1.0
        p->_dryGain = 0.24f; // 0.4;            // 0.0 to 1.0
        p->_roomHf = -1000.f; // 0.0;
        p->_hfReference = 5000.0f;
        break;

    case I3dl2ReverbType::Preset_Hall:
        p->_room = -1045.f; //in milliBels
        p->_reflections = -796.f; //in milliBels
        p->_reverb = -310.f;  //in milliBels
        p->_diffusion = 80.f;
        p->_reflectionsDelay = 0.0058f; //in sec
        p->_reverbDelay = 0.050f;         //in sec
        p->_density = 69.0f;
        p->_decayTime = 2.2f;          // seconds 0.1 to 20.0
        p->_hfDecayRatio = 0.8f;      // 0.1 to 1.0
        p->_dryGain = 0.24f; // 0.4;            // 0.0 to 1.0
        p->_roomHf = -1000.f; // 0.0;
        p->_hfReference = 5000.0f;
        break;

    case I3dl2ReverbType::Preset_CavernousCathedral:
        p->_room = -1398.f; //in milliBels
        p->_reflections = -1398.f; //in milliBels
        p->_reverb = -602.f;  //in milliBels
        p->_diffusion = 50.f;
        p->_reflectionsDelay = 0.090f; //in sec
        p->_reverbDelay = 0.050f;         //in sec
        p->_density = 61.0f;
        p->_decayTime = 6.0f;// 12.0;         // seconds 0.1 to 20.0
        p->_hfDecayRatio = 0.65f;     // 0.1 to 1.0
        p->_dryGain = 0.4f;            // 0.0 to 1.0
        p->_roomHf = -1000.f; // 0.0;
        p->_hfReference = 5000.0f;
        break;

    case I3dl2ReverbType::Preset_MetalCorridor:
        p->_room = -1045.f; //in milliBels
        p->_reflections = -10000.f; //in milliBels
        p->_reverb = -444.f;  //in milliBels
        p->_diffusion = 100.f;
        p->_reflectionsDelay = 0.006f; //in sec
        p->_reverbDelay = 0.0f;         //in sec
        p->_density = 0.0f;
        p->_decayTime = 1.1f;          // seconds 0.1 to 20.0
        p->_hfDecayRatio = 0.3f;      // 0.1 to 1.0
        p->_dryGain = 0.4f;            // 0.0 to 1.0
        p->_roomHf = -1000.f; // 0.0;
        p->_hfReference = 5000.0f;
        break;
    default:
        break;
    }
}

void SetI3dl2ReverbParameters(I3dl2ReverbType* pOutReverb, const I3dl2ReverbParameterSet* pParameterSet) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pOutReverb, EffectType_I3dl2Reverb);
    I3dl2ReverbParameter* p = &pOutReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter;
    if (p->_parameterStatus != EffectParameterStatus_Init)
    {
        p->_parameterStatus = EffectParameterStatus_UpdateParam;
    }
    NN_SDK_REQUIRES_NOT_NULL(p);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pOutReverb);

    NN_SDK_REQUIRES_MINMAX(pParameterSet->roomHfGain, -10000.0f, 0.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->hfReference, 20.0f, 20000.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->lateReverbDecayTime, 0.1f, 20.0f);  // in sec
    NN_SDK_REQUIRES_MINMAX(pParameterSet->lateReverbHfDecayRatio, 0.1f, 2.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->roomGain, -10000.0f, 0.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->reflectionsGain, -10000.0f, 1000.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->reverbGain, -10000.0f, 2000.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->reverbDiffusion, 0.0f, 100.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->reflectionsDelayTime, 0.0f, 0.3f);  // in sec
    NN_SDK_REQUIRES_MINMAX(pParameterSet->reverbDelayTime, 0.0f, 0.1f);   // in sec
    NN_SDK_REQUIRES_MINMAX(pParameterSet->reverbDensity, 0.0f, 100.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->dryGain, 0.0f, 1.0f);

    p->_roomHf = pParameterSet->roomHfGain;
    p->_hfReference = pParameterSet->hfReference;
    p->_decayTime = pParameterSet->lateReverbDecayTime;
    p->_hfDecayRatio = pParameterSet->lateReverbHfDecayRatio;
    p->_room = pParameterSet->roomGain;
    p->_reflections = pParameterSet->reflectionsGain;
    p->_reverb = pParameterSet->reverbGain;
    p->_diffusion = pParameterSet->reverbDiffusion;
    p->_reflectionsDelay = pParameterSet->reflectionsDelayTime;
    p->_reverbDelay = pParameterSet->reverbDelayTime;
    p->_density = pParameterSet->reverbDensity;
    p->_dryGain = pParameterSet->dryGain;

}

I3dl2ReverbParameterSet GetI3dl2ReverbParameters(const I3dl2ReverbType* pReverb) NN_NOEXCEPT
{
    I3dl2ReverbParameterSet outParameterSet;
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_I3dl2Reverb);
    I3dl2ReverbParameter* pI3dl2ReverbParameter = &pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter;
    NN_SDK_REQUIRES_NOT_NULL(pI3dl2ReverbParameter);

    outParameterSet.roomHfGain = pI3dl2ReverbParameter->_roomHf;
    outParameterSet.hfReference = pI3dl2ReverbParameter->_hfReference;
    outParameterSet.lateReverbDecayTime = pI3dl2ReverbParameter->_decayTime;
    outParameterSet.lateReverbHfDecayRatio = pI3dl2ReverbParameter->_hfDecayRatio;
    outParameterSet.roomGain = pI3dl2ReverbParameter->_room;
    outParameterSet.reflectionsGain = pI3dl2ReverbParameter->_reflections;
    outParameterSet.reverbGain = pI3dl2ReverbParameter->_reverb;
    outParameterSet.reverbDiffusion = pI3dl2ReverbParameter->_diffusion;
    outParameterSet.reflectionsDelayTime = pI3dl2ReverbParameter->_reflectionsDelay;
    outParameterSet.reverbDelayTime = pI3dl2ReverbParameter->_reverbDelay;
    outParameterSet.reverbDensity = pI3dl2ReverbParameter->_density;
    outParameterSet.dryGain = pI3dl2ReverbParameter->_dryGain;

    return outParameterSet;
}

void LoadI3dl2ReverbPreset(I3dl2ReverbParameterSet* pOutParameter, I3dl2ReverbType::Preset preset) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutParameter);
    auto p = pOutParameter;
    NN_SDK_REQUIRES_MINMAX(preset, I3dl2ReverbType::Preset_SmallRoom, I3dl2ReverbType::Preset_MetalCorridor);

    switch (preset)
    {
    case I3dl2ReverbType::Preset_SmallRoom:
        p->roomGain = -1045.f;  //in milliBels
        p->reflectionsGain = -310.f;   //in milliBels
        p->reverbGain = -602.f;   //in milliBels
        p->reverbDiffusion = 100.f;
        p->reflectionsDelayTime = 0.0046f;  //in sec
        p->reverbDelayTime = 0.012f;   //in sec
        p->reverbDensity = 61.0f;
        p->lateReverbDecayTime = 1.0f;     // seconds 0.1 to 20.0
        p->lateReverbHfDecayRatio = 0.2f;     // 0.1 to 1.0
        p->dryGain = 0.24f;    // 0.4;            // 0.0 to 1.0
        p->roomHfGain = -1000.f;  // 0.0;
        p->hfReference = 5000.0f;
        break;

    case I3dl2ReverbType::Preset_LargeRoom:
        p->roomGain = -1045.f;  //in milliBels
        p->reflectionsGain = -310.f;   //in milliBels
        p->reverbGain = -310.f;   //in milliBels
        p->reverbDiffusion = 100.f;
        p->reflectionsDelayTime = 0.0092f;  //in sec
        p->reverbDelayTime = 0.040f;   //in sec
        p->reverbDensity = 61.0f;
        p->lateReverbDecayTime = 1.7f;     // seconds 0.1 to 20.0
        p->lateReverbHfDecayRatio = 0.3f;     // 0.1 to 1.0
        p->dryGain = 0.24f;    // 0.4;            // 0.0 to 1.0
        p->roomHfGain = -1000.f;  // 0.0;
        p->hfReference = 5000.0f;
        break;

    case I3dl2ReverbType::Preset_Hall:
        p->roomGain = -1045.f;  //in milliBels
        p->reflectionsGain = -796.f;   //in milliBels
        p->reverbGain = -310.f;   //in milliBels
        p->reverbDiffusion = 80.f;
        p->reflectionsDelayTime = 0.0058f;  //in sec
        p->reverbDelayTime = 0.050f;   //in sec
        p->reverbDensity = 69.0f;
        p->lateReverbDecayTime = 2.2f;     // seconds 0.1 to 20.0
        p->lateReverbHfDecayRatio = 0.8f;     // 0.1 to 1.0
        p->dryGain = 0.24f;    // 0.4;            // 0.0 to 1.0
        p->roomHfGain = -1000.f;  // 0.0;
        p->hfReference = 5000.0f;
        break;

    case I3dl2ReverbType::Preset_CavernousCathedral:
        p->roomGain = -1398.f;  //in milliBels
        p->reflectionsGain = -1398.f;  //in milliBels
        p->reverbGain = -602.f;   //in milliBels
        p->reverbDiffusion = 50.f;
        p->reflectionsDelayTime = 0.090f;   //in sec
        p->reverbDelayTime = 0.050f;   //in sec
        p->reverbDensity = 61.0f;
        p->lateReverbDecayTime = 6.0f;     // 12.0;         // seconds 0.1 to 20.0
        p->lateReverbHfDecayRatio = 0.65f;    // 0.1 to 1.0
        p->dryGain = 0.4f;     // 0.0 to 1.0
        p->roomHfGain = -1000.f;  // 0.0;
        p->hfReference = 5000.0f;
        break;

    case I3dl2ReverbType::Preset_MetalCorridor:
        p->roomGain = -1045.f;  //in milliBels
        p->reflectionsGain = -10000.f; //in milliBels
        p->reverbGain = -444.f;   //in milliBels
        p->reverbDiffusion = 100.f;
        p->reflectionsDelayTime = 0.006f;   //in sec
        p->reverbDelayTime = 0.0f;     //in sec
        p->reverbDensity = 0.0f;
        p->lateReverbDecayTime = 1.1f;     // seconds 0.1 to 20.0
        p->lateReverbHfDecayRatio = 0.3f;     // 0.1 to 1.0
        p->dryGain = 0.4f;     // 0.0 to 1.0
        p->roomHfGain = -1000.f;  // 0.0;
        p->hfReference = 5000.0f;
        break;
    default:
        break;
    }
}

void SetI3dl2ReverbEnabled(I3dl2ReverbType* pReverb, bool enable) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_I3dl2Reverb);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pReverb);
    pReverb->_pEffectInfo->_inParameter._enabled = enable;
    pReverb->_pEffectInfo->_inParameter._i3dl2ReverbParameter._parameterStatus = enable ? EffectParameterStatus_Init : EffectParameterStatus_UpdateParam;
}

bool IsI3dl2ReverbEnabled(const I3dl2ReverbType* pReverb) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pReverb, EffectType_I3dl2Reverb);
    return pReverb->_pEffectInfo->_inParameter._enabled;
}

template<typename T> Result AddBiquadFilterImpl(AudioRendererConfig* pOutConfig, BiquadFilterType* pBiquadFilter, T* pMixNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutConfig);
    NN_SDK_REQUIRES_NOT_NULL(pBiquadFilter);
    NN_SDK_REQUIRES_NOT_NULL(pMixNode);
    NN_SDK_REQUIRES_NOT_NULL(pMixNode->_pMixInfo);

    auto pEffectManager = pOutConfig->_pEffectManager;
    NN_AUDIO_CHECK_IF_INITIALIZED(pEffectManager, pBiquadFilter);

    NN_RESULT_DO(pEffectManager->Add(&pBiquadFilter->_pEffectInfo, EffectType_BiquadFilter, nullptr, 0, pMixNode->_pMixInfo));
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pBiquadFilter);

    // Set default values
    auto& inParameter = pBiquadFilter->_pEffectInfo->_inParameter;
    inParameter._enabled = true;
    inParameter._biquadFilterParameter._parameterStatus = EffectParameterStatus_Init;

    NN_RESULT_SUCCESS;
}

template<typename T> void RemoveBiquadFilterImpl(AudioRendererConfig* pOutConfig, BiquadFilterType* pBiquadFilter, T* pMixNode) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutConfig);
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pBiquadFilter, EffectType_BiquadFilter);
    NN_SDK_REQUIRES_NOT_NULL(pMixNode);
    NN_SDK_REQUIRES_NOT_NULL(pMixNode->_pMixInfo);
    NN_SDK_REQUIRES_NOT_NULL(pBiquadFilter->_pEffectInfo);
    NN_SDK_REQUIRES_EQUAL(pBiquadFilter->_pEffectInfo->_inParameter._type, EffectType_BiquadFilter);
    NN_SDK_REQUIRES(IsBiquadFilterRemovable(pBiquadFilter), "pBiquadFilter is being used. Check state with \"IsBiquadFilterRemovable()\"");
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pBiquadFilter);

    NN_UNUSED(pMixNode);

    auto pEffectManager = pOutConfig->_pEffectManager;
    pEffectManager->Remove(pBiquadFilter->_pEffectInfo, pMixNode->_pMixInfo);
    pBiquadFilter->_pEffectInfo = nullptr;
}

Result AddBiquadFilter(AudioRendererConfig* pOutConfig, BiquadFilterType* pBiquadFilter, FinalMixType* pFinalMix) NN_NOEXCEPT
{
    return AddBiquadFilterImpl(pOutConfig, pBiquadFilter, pFinalMix);
}

Result AddBiquadFilter(AudioRendererConfig* pOutConfig, BiquadFilterType* pBiquadFilter, SubMixType* pSubMix) NN_NOEXCEPT
{
    return AddBiquadFilterImpl(pOutConfig, pBiquadFilter, pSubMix);
}

void RemoveBiquadFilter(AudioRendererConfig* pOutConfig, BiquadFilterType* pBiquadFilter, FinalMixType* pFinalMix) NN_NOEXCEPT
{
    RemoveBiquadFilterImpl(pOutConfig, pBiquadFilter, pFinalMix);
}

void RemoveBiquadFilter(AudioRendererConfig* pOutConfig, BiquadFilterType* pBiquadFilter, SubMixType* pSubMix) NN_NOEXCEPT
{
    RemoveBiquadFilterImpl(pOutConfig, pBiquadFilter, pSubMix);
}

void GetBiquadFilterInputOutput(const BiquadFilterType* pBiquadFilter, int8_t* outInput, int8_t* outOutput, int* pOutCount, int count) NN_NOEXCEPT
{
    const auto& parameter = pBiquadFilter->_pEffectInfo->_inParameter._biquadFilterParameter;

    NN_AUDIO_REQUIRES_EFFECT_TYPE(pBiquadFilter, EffectType_BiquadFilter);
    NN_SDK_REQUIRES_NOT_NULL(outInput);
    NN_SDK_REQUIRES_NOT_NULL(outOutput);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES_MINMAX(count, 0, BiquadFilterType::ChannelCountMax);

    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pBiquadFilter);

    const auto copyCount = std::min(count, static_cast<int>(parameter._channelCount));

    for (int i = 0; i < copyCount; ++i)
    {
        outInput[i] = parameter._input[i];
        outOutput[i] = parameter._output[i];
    }

    *pOutCount = copyCount;
}

void SetBiquadFilterInputOutput(BiquadFilterType* pBiquadFilter, const int8_t* input, const int8_t* output, int count) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pBiquadFilter, EffectType_BiquadFilter);
    NN_SDK_REQUIRES_NOT_NULL(input);
    NN_SDK_REQUIRES_NOT_NULL(output);
    NN_SDK_REQUIRES_MINMAX(count, 0, BiquadFilterType::ChannelCountMax);
    NN_AUDIO_DETAIL_EFFECT_EXCLUSION_SCOPED_CHEKCER(pBiquadFilter);

    auto& parameter = pBiquadFilter->_pEffectInfo->_inParameter._biquadFilterParameter;

    const auto copyCount = static_cast<int8_t>(std::min(count, nn::audio::BiquadFilterType::ChannelCountMax));

    for (int i = 0; i < copyCount; ++i)
    {
        parameter._input[i] = input[i];
        parameter._output[i] = output[i];
    }

    parameter._channelCount = copyCount;
    parameter._parameterStatus = EffectParameterStatus_UpdateParam;
}

BiquadFilterParameter GetBiquadFilterParameter(const BiquadFilterType* pBiquadFilter) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pBiquadFilter, EffectType_BiquadFilter);
    const auto& parameter = pBiquadFilter->_pEffectInfo->_inParameter._biquadFilterParameter;

    const BiquadFilterParameter filterParameter = {
        IsBiquadFilterEnabled(pBiquadFilter),
        { parameter._numerator[0], parameter._numerator[1], parameter._numerator[2] },
        { parameter._denominator[0], parameter._denominator[1] },
    };

    return filterParameter;
}

void SetBiquadFilterParameter(BiquadFilterType* pBiquadFilter, const BiquadFilterParameter& parameter) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pBiquadFilter, EffectType_BiquadFilter);
    auto& dstParameter = pBiquadFilter->_pEffectInfo->_inParameter._biquadFilterParameter;

    std::memcpy(dstParameter._numerator, parameter.numerator, sizeof(parameter.numerator));
    std::memcpy(dstParameter._denominator, parameter.denominator, sizeof(parameter.denominator));

    SetBiquadFilterEnabled(pBiquadFilter, parameter.enable);
    dstParameter._parameterStatus = EffectParameterStatus_UpdateParam;
}

bool IsBiquadFilterEnabled(const BiquadFilterType* pBiquadFilter) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pBiquadFilter, EffectType_BiquadFilter);
    const auto& inParameter = pBiquadFilter->_pEffectInfo->_inParameter;

    return inParameter._enabled;
}

void SetBiquadFilterEnabled(BiquadFilterType* pBiquadFilter, bool enable) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pBiquadFilter, EffectType_BiquadFilter);
    auto& inParameter = pBiquadFilter->_pEffectInfo->_inParameter;

    inParameter._enabled = enable;
    inParameter._biquadFilterParameter._parameterStatus = EffectParameterStatus_UpdateParam;
}

bool IsBiquadFilterRemovable(const BiquadFilterType* pBiquadFilter) NN_NOEXCEPT
{
    NN_AUDIO_REQUIRES_EFFECT_TYPE(pBiquadFilter, EffectType_BiquadFilter);

    return IsRemovable(pBiquadFilter, EffectType_BiquadFilter)
            && (IsBiquadFilterEnabled(pBiquadFilter) == false);
}

}  // namespace audio
}  // namespace nn

