﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <algorithm>
#include <cstring>

#include <nn/nn_SdkAssert.h>
#include <nn/nn_SdkLog.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_BitUtil.h>
#include <nn/util/util_BytePtr.h>
#include <nn/nn_TimeSpan.h>
#include <nn/os/os_Cache.h>

#include <nn/audio/audio_Common.h>
#include <nn/audio/audio_Result.h>
#include <nn/audio/audio_EffectTypes.h>
#include <nn/audio/audio_AuxUtilityEffectTypes.h>
#include <nn/audio/audio_EffectApi.h>
#include <nn/audio/audio_AuxUtilityEffectApi.h>

#include "audio_EffectUtil.h"
#include "audio_EffectPrivateTypes.h"
#include "dsp/audio_EffectCommon.h"
#include "dsp/audio_EffectReverb.h"
#include "dsp/audio_EffectI3dl2Reverb.h"
#include "dsp/audio_EffectDelay.h"
#include "common/audio_Command.h"

namespace nn {
namespace audio {

/////////////////////////////////////////////////////////////////////////////
// Aux I3dl2 Reverb
/////////////////////////////////////////////////////////////////////////////

size_t GetRequiredBufferSizeForAuxI3dl2Reverb(int sampleRate, int channelCountMax) NN_NOEXCEPT
{
    NN_SDK_REQUIRES((sampleRate == 32000) || (sampleRate == 48000));
    NN_SDK_REQUIRES((channelCountMax == 1) || (channelCountMax == 2) || (channelCountMax == 4) || (channelCountMax == 6));

    size_t delayLineSize = nn::util::align_up(I3DL2ReverbGetRequiredDelayLineBufferSize(sampleRate, channelCountMax), nn::audio::BufferAlignSize);
    size_t workBufferSize = delayLineSize + sizeof(I3dl2ReverbParameter) + sizeof(I3dl2ReverbState);
    return workBufferSize;
}


void InitializeAuxI3dl2Reverb(AuxI3dl2ReverbType* pOutReverb, void* buffer, size_t bufferSize, int sampleRate, int channelCountMax) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutReverb);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    auto delayLineBufferSize = GetRequiredBufferSizeForI3dl2Reverb(sampleRate, channelCountMax);
    NN_SDK_ASSERT_GREATER_EQUAL(bufferSize, sizeof(I3dl2ReverbState) + sizeof(I3dl2ReverbParameter) + delayLineBufferSize);
    NN_SDK_REQUIRES(channelCountMax == 1 || channelCountMax == 2 || channelCountMax == 4 || channelCountMax == 6);
    NN_UNUSED(delayLineBufferSize);

    pOutReverb->_workBuffer = buffer;
    pOutReverb->_bufferSize = bufferSize;

    auto pI3dl2ReverbParameter = reinterpret_cast<I3dl2ReverbParameter*> (buffer);
    memset(pI3dl2ReverbParameter, 0, sizeof(I3dl2ReverbParameter));
    pI3dl2ReverbParameter->_numChannelCountMax = static_cast<uint16_t>(channelCountMax);
    pI3dl2ReverbParameter->_numChannels = static_cast<uint16_t>(channelCountMax);
    pI3dl2ReverbParameter->_sampleRate = sampleRate; // [Hz]
    pI3dl2ReverbParameter->_parameterStatus = EffectParameterStatus_Init;
}

int GetAuxI3dl2ReverbChannelCountMax(const AuxI3dl2ReverbType* pReverb) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pReverb);
    const auto pI3dl2ReverbParameter = reinterpret_cast<I3dl2ReverbParameter*>(pReverb->_workBuffer);
    return pI3dl2ReverbParameter->_numChannelCountMax;
}

void SetAuxI3dl2ReverbInputOutput(AuxI3dl2ReverbType* pOutReverb, const int8_t* input, const int8_t* output, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutReverb);
    NN_SDK_REQUIRES_NOT_NULL(input);
    NN_SDK_REQUIRES_NOT_NULL(output);

    NN_SDK_REQUIRES((count == 1) || (count == 2) || (count == 4) || (count == 6));

    auto pI3dl2ReverbParameter = reinterpret_cast<I3dl2ReverbParameter*> (pOutReverb->_workBuffer);

    auto c = std::min(count,
        std::min(static_cast<int>(pI3dl2ReverbParameter->_numChannelCountMax),
            nn::audio::I3dl2ReverbParameter::SupportedChannelCountMax));
    for (int i = 0; i < c; ++i)
    {
        pI3dl2ReverbParameter->_input[i] = input[i];
        pI3dl2ReverbParameter->_output[i] = output[i];
    }

    pI3dl2ReverbParameter->_numChannels = static_cast<uint16_t>(c);
    pI3dl2ReverbParameter->_parameterStatus = EffectParameterStatus_Init;

}

void GetAuxI3dl2ReverbInputOutput(int8_t* outInput, int8_t* outOutput, int* pOutCount, const AuxI3dl2ReverbType* pReverb, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pReverb);
    NN_SDK_REQUIRES_NOT_NULL(outInput);
    NN_SDK_REQUIRES_NOT_NULL(outOutput);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES((count == 1) || (count == 2) || (count == 4) || (count == 6));

    const auto pI3dl2ReverbParameter = reinterpret_cast<I3dl2ReverbParameter*> (pReverb->_workBuffer);

    auto c = std::min(static_cast<uint16_t>(count), pI3dl2ReverbParameter->_numChannels);
    for (int i = 0; i < c; ++i)
    {
        outInput[i] = pI3dl2ReverbParameter->_input[i];
        outOutput[i] = pI3dl2ReverbParameter->_output[i];
    }
    *pOutCount = pI3dl2ReverbParameter->_numChannels;
}

void SetAuxI3dl2ReverbParameters(AuxI3dl2ReverbType* pOutReverb, const I3dl2ReverbParameterSet* pParameterSet) NN_NOEXCEPT
{
    auto pI3dl2ReverbParameter = reinterpret_cast<I3dl2ReverbParameter*> (pOutReverb->_workBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pI3dl2ReverbParameter);
    if (pI3dl2ReverbParameter->_parameterStatus != EffectParameterStatus_Init)
    {
        pI3dl2ReverbParameter->_parameterStatus = EffectParameterStatus_UpdateParam;
    }

    NN_SDK_REQUIRES_MINMAX(pParameterSet->roomHfGain, -10000.0f, 0.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->hfReference, 20.0f, 20000.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->lateReverbDecayTime, 0.1f, 20.0f);  // in sec
    NN_SDK_REQUIRES_MINMAX(pParameterSet->lateReverbHfDecayRatio, 0.1f, 2.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->roomGain, -10000.0f, 0.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->reflectionsGain, -10000.0f, 1000.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->reverbGain, -10000.0f, 2000.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->reverbDiffusion, 0.0f, 100.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->reflectionsDelayTime, 0.0f, 0.3f);  // in sec
    NN_SDK_REQUIRES_MINMAX(pParameterSet->reverbDelayTime, 0.0f, 0.1f);   // in sec
    NN_SDK_REQUIRES_MINMAX(pParameterSet->reverbDensity, 0.0f, 100.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->dryGain, 0.0f, 1.0f);

    pI3dl2ReverbParameter->_roomHf = pParameterSet->roomHfGain;
    pI3dl2ReverbParameter->_hfReference = pParameterSet->hfReference;
    pI3dl2ReverbParameter->_decayTime = pParameterSet->lateReverbDecayTime;
    pI3dl2ReverbParameter->_hfDecayRatio = pParameterSet->lateReverbHfDecayRatio;
    pI3dl2ReverbParameter->_room = pParameterSet->roomGain;
    pI3dl2ReverbParameter->_reflections = pParameterSet->reflectionsGain;
    pI3dl2ReverbParameter->_reverb = pParameterSet->reverbGain;
    pI3dl2ReverbParameter->_diffusion = pParameterSet->reverbDiffusion;
    pI3dl2ReverbParameter->_reflectionsDelay = pParameterSet->reflectionsDelayTime;
    pI3dl2ReverbParameter->_reverbDelay = pParameterSet->reverbDelayTime;
    pI3dl2ReverbParameter->_density = pParameterSet->reverbDensity;
    pI3dl2ReverbParameter->_dryGain = pParameterSet->dryGain;

}

nn::audio::I3dl2ReverbParameterSet GetAuxI3dl2ReverbParameters(const AuxI3dl2ReverbType* pReverb) NN_NOEXCEPT
{
    nn::audio::I3dl2ReverbParameterSet outParameterSet;
    auto pI3dl2ReverbParameter = reinterpret_cast<I3dl2ReverbParameter*> (pReverb->_workBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pI3dl2ReverbParameter);

    outParameterSet.roomHfGain = pI3dl2ReverbParameter->_roomHf;
    outParameterSet.hfReference = pI3dl2ReverbParameter->_hfReference;
    outParameterSet.lateReverbDecayTime = pI3dl2ReverbParameter->_decayTime;
    outParameterSet.lateReverbHfDecayRatio = pI3dl2ReverbParameter->_hfDecayRatio;
    outParameterSet.roomGain = pI3dl2ReverbParameter->_room;
    outParameterSet.reflectionsGain = pI3dl2ReverbParameter->_reflections;
    outParameterSet.reverbGain = pI3dl2ReverbParameter->_reverb;
    outParameterSet.reverbDiffusion = pI3dl2ReverbParameter->_diffusion;
    outParameterSet.reflectionsDelayTime = pI3dl2ReverbParameter->_reflectionsDelay;
    outParameterSet.reverbDelayTime = pI3dl2ReverbParameter->_reverbDelay;
    outParameterSet.reverbDensity = pI3dl2ReverbParameter->_density;
    outParameterSet.dryGain = pI3dl2ReverbParameter->_dryGain;

    return outParameterSet;
}

void ProcessAuxI3dl2Reverb(AuxI3dl2ReverbType* pOutReverb, int32_t* outBuffer, const int32_t* inBuffer, const uint32_t samplesPerFrame) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutReverb);
    NN_SDK_REQUIRES_NOT_NULL(inBuffer);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);

    const int32_t** ppInData;
    int32_t** ppOutData;
    auto pI3dl2ReverbParameter = reinterpret_cast<I3dl2ReverbParameter*> (pOutReverb->_workBuffer);
    int16_t channelCount = pI3dl2ReverbParameter->_numChannels;
    const int8_t* input = pI3dl2ReverbParameter->_input;
    const int8_t* output = pI3dl2ReverbParameter->_output;

    const int32_t*  inFrameAddr[I3dl2ReverbParameter::ChannelCountMax];
    int32_t*  outFrameAddr[I3dl2ReverbParameter::ChannelCountMax];
    bool notByPassed = true;

    for (auto ch = 0; ch < channelCount; ++ch)
    {
        inFrameAddr[ch] = inBuffer + input[ch] * samplesPerFrame;
        outFrameAddr[ch] = outBuffer + output[ch] * samplesPerFrame;
    }
    ppInData = &inFrameAddr[0];
    ppOutData = &outFrameAddr[0];

    auto pState = reinterpret_cast<I3dl2ReverbState*> (reinterpret_cast<uint64_t> (pOutReverb->_workBuffer) + sizeof(I3dl2ReverbParameter));
    auto pDelayLineBuffer = reinterpret_cast<void*>(reinterpret_cast<uint64_t> (pOutReverb->_workBuffer) + sizeof(I3dl2ReverbParameter) + sizeof(I3dl2ReverbState));

    if(pI3dl2ReverbParameter->_parameterStatus == EffectParameterStatus_Init)
    {
        InitializeI3dl2ReverbEffect(pI3dl2ReverbParameter, pState, pDelayLineBuffer);
        pI3dl2ReverbParameter->_parameterStatus = EffectParameterStatus_Updated;
    }
    else if(pI3dl2ReverbParameter->_parameterStatus == EffectParameterStatus_UpdateParam)
    {
        UpdateI3dl2ReverbEffectParameter(pI3dl2ReverbParameter, pState, false);
        pI3dl2ReverbParameter->_parameterStatus = EffectParameterStatus_Updated;
    }

    ApplyI3dl2ReverbEffect(pI3dl2ReverbParameter, pState, notByPassed, ppInData, ppOutData, samplesPerFrame);
}

void ResetAuxI3dl2Reverb(AuxI3dl2ReverbType* pOutReverb) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutReverb);
    auto pI3dl2ReverbParameter = reinterpret_cast<I3dl2ReverbParameter*> (pOutReverb->_workBuffer);
    pI3dl2ReverbParameter->_parameterStatus = EffectParameterStatus_Init;
}

size_t GetRequiredBufferSizeForAuxReverb(int sampleRate, int channelCountMax) NN_NOEXCEPT
{
    NN_SDK_REQUIRES((sampleRate == 32000) || (sampleRate == 48000));
    NN_SDK_REQUIRES((channelCountMax == 1) || (channelCountMax == 2) || (channelCountMax == 4) || (channelCountMax == 6));

    qf fs = sampleRate / 1000 * ( 1 << QF_FRACTIONAL_BIT_COUNT ); // [kHz]
    size_t workBufferSize = nn::util::align_up(ReverbGetRequiredDelayLineBufferSize(fs, channelCountMax), nn::audio::BufferAlignSize);
    workBufferSize += sizeof(ReverbState) + sizeof(ReverbParameter);
    return workBufferSize;
}

void InitializeAuxReverb(AuxReverbType* pOutReverb, void* buffer, size_t bufferSize, int sampleRate, int channelCountMax) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutReverb);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    auto delayLineBufferSize = GetRequiredBufferSizeForReverb(sampleRate, channelCountMax);
    NN_SDK_ASSERT_GREATER_EQUAL(bufferSize, sizeof(ReverbState) + sizeof(ReverbParameter) + delayLineBufferSize);
    NN_SDK_REQUIRES((channelCountMax == 1) || (channelCountMax == 2) || (channelCountMax == 4) || (channelCountMax == 6));
    NN_UNUSED(delayLineBufferSize);

    pOutReverb->_workBuffer = buffer;
    pOutReverb->_bufferSize = bufferSize;

    auto pReverbParameter = reinterpret_cast<ReverbParameter*>(buffer);
    memset(pReverbParameter, 0, sizeof(ReverbParameter));
    pReverbParameter->_numChannelCountMax = static_cast<uint16_t>(channelCountMax);
    pReverbParameter->_numChannels = static_cast<uint16_t>(channelCountMax);
    pReverbParameter->_sampleRate = sampleRate;
    pReverbParameter->_parameterStatus = EffectParameterStatus_Init;
    pReverbParameter->_sampleRate = sampleRate / 1000 * (1 << QF_FRACTIONAL_BIT_COUNT); // [kHz]
}

int GetAuxReverbChannelCountMax(const AuxReverbType* pReverb) NN_NOEXCEPT
{
    NN_SDK_ASSERT_NOT_NULL(pReverb);
    const auto pReverbParameter = reinterpret_cast<ReverbParameter*>(pReverb->_workBuffer);
    return pReverbParameter->_numChannelCountMax;
}

void GetAuxReverbInputOutput(int8_t* outInput, int8_t* outOutput, int* pOutCount, const AuxReverbType* pReverb, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pReverb);
    NN_SDK_REQUIRES_NOT_NULL(outInput);
    NN_SDK_REQUIRES_NOT_NULL(outOutput);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES((count == 1) || (count == 2) || (count == 4) || (count == 6));

    const auto pReverbParameter = reinterpret_cast<ReverbParameter*>(pReverb->_workBuffer);
    auto c = std::min(static_cast<uint16_t>(count), pReverbParameter->_numChannels);
    for(int i = 0; i < c; ++i)
    {
        outInput[i] = pReverbParameter->_input[i];
        outOutput[i] = pReverbParameter->_output[i];
    }
    *pOutCount = pReverbParameter->_numChannels;
}

void SetAuxReverbParameters(AuxReverbType* pOutReverb, const ReverbParameterSet* pParameter) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutReverb);
    auto pReverbParameter = reinterpret_cast<ReverbParameter*>(pOutReverb->_workBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pParameter);
    if(pReverbParameter->_parameterStatus != EffectParameterStatus_Init)
    {
        pReverbParameter->_parameterStatus = EffectParameterStatus_UpdateParam;
    }
    NN_SDK_REQUIRES_MINMAX(pParameter->earlyGain, 0.0f, 1.0f);
    NN_SDK_REQUIRES_MINMAX(pParameter->lateGain, 0.0f, 1.0f);
    NN_SDK_REQUIRES_MINMAX(pParameter->decayTimeSeconds, 0.1f, 20.0f);
    NN_SDK_REQUIRES_MINMAX(pParameter->highFreqDecayRatio, 0.1f, 1.0f);
    NN_SDK_REQUIRES_MINMAX(pParameter->coloration, 0.0f, 1.0f);
    NN_SDK_REQUIRES_MINMAX(pParameter->reverbGain, 0.0f, 1.0f);
    NN_SDK_REQUIRES_MINMAX(pParameter->outGain, 0.0f, 1.0f);
    NN_SDK_REQUIRES_MINMAX(pParameter->dryGain, 0.0f, 1.0f);
    pReverbParameter->_earlyMode = ConvertToDspType(pParameter->earlyMode);
    pReverbParameter->_earlyGain = qf(pParameter->earlyGain * ( 1 << QF_FRACTIONAL_BIT_COUNT));
    pReverbParameter->_predelayTime = qf(pParameter->predelayTimeMilliSeconds * ( 1 << QF_FRACTIONAL_BIT_COUNT));;
    pReverbParameter->_lateMode = ConvertToDspType(pParameter->lateMode);
    pReverbParameter->_lateGain = qf(pParameter->lateGain * ( 1 << QF_FRACTIONAL_BIT_COUNT));
    pReverbParameter->_decayTime = qf(pParameter->decayTimeSeconds * ( 1 << QF_FRACTIONAL_BIT_COUNT)); //seconds conversion
    pReverbParameter->_highFrequencyDecayRatio = qf(pParameter->highFreqDecayRatio * ( 1 << QF_FRACTIONAL_BIT_COUNT));
    pReverbParameter->_coloration = qf(pParameter->coloration * ( 1 << QF_FRACTIONAL_BIT_COUNT));
    pReverbParameter->_reverbGain = qf(pParameter->reverbGain * ( 1 << QF_FRACTIONAL_BIT_COUNT));
    pReverbParameter->_outGain = qf(pParameter->outGain * ( 1 << QF_FRACTIONAL_BIT_COUNT));
    pReverbParameter->_dryGain = qf(pParameter->dryGain * ( 1 << QF_FRACTIONAL_BIT_COUNT));
}

ReverbParameterSet GetAuxReverbParameters(const AuxReverbType* pOutReverb)
{
    NN_SDK_REQUIRES_NOT_NULL(pOutReverb);
    const ReverbParameter* reverb = reinterpret_cast<const ReverbParameter*>(pOutReverb->_workBuffer);
    ReverbParameterSet auxRevParamSet;
    auxRevParamSet.earlyMode = ConvertFromDspType(reverb->_earlyMode);
    auxRevParamSet.earlyGain = static_cast<float>(reverb->_earlyGain) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    auxRevParamSet.predelayTimeMilliSeconds = static_cast<float>(reverb->_predelayTime) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    auxRevParamSet.lateMode = ConvertFromDspType(reverb->_lateMode);
    auxRevParamSet.lateGain = static_cast<float>(reverb->_lateGain) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    auxRevParamSet.decayTimeSeconds = static_cast<float>(reverb->_decayTime) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    auxRevParamSet.highFreqDecayRatio = static_cast<float>(reverb->_highFrequencyDecayRatio) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    auxRevParamSet.coloration = static_cast<float>(reverb->_coloration) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    auxRevParamSet.reverbGain = static_cast<float>(reverb->_reverbGain) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    auxRevParamSet.outGain = static_cast<float>(reverb->_outGain) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    auxRevParamSet.dryGain = static_cast<float>(reverb->_dryGain) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    return auxRevParamSet;
}

void SetAuxReverbInputOutput(AuxReverbType* pOutReverb, const int8_t* input, const int8_t* output, int count) NN_NOEXCEPT
{
    auto pReverbParameter = reinterpret_cast<ReverbParameter*> (pOutReverb->_workBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pReverbParameter);
    pReverbParameter->_parameterStatus = EffectParameterStatus_Init;
    auto c = std::min(count,
                      std::min(static_cast<int>(pReverbParameter->_numChannelCountMax),
                               nn::audio::ReverbParameter::SupportedChannelCountMax));
    for (int i = 0; i < c; ++i)
    {
        pReverbParameter->_input[i] = input[i];
        pReverbParameter->_output[i] = output[i];
    }
    pReverbParameter->_numChannels = static_cast<uint16_t>(c);

}

void ResetAuxReverb(AuxReverbType* pOutReverb) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutReverb);
    auto pReverbParameter = reinterpret_cast<ReverbParameter*>(pOutReverb->_workBuffer);
    pReverbParameter->_parameterStatus = EffectParameterStatus_Init;
}

void ProcessAuxReverb(AuxReverbType* pOutReverb, int32_t* outBuffer, const int32_t* inBuffer, const uint32_t samplesPerFrame) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutReverb);
    NN_SDK_REQUIRES_NOT_NULL(inBuffer);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);

    const int32_t** ppInData;
    int32_t** ppOutData;
    auto pReverbParameter = reinterpret_cast<ReverbParameter*> (pOutReverb->_workBuffer);
    int16_t channelCount = pReverbParameter->_numChannels;
    const int8_t* input = pReverbParameter->_input;
    const int8_t* output = pReverbParameter->_output;
    bool longSizePreDelaySupport = true;
    const int32_t*  inFrameAddr[ReverbParameter::ChannelCountMax];
    int32_t*  outFrameAddr[ReverbParameter::ChannelCountMax];
    bool notByPassed = true;

    for (auto ch = 0; ch < channelCount; ++ch)
    {
        inFrameAddr[ch] = inBuffer + input[ch] * samplesPerFrame;
        outFrameAddr[ch] = outBuffer + output[ch] * samplesPerFrame;
    }
    ppInData = &inFrameAddr[0];
    ppOutData = &outFrameAddr[0];

    auto pState = reinterpret_cast<ReverbState*> (reinterpret_cast<uint64_t> (pOutReverb->_workBuffer) + sizeof(ReverbParameter));
    auto pDelayLineBuffer = reinterpret_cast<void*>(reinterpret_cast<uint64_t> (pOutReverb->_workBuffer) + sizeof(ReverbParameter) + sizeof(ReverbState));

    if(pReverbParameter->_parameterStatus == EffectParameterStatus_Init)
    {
        InitializeReverbEffect(pReverbParameter, pState, pDelayLineBuffer, longSizePreDelaySupport);
        pReverbParameter->_parameterStatus = EffectParameterStatus_Updated;
    }
    else if(pReverbParameter->_parameterStatus == EffectParameterStatus_UpdateParam)
    {
        UpdateReverbEffectParameter(pReverbParameter, pState);
        pReverbParameter->_parameterStatus = EffectParameterStatus_Updated;
    }

    ApplyReverbEffect(pReverbParameter, pState, notByPassed, ppInData, ppOutData, samplesPerFrame);
}

size_t GetRequiredBufferSizeForAuxDelay(nn::TimeSpan delayTime, int sampleRate, int channelCount) NN_NOEXCEPT
{
    NN_SDK_REQUIRES((sampleRate == 32000) || (sampleRate == 48000));
    NN_SDK_REQUIRES((channelCount == 1) || (channelCount == 2) || (channelCount == 4) || (channelCount == 6));
    NN_SDK_REQUIRES_LESS(delayTime.GetMilliSeconds(), std::numeric_limits<int32_t>::max());

    const auto bufferSize = nn::util::align_up(sizeof(qf) * static_cast<size_t>(channelCount * sampleRate * delayTime.GetMilliSeconds() / 1000), nn::audio::BufferAlignSize)
        + sizeof(DelayParameter) + sizeof(DelayState);
    return bufferSize;
}

void InitializeAuxDelay(AuxDelayType* pDelay, void* buffer, size_t bufferSize, nn::TimeSpan delayTimeMax, int sampleRate, int channelCountMax) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDelay);
    NN_SDK_REQUIRES_NOT_NULL(buffer);
    NN_SDK_REQUIRES(channelCountMax == 1 || channelCountMax == 2 || channelCountMax == 4 || channelCountMax == 6);
    NN_SDK_REQUIRES_GREATER_EQUAL(bufferSize, GetRequiredBufferSizeForAuxDelay(delayTimeMax, sampleRate, channelCountMax));
    pDelay->_workBuffer = buffer;
    pDelay->_bufferSize = bufferSize;

    auto pDelayParameter = reinterpret_cast<DelayParameter*>(pDelay->_workBuffer);
    memset(pDelayParameter, 0, pDelay->_bufferSize);
    pDelayParameter->_numChannels = static_cast<uint16_t>(channelCountMax);
    pDelayParameter->_numChannelCountMax = static_cast<uint16_t>(channelCountMax);
    pDelayParameter->_sampleRate = sampleRate * (1 << QF_FRACTIONAL_BIT_COUNT);
    pDelayParameter->_delayTimeMax = static_cast<int32_t>(delayTimeMax.GetMilliSeconds());
}

void GetAuxDelayInputOutput(int8_t* outInput, int8_t* outOutput, int* pOutCount, const AuxDelayType* pDelay, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDelay);
    NN_SDK_REQUIRES_NOT_NULL(outInput);
    NN_SDK_REQUIRES_NOT_NULL(outOutput);
    NN_SDK_REQUIRES_NOT_NULL(pOutCount);
    NN_SDK_REQUIRES((count == 1) || (count == 2) || (count == 4) || (count == 6));
    auto pDelayParameter = reinterpret_cast<DelayParameter*>(pDelay->_workBuffer);
    auto c = std::min(static_cast<uint16_t>(count), pDelayParameter->_numChannels);
    for(int i = 0; i < c; ++i)
    {
        outInput[i] = pDelayParameter->_input[i];
        outOutput[i] = pDelayParameter->_output[i];
    }
    *pOutCount = pDelayParameter->_numChannels;
}

void SetAuxDelayInputOutput(AuxDelayType* pOutDelay, const int8_t* input, const int8_t* output, int count) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutDelay);
    NN_SDK_REQUIRES_NOT_NULL(input);
    NN_SDK_REQUIRES_NOT_NULL(output);
    auto pDelayParameter = reinterpret_cast<DelayParameter*> (pOutDelay->_workBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pDelayParameter);
    pDelayParameter->_parameterStatus = EffectParameterStatus_Init;
    auto c = std::min(count,
                      std::min(static_cast<int>(pDelayParameter->_numChannelCountMax),
                               nn::audio::DelayParameter::SupportedChannelCountMax));
    for (int i = 0; i < c; ++i)
    {
        pDelayParameter->_input[i] = input[i];
        pDelayParameter->_output[i] = output[i];
    }
    pDelayParameter->_numChannels = static_cast<uint16_t>(c);
}

DelayParameterSet GetAuxDelayParameters(const AuxDelayType* pDelay) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDelay);
    auto pDelayParameter = reinterpret_cast<DelayParameter*>(pDelay->_workBuffer);
    DelayParameterSet delayParameterSet;
    delayParameterSet.delayTime = nn::TimeSpan::FromMilliSeconds(pDelayParameter->_delayTime);
    delayParameterSet.inGain = static_cast<float>(pDelayParameter->_inGain) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    delayParameterSet.feedbackGain = static_cast<float>(pDelayParameter->_feedbackGain) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    delayParameterSet.dryGain = static_cast<float>(pDelayParameter->_dryGain) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    delayParameterSet.channelSpread = static_cast<float>(pDelayParameter->_channelSpread) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    delayParameterSet.lowPassAmount = static_cast<float>(pDelayParameter->_lowPassAmount) / static_cast<float>(1 << QF_FRACTIONAL_BIT_COUNT);
    return delayParameterSet;
}

void SetAuxDelayParameters(AuxDelayType* pDelay, const DelayParameterSet* pParameterSet) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDelay);
    NN_SDK_REQUIRES_NOT_NULL(pParameterSet);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->delayTime, nn::TimeSpan::FromMilliSeconds(0), GetAuxDelayTimeMax(pDelay));
    NN_SDK_REQUIRES_MINMAX(pParameterSet->inGain, 0.0f, 1.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->feedbackGain, 0.0f, 1.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->dryGain, 0.0f, 1.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->channelSpread, 0.0f, 1.0f);
    NN_SDK_REQUIRES_MINMAX(pParameterSet->lowPassAmount, 0.0f, 1.0f);
    auto pDelayParameter = reinterpret_cast<DelayParameter*>(pDelay->_workBuffer);
    NN_SDK_REQUIRES_NOT_NULL(pDelayParameter);
    int32_t delayTimeMilliSeconds = static_cast<int32_t>(pParameterSet->delayTime.GetMilliSeconds());
    if((delayTimeMilliSeconds != pDelayParameter->_delayTime))
    {
        pDelayParameter->_delayTime = delayTimeMilliSeconds;
        pDelayParameter->_parameterStatus = EffectParameterStatus_Init;
        auto pDelayState = reinterpret_cast<DelayState*>(reinterpret_cast<uint64_t>(pDelay->_workBuffer) + sizeof(DelayParameter));
        auto pWorkBuffer = reinterpret_cast<void*>(reinterpret_cast<uint64_t>(pDelay->_workBuffer) + sizeof(DelayParameter) + sizeof(DelayState));
        InitializeDelayEffect(pDelayParameter, pDelayState, pWorkBuffer);
    }
    pDelayParameter->_inGain = qf(pParameterSet->inGain * (1 << QF_FRACTIONAL_BIT_COUNT));
    pDelayParameter->_feedbackGain = qf(pParameterSet->feedbackGain * (1 << QF_FRACTIONAL_BIT_COUNT));
    pDelayParameter->_outGain = qf(1.0f * (1 << QF_FRACTIONAL_BIT_COUNT)); // outGain is fixed to 1.0f
    pDelayParameter->_dryGain = qf(pParameterSet->dryGain * (1 << QF_FRACTIONAL_BIT_COUNT));
    pDelayParameter->_channelSpread = qf(pParameterSet->channelSpread * (1 << QF_FRACTIONAL_BIT_COUNT));
    pDelayParameter->_lowPassAmount = qf(pParameterSet->lowPassAmount * (1 << QF_FRACTIONAL_BIT_COUNT));
    pDelayParameter->_parameterStatus = (pDelayParameter->_parameterStatus == EffectParameterStatus_Init) ? pDelayParameter->_parameterStatus : EffectParameterStatus_UpdateParam;
    auto pDelayState = reinterpret_cast<DelayState*>(reinterpret_cast<uint64_t>(pDelay->_workBuffer) + sizeof(DelayParameter));
    SetDelayEffectParameter(pDelayParameter, pDelayState);
}

void ProcessAuxDelay(AuxDelayType* pOutDelay, int32_t* outBuffer, const int32_t* inBuffer, const uint32_t samplesPerFrame) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutDelay);
    NN_SDK_REQUIRES_NOT_NULL(inBuffer);
    NN_SDK_REQUIRES_NOT_NULL(outBuffer);
    const int32_t** ppInData;
    int32_t** ppOutData;
    auto pDelayParameter = reinterpret_cast<DelayParameter*>(pOutDelay->_workBuffer);

    const int8_t* input = pDelayParameter->_input;
    int8_t* output = pDelayParameter->_output;
    const int32_t*  inFrameAddr[DelayParameter::ChannelCountMax];
    int32_t*  outFrameAddr[DelayParameter::ChannelCountMax];
    bool notByPassed = true;
    for (auto ch = 0; ch < pDelayParameter->_numChannels; ++ch)
    {
        inFrameAddr[ch] = inBuffer + input[ch] * samplesPerFrame;
        outFrameAddr[ch] = outBuffer + output[ch] * samplesPerFrame;
    }
    ppInData = &inFrameAddr[0];
    ppOutData = &outFrameAddr[0];


    auto pDelayState = reinterpret_cast<DelayState*>(reinterpret_cast<uint64_t>(pOutDelay->_workBuffer) + sizeof(DelayParameter));
    ApplyDelayEffect(pDelayParameter, pDelayState, notByPassed, ppInData, ppOutData, samplesPerFrame);
    if(pDelayParameter->_parameterStatus == EffectParameterStatus_Init || pDelayParameter->_parameterStatus == EffectParameterStatus_UpdateParam)
    {
        pDelayParameter->_parameterStatus = EffectParameterStatus_Updated;
    }
}

nn::TimeSpan GetAuxDelayTimeMax(const AuxDelayType* pDelay) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pDelay);
    auto pDelayParameter = reinterpret_cast<DelayParameter*>(pDelay->_workBuffer);
    return nn::TimeSpan::FromMilliSeconds(pDelayParameter->_delayTimeMax);
}

void ResetAuxDelay(AuxDelayType* pOutDelay) NN_NOEXCEPT
{
    NN_SDK_REQUIRES_NOT_NULL(pOutDelay);
    auto pDelayParameter = reinterpret_cast<DelayParameter*>(pOutDelay->_workBuffer);
    pDelayParameter->_parameterStatus = EffectParameterStatus_Init;
}
}  // namespace audio
}  // namespace nn

