﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>

#include <nn/apm/apm_Types.h>
#include <nn/apm/server/apm_Types.h>
#include <nn/apm/server/apm_Configuration.h>
#include <nn/fgm/fgm.h>

#include "apm_PerformanceRequests.h"

namespace nn { namespace apm { namespace server {

const nn::fgm::Setting RequestManager::DefaultCpuSetting = 1020000000;
const nn::fgm::Setting RequestManager::DefaultGpuSetting = 307200000;
const nn::fgm::Setting RequestManager::DefaultEmcSetting = 1331200000;

RequestManager::RequestManager() NN_NOEXCEPT
    : m_RequestCpu()
    , m_RequestGpu()
    , m_RequestEmc()
    , m_CpuFixedSettingEnabled(false)
    , m_CpuFixedSetting(DefaultCpuSetting)
{
    // 何もしません。
}

RequestManager::~RequestManager() NN_NOEXCEPT
{
    // 何もしません。
}

void RequestManager::Initialize() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_RequestCpu.Initialize(nn::fgm::Module_Cpu, nn::fgm::Priority_Default));
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_RequestGpu.Initialize(nn::fgm::Module_Gpu, nn::fgm::Priority_Default));
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_RequestEmc.Initialize(nn::fgm::Module_Emc, nn::fgm::Priority_Default));
}

void RequestManager::Finalize() NN_NOEXCEPT
{
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_RequestCpu.Finalize());
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_RequestGpu.Finalize());
    NN_ABORT_UNLESS_RESULT_SUCCESS(m_RequestEmc.Finalize());
}

void RequestManager::BeginCpuFixedSetting(nn::fgm::Setting setting) NN_NOEXCEPT
{
    m_CpuFixedSetting = setting;
    m_CpuFixedSettingEnabled = true;
}

void RequestManager::EndCpuFixedSetting() NN_NOEXCEPT
{
    m_CpuFixedSettingEnabled = false;
}

void RequestManager::Update(nn::apm::PerformanceConfiguration config) NN_NOEXCEPT
{
    for ( const ConfigurationEntry& entry : ConfigurationTable )
    {
        if ( entry.config == config )
        {
            if ( m_CpuFixedSettingEnabled )
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_RequestCpu.SetAndWait(m_CpuFixedSetting, m_CpuFixedSetting));
            }
            else
            {
                NN_ABORT_UNLESS_RESULT_SUCCESS(m_RequestCpu.SetAndWait(entry.cpuMin, entry.cpuMax));
            }
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_RequestGpu.SetAndWait(entry.gpuMin, entry.gpuMax));
            NN_ABORT_UNLESS_RESULT_SUCCESS(m_RequestEmc.SetAndWait(entry.emcMin, entry.emcMax));
        }
    }
}

bool RequestManager::IsThrottlingNeeded(PerformanceConfiguration performanceConfiguration) NN_NOEXCEPT
{
    for ( const ConfigurationEntry& entry : ConfigurationTable )
    {
        if ( entry.config == performanceConfiguration )
        {
            return entry.throttlingNeeded;
        }
    }

    return true;
}

nn::fgm::Setting RequestManager::GetCpuMinSetting(PerformanceConfiguration performanceConfiguration) NN_NOEXCEPT
{
    if ( m_CpuFixedSettingEnabled )
    {
        return m_CpuFixedSetting;
    }

    for ( const ConfigurationEntry& entry : ConfigurationTable )
    {
        if ( entry.config == performanceConfiguration )
        {
            return entry.cpuMin;
        }
    }

    return DefaultCpuSetting;
}

nn::fgm::Setting RequestManager::GetGpuMinSetting(PerformanceConfiguration performanceConfiguration) NN_NOEXCEPT
{
    for ( const ConfigurationEntry& entry : ConfigurationTable )
    {
        if ( entry.config == performanceConfiguration )
        {
            return entry.gpuMin;
        }
    }

    return DefaultGpuSetting;
}

nn::fgm::Setting RequestManager::GetEmcMinSetting(PerformanceConfiguration performanceConfiguration) NN_NOEXCEPT
{
    for ( const ConfigurationEntry& entry : ConfigurationTable )
    {
        if ( entry.config == performanceConfiguration )
        {
            return entry.emcMin;
        }
    }

    return DefaultEmcSetting;
}

}}} // namespace nn::apm::server
