﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/


#include <nn/aocsrv/detail/aocsrv_AddOnContentManagerImpl.h>
#include <nn/aoc/aoc_ResultPrivate.h>

#include <nn/arp/arp_Api.h>
#include <nn/ncm/ncm_ContentMetaDatabase.h>
#include <nn/ncm/ncm_Service.h>
#include <nn/ns/ns_ApplicationManagerApi.h>
#include <nn/ns/ns_ApplicationManagerSystemApi.h>
#include <nn/ns/ns_ApplicationContentMetaSystemApi.h>
#include <nn/ns/ns_Result.h>
#include <nn/os/os_Types.h>
#include <nn/os/os_Mutex.h>
#include <nn/result/result_HandlingUtility.h>
#include <nn/lr/lr_AddOnContentLocationResolver.h>
#include <nn/lr/lr_Service.h>

#include <mutex>

namespace nn { namespace aocsrv { namespace detail {
    namespace {

        Result GetAocBaseIdByProcessId(nn::Bit64* out, nn::Bit64 processId) NN_NOEXCEPT
        {
            nn::os::ProcessId caller = { processId };

            std::unique_ptr<nn::ns::ApplicationControlProperty> acp(new nn::ns::ApplicationControlProperty());
            if (nn::arp::GetApplicationControlProperty(acp.get(), caller).IsSuccess())
            {
                *out = acp->addOnContentBaseId;
                NN_RESULT_SUCCESS;
            }

            // INFO: 0.9.0 での暫定。 acp が存在しない場合のケア
            nn::arp::ApplicationLaunchProperty alp;
            NN_RESULT_DO(nn::arp::GetApplicationLaunchProperty(&alp, caller));
            *out = alp.id.value + 0x1000;
            NN_RESULT_SUCCESS;
        }

        Result GetAocBaseIdByApplicationId(nn::Bit64* out, nn::ncm::ApplicationId applicationId) NN_NOEXCEPT
        {
            size_t bufferSize = sizeof(nn::ns::ApplicationControlProperty);
            std::unique_ptr<nn::ns::ApplicationControlProperty> acp(new nn::ns::ApplicationControlProperty());
            size_t outSize;
            if (nn::ns::GetApplicationControlData(
                &outSize, acp.get(), bufferSize, nn::ns::ApplicationControlSource::Storage, applicationId).IsSuccess())
            {
                *out = acp->addOnContentBaseId;
                NN_RESULT_SUCCESS;
            }

            // INFO: 0.9.0 での暫定。 acp が存在しない場合のケア
            *out = applicationId.value + 0x1000;
            NN_RESULT_SUCCESS;
        }

        Result RegisterAddOnContentStorage(const nn::ncm::ApplicationId& appId, const nn::ncm::DataId& aocId) NN_NOEXCEPT
        {
            nn::lr::AddOnContentLocationResolver ar;
            NN_RESULT_DO(nn::lr::OpenAddOnContentLocationResolver(&ar));

            ns::ApplicationContentMetaStatus status;

            NN_RESULT_TRY(nn::ns::GetOwnedApplicationContentMetaStatus(&status, appId, aocId.value))
                NN_RESULT_CATCH(ns::ResultContentMetaNotFound)
                {
                    NN_RESULT_THROW(aoc::ResultNotFoundAddOnContent());
                }
            NN_RESULT_END_TRY;

            // INFO: PrepareAddOnContent 後にすぐにマウントが実行されることを期待し、毎回 UnregisterAll している
            ar.UnregisterAllAddOnContentPath();

            NN_RESULT_DO(ar.RegisterAddOnContentStorage(aocId, status.installedStorage));
            NN_RESULT_SUCCESS;
        }

        Result PrepareAddOnContentImpl(const nn::ncm::ApplicationId& appId, const nn::ncm::DataId& aocId) NN_NOEXCEPT
        {
            NN_RESULT_DO(nn::ns::RegisterContentsExternalKey(appId, aocId.value));
            NN_RESULT_DO(RegisterAddOnContentStorage(appId, aocId));
            NN_RESULT_SUCCESS;
        }

        ncm::DataId MakeAocId(nn::Bit64 aocBaseId, nn::aoc::AddOnContentIndex aocIndex) NN_NOEXCEPT
        {
            return { aocBaseId + aocIndex };
        }

        Result ListOwnedAddOnContent(int* outCount, ns::ApplicationContentMetaStatus* listBuffer, int offset, int count, nn::ncm::ApplicationId targetApplication) NN_NOEXCEPT
        {
            NN_RESULT_TRY(ns::ListAvailableAddOnContent(outCount, listBuffer, count, targetApplication, offset))
                NN_RESULT_CATCH(ns::ResultApplicationRecordNotFound)
                {
                    // INFO: アプリ本体も追加コンテンツも未インストールで RunOnTarget 実行するようなケースでひっかかる
                    //       エラーにせずにリストアップのカウントを 0 にする
                    *outCount = 0;
                }
            NN_RESULT_END_TRY;

            NN_RESULT_SUCCESS;
        }

        Result GetApplicationId(nn::ncm::ApplicationId* outValue, nn::Bit64 processId) NN_NOEXCEPT
        {
            nn::os::ProcessId pid = { processId };
            nn::arp::ApplicationLaunchProperty alp;
            NN_RESULT_DO(nn::arp::GetApplicationLaunchProperty(&alp, pid));

            *outValue = alp.id;
            NN_RESULT_SUCCESS;
        }
    } // namespace

    AddOnContentManagerImpl::AddOnContentManagerImpl() NN_NOEXCEPT
        : AddOnContentManagerImpl(nullptr, 0)
    {
    }

    AddOnContentManagerImpl::AddOnContentManagerImpl(nn::ns::ApplicationContentMetaStatus* pListupBuffer, int maxListupCount) NN_NOEXCEPT
        : m_MaxListupCount(maxListupCount), m_pListBuffer(pListupBuffer), m_ListLockMutex(false)
    {
    }

    Result AddOnContentManagerImpl::CountAddOnContentByApplicationId(nn::sf::Out<std::int32_t> outCount, nn::ncm::ApplicationId targetApplication) NN_NOEXCEPT
    {
        int listCount;

        {
            std::lock_guard<nn::os::Mutex> listBufferLock(m_ListLockMutex);
            NN_RESULT_DO(ListOwnedAddOnContent(&listCount, m_pListBuffer, 0, m_MaxListupCount, targetApplication));
        }

        outCount.Set(listCount);
        NN_RESULT_SUCCESS;
    }

    Result AddOnContentManagerImpl::CountAddOnContent(nn::sf::Out<std::int32_t> outCount, nn::Bit64 processId) NN_NOEXCEPT
    {
        nn::ncm::ApplicationId appId;
        NN_RESULT_DO(GetApplicationId(&appId, processId));

        return CountAddOnContentByApplicationId(outCount, appId);
    }

    Result AddOnContentManagerImpl::ListAddOnContentByApplicationId(
        nn::sf::Out<std::int32_t>                           outCount,
        const nn::sf::OutArray<nn::aoc::AddOnContentIndex>& outIndices,
        std::int32_t                                        offset,
        std::int32_t                                        count,
        nn::ncm::ApplicationId                              targetApplication) NN_NOEXCEPT
    {
        nn::Bit64 aocBaseId;
        NN_RESULT_DO(GetAocBaseIdByApplicationId(&aocBaseId, targetApplication));

        int listCount;

        {
            std::lock_guard<nn::os::Mutex> listBufferLock(m_ListLockMutex);
            NN_RESULT_DO(ListOwnedAddOnContent(&listCount, m_pListBuffer, offset, std::min(count, m_MaxListupCount), targetApplication));

            for (int i = 0; i < listCount; ++i)
            {
                outIndices[i] = static_cast<nn::aoc::AddOnContentIndex>(m_pListBuffer[i].id - aocBaseId);
            }
        }

        *outCount = listCount;
        NN_RESULT_SUCCESS;
    }

    Result AddOnContentManagerImpl::ListAddOnContent(
        nn::sf::Out<std::int32_t>                           outCount,
        const nn::sf::OutArray<nn::aoc::AddOnContentIndex>& outIndices,
        nn::Bit64                                           processId,
        std::int32_t                                        offset,
        std::int32_t                                        count) NN_NOEXCEPT
    {
        nn::ncm::ApplicationId appId;
        NN_RESULT_DO(GetApplicationId(&appId, processId));

        return ListAddOnContentByApplicationId(outCount, outIndices, offset, count, appId);
    }

    Result AddOnContentManagerImpl::GetAddOnContentBaseIdByApplicationId(nn::sf::Out<nn::Bit64> outValue, nn::ncm::ApplicationId targetApplication) NN_NOEXCEPT
    {
        NN_RESULT_DO(GetAocBaseIdByApplicationId(outValue.GetPointer(), targetApplication));
        NN_RESULT_SUCCESS;
    }

    Result AddOnContentManagerImpl::GetAddOnContentBaseId(nn::sf::Out<nn::Bit64> outValue, nn::Bit64 processId) NN_NOEXCEPT
    {
        NN_RESULT_DO(GetAocBaseIdByProcessId(outValue.GetPointer(), processId));
        NN_RESULT_SUCCESS;
    }

    Result AddOnContentManagerImpl::PrepareAddOnContentByApplicationId(nn::aoc::AddOnContentIndex targetIndex, nn::ncm::ApplicationId targetApplication) NN_NOEXCEPT
    {
        nn::Bit64 aocBaseId;
        NN_RESULT_DO(GetAocBaseIdByApplicationId(&aocBaseId, targetApplication));
        NN_RESULT_DO(PrepareAddOnContentImpl(targetApplication, MakeAocId(aocBaseId, targetIndex)));
        NN_RESULT_SUCCESS;
    }

    Result AddOnContentManagerImpl::PrepareAddOnContent(nn::aoc::AddOnContentIndex targetIndex, nn::Bit64 processId) NN_NOEXCEPT
    {
        nn::Bit64 aocBaseId;
        NN_RESULT_DO(GetAocBaseIdByProcessId(&aocBaseId, processId));

        nn::ncm::ApplicationId appId;
        NN_RESULT_DO(GetApplicationId(&appId, processId));

        NN_RESULT_DO(PrepareAddOnContentImpl(appId, MakeAocId(aocBaseId, targetIndex)));
        NN_RESULT_SUCCESS;
    }

    Result AddOnContentManagerImpl::GetAddOnContentListChangedEvent(nn::sf::Out<nn::sf::NativeHandle> outHandle) NN_NOEXCEPT
    {
        *outHandle = ns::GetDynamicCommitEvent();
        NN_RESULT_SUCCESS;
    }

}}} // namespace nn::aocsrv::detail
