﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/window/am_WindowSystem.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <list>
#include <nn/util/util_Exchange.h>
#include <nn/am/service/am_StuckChecker.h>

namespace nn { namespace am { namespace service { namespace window {

namespace detail {
    IWindowTransiter g_NullTransiter;
}

class WindowManager::UpdatePlan
{
public:

    explicit UpdatePlan(WindowManager* pManager) NN_NOEXCEPT;
    explicit UpdatePlan(Window* pWindow) NN_NOEXCEPT;

    void ExecutePlan() NN_NOEXCEPT;
    static void ChangeTransiter(WindowManager* pManager, Window* pWindow, IWindowTransiter* pTransiter) NN_NOEXCEPT;

private:

    void CalculateExpectedStatus() NN_NOEXCEPT;

    class PlanEntry
    {
        friend UpdatePlan;
    public:

        explicit PlanEntry(Window* pWindow) NN_NOEXCEPT;

        void Set() NN_NOEXCEPT;

        void SetExpectedStatus(const WindowStatus& expectedStatus) NN_NOEXCEPT
        {
            this->m_ExpectedStatus = expectedStatus;
        }

        void SetExpectedStatusAsNotActive() NN_NOEXCEPT
        {
            WindowController::SetNotActive(&m_ExpectedStatus);
        }

        bool operator<(const PlanEntry& rhs) const NN_NOEXCEPT
        {
            auto& lhs = *this;
            if (lhs.m_IsActive != rhs.m_IsActive)
            {
                // isActive が true なほうが前
                return lhs.m_IsActive;
            }
            else if (lhs.m_WindowProperty.globalOrder != rhs.m_WindowProperty.globalOrder)
            {
                // m_GlobalOrder が小さいほうが前
                return static_cast<int>(lhs.m_WindowProperty.globalOrder) < static_cast<int>(rhs.m_WindowProperty.globalOrder);
            }
            else if (lhs.m_Order != rhs.m_Order)
            {
                // order が大きいほうが前
                return lhs.m_Order > rhs.m_Order;
            }
            else
            {
                // subOrder が大きいほうが前
                return lhs.m_SubOrder > rhs.m_SubOrder;
            }
        }

        // CalculateExpectedStatus 用
        bool IsActive() const
        {
            return m_IsActive;
        }

        const WindowProperty& GetWindowProperty() NN_NOEXCEPT
        {
            return m_WindowProperty;
        }

        WindowStatus& RefExpectedStatus() NN_NOEXCEPT
        {
            return m_ExpectedStatus;
        }

    private:

        void BeginUpdate() NN_NOEXCEPT;
        void UpdateNegativeImmediately() NN_NOEXCEPT;
        void UpdateNegative() NN_NOEXCEPT;
        void UpdateImmediately() NN_NOEXCEPT;
        void UpdatePositive() NN_NOEXCEPT;
        void UpdatePositiveImmediately() NN_NOEXCEPT;
        void EndUpdate() NN_NOEXCEPT;

        Window* const m_pWindow;
        bool m_IsActive;
        int64_t m_Order;
        uint32_t m_SubOrder;
        WindowProperty m_WindowProperty;
        WindowStatus m_ExpectedStatus;

    };

    WindowManager* m_pManager;
    std::list<PlanEntry, StaticAllocator<PlanEntry>> m_Entries;

};

inline WindowManager::UpdatePlan::PlanEntry::PlanEntry(Window* pWindow) NN_NOEXCEPT
    : m_pWindow(pWindow)
    , m_IsActive(pWindow->m_IsActive)
    , m_Order(pWindow->m_pGroup ? pWindow->m_pGroup->m_Order : 0)
    , m_SubOrder(pWindow->m_SubOrder)
    , m_WindowProperty(pWindow->GetProperty())
{
}

inline void WindowManager::UpdatePlan::PlanEntry::BeginUpdate() NN_NOEXCEPT
{
    // 明示的な設定をしないときのデフォルト値
    this->m_ExpectedStatus = m_pWindow->GetCurrentStatus();
}

inline void WindowManager::UpdatePlan::PlanEntry::UpdateNegativeImmediately() NN_NOEXCEPT
{
    WindowController::UpdateNegativeImmediately(m_pWindow, m_pWindow->GetCurrentStatus(), m_ExpectedStatus);
}

inline void WindowManager::UpdatePlan::PlanEntry::UpdateNegative() NN_NOEXCEPT
{
    WindowController::UpdateNegative(m_pWindow, m_pWindow->GetCurrentStatus(), m_ExpectedStatus);
}

inline void WindowManager::UpdatePlan::PlanEntry::UpdateImmediately() NN_NOEXCEPT
{
    WindowController::UpdateImmediately(m_pWindow, m_pWindow->GetCurrentStatus(), m_ExpectedStatus);
}

inline void WindowManager::UpdatePlan::PlanEntry::UpdatePositive() NN_NOEXCEPT
{
    WindowController::UpdatePositive(m_pWindow, m_pWindow->GetCurrentStatus(), m_ExpectedStatus);
}

inline void WindowManager::UpdatePlan::PlanEntry::UpdatePositiveImmediately() NN_NOEXCEPT
{
    WindowController::UpdatePositiveImmediately(m_pWindow, m_pWindow->GetCurrentStatus(), m_ExpectedStatus);
}

inline void WindowManager::UpdatePlan::PlanEntry::EndUpdate() NN_NOEXCEPT
{
    m_pWindow->SetCurrentStatus(m_ExpectedStatus);
}


WindowManager::UpdatePlan::UpdatePlan(WindowManager* pManager) NN_NOEXCEPT
    : m_pManager(pManager)
{
    {
        auto lk2 = pManager->AcquireStructureLock();
        auto lk3 = pManager->AcquireUpdateLock();
        for (auto&& w : pManager->m_Windows)
        {
            m_Entries.emplace_back(&w);
        }
    }
    m_Entries.sort();
}

WindowManager::UpdatePlan::UpdatePlan(Window* pWindow) NN_NOEXCEPT
    : m_pManager(pWindow->m_pManager)
{
    auto lk2 = m_pManager->AcquireStructureLock();
    auto lk3 = m_pManager->AcquireUpdateLock();
    m_Entries.emplace_back(pWindow);
}

void WindowManager::UpdatePlan::ExecutePlan() NN_NOEXCEPT
{
    for (auto&& e: m_Entries)
    {
        e.BeginUpdate();
    }
    CalculateExpectedStatus();
    {
        // immediate は、メッセージ処理と排他される
        auto lk4 = m_pManager->AcquireWindowEventStatusLock();
        for (auto&& e: m_Entries)
        {
            e.UpdateNegativeImmediately();
        }
    }
    for (auto&& e: m_Entries)
    {
        e.UpdateNegative();
    }
    {
        // immediate は、メッセージ処理と排他される
        auto lk4 = m_pManager->AcquireWindowEventStatusLock();
        for (auto&& e: m_Entries)
        {
            e.UpdateImmediately();
        }
    }
    for (auto&& e: m_Entries)
    {
        e.UpdatePositive();
    }
    {
        // immediate は、メッセージ処理と排他される
        auto lk4 = m_pManager->AcquireWindowEventStatusLock();
        for (auto&& e: m_Entries)
        {
            e.UpdatePositiveImmediately();
        }
    }
    for (auto&& e: m_Entries)
    {
        e.EndUpdate();
    }
}

void WindowManager::UpdatePlan::CalculateExpectedStatus() NN_NOEXCEPT
{
    WindowController::CalculationContext context;
    m_pManager->m_Controller.InitializeCalculationContext(&context);
    for (auto&& e: m_Entries)
    {
        if (e.IsActive())
        {
            m_pManager->m_Controller.CalculateStatus(&e.RefExpectedStatus(), e.GetWindowProperty(), e.GetWindowProperty(), &context);
        }
        else
        {
            e.SetExpectedStatusAsNotActive();
        }
    }
}

void WindowManager::UpdatePlan::ChangeTransiter(WindowManager* pManager, Window* pWindow, IWindowTransiter* pTransiter) NN_NOEXCEPT
{
    WindowStatus status;
    {
        PlanEntry e{pWindow};
        e.BeginUpdate();
        status = e.m_ExpectedStatus;
        e.SetExpectedStatusAsNotActive();
        {
            // immediate は、メッセージ処理と排他される
            auto lk4 = pManager->AcquireWindowEventStatusLock();
            e.UpdateNegativeImmediately();
        }
        e.UpdateNegative();
        {
            // immediate は、メッセージ処理と排他される
            auto lk4 = pManager->AcquireWindowEventStatusLock();
            e.UpdateImmediately();
        }
        e.UpdatePositive();
        {
            // immediate は、メッセージ処理と排他される
            auto lk4 = pManager->AcquireWindowEventStatusLock();
            e.UpdatePositiveImmediately();
        }
        e.EndUpdate();
    }
    {
        auto lk3 = pManager->AcquireUpdateLock();
        pWindow->m_pTransiter = pTransiter;
    }
    {
        PlanEntry e{pWindow};
        e.BeginUpdate();
        e.m_ExpectedStatus = status;
        {
            // immediate は、メッセージ処理と排他される
            auto lk4 = pManager->AcquireWindowEventStatusLock();
            e.UpdateNegativeImmediately();
        }
        e.UpdateNegative();
        {
            // immediate は、メッセージ処理と排他される
            auto lk4 = pManager->AcquireWindowEventStatusLock();
            e.UpdateImmediately();
        }
        e.UpdatePositive();
        {
            // immediate は、メッセージ処理と排他される
            auto lk4 = pManager->AcquireWindowEventStatusLock();
            e.UpdatePositiveImmediately();
        }
        e.EndUpdate();
    }
}


void WindowManager::ChangeWindowTransiter(Window* pWindow, IWindowTransiter* pTransiter) NN_NOEXCEPT
{
    // atomic に行う
    //   この関数は IPC スレッドから呼ばれる前提のものであり、
    //   本来はほぼブロックする可能性のない AcquireWindowAccessLock() を取りたいが、
    //   現状の構造においてはどうにもならないため、
    //   ブロックする可能性のある AcquireWindowDeletionLock() を呼んでいる。
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(am_WindowSystem_ChangeWindowTransiter, 120);
    auto lk1 = AcquireWindowDeletionLock();
    UpdatePlan::ChangeTransiter(this, pWindow, pTransiter);
}

void WindowManager::Run() NN_NOEXCEPT
{
    for (;;)
    {
        m_UpdateEvent.Wait();
        NN_AM_SERVICE_SCOPED_STUCK_CHECK(am_WindowSystem_Run, 120);
        auto lk1 = AcquireWindowAccessLock();
        UpdatePlan{this}.ExecutePlan();
    }
}


template <typename Controller>
void WindowManager::NotifyEvent(const typename Controller::EventType& e) NN_NOEXCEPT
{
    NN_AM_SERVICE_SCOPED_STUCK_CHECK(am_WindowSystem_NotifyEvent, 120);
    auto lk1 = AcquireWindowAccessLock();
    auto lk2 = AcquireStructureLock();
    auto lk4 = AcquireWindowEventStatusLock();
    for (auto&& w : m_Windows)
    {
        static_cast<Controller&>(m_Controller).HandleEvent(&w, w.RefEventStatus(), e);
    }
}

void WindowManager::NotifyHomeButtonShortPressed() NN_NOEXCEPT
{
    HomeButtonWindowController::EventType e = {};
    e.onPressedShort = true;
    NotifyEvent<HomeButtonWindowController>(e);
}

void WindowManager::NotifyHomeButtonLongPressed() NN_NOEXCEPT
{
    HomeButtonWindowController::EventType e = {};
    e.onPressedLong = true;
    NotifyEvent<HomeButtonWindowController>(e);
}

void WindowManager::NotifyPowerButtonShortPressed() NN_NOEXCEPT
{
    PowerButtonWindowController::EventType e = {};
    e.onPressedShort = true;
    NotifyEvent<PowerButtonWindowController>(e);
}

void WindowManager::NotifyPowerButtonMiddlePressed() NN_NOEXCEPT
{
    PowerButtonWindowController::EventType e = {};
    e.onPressedMiddle = true;
    NotifyEvent<PowerButtonWindowController>(e);
}

void WindowManager::NotifyPowerButtonLongPressed() NN_NOEXCEPT
{
    PowerButtonWindowController::EventType e = {};
    e.onPressedLong = true;
    NotifyEvent<PowerButtonWindowController>(e);
}

void WindowManager::NotifyCaptureButtonShortPressed() NN_NOEXCEPT
{
    CaptureButtonWindowController::EventType e = {};
    e.onPressedShort = true;
    NotifyEvent<CaptureButtonWindowController>(e);
}

void WindowManager::NotifyCaptureButtonLongPressed() NN_NOEXCEPT
{
    CaptureButtonWindowController::EventType e = {};
    e.onPressedLong = true;
    NotifyEvent<CaptureButtonWindowController>(e);
}

void WindowManager::NotifyOperationModeChanged() NN_NOEXCEPT
{
    OmmEventWindowController::EventType e = {};
    e.onOperationModeChanged = true;
    NotifyEvent<OmmEventWindowController>(e);
}

void WindowManager::NotifyPerformanceModeChanged() NN_NOEXCEPT
{
    ApmEventWindowController::EventType e = {};
    e.onPerformanceModeChanged = true;
    NotifyEvent<ApmEventWindowController>(e);
}

void WindowManager::NotifyIdlePolicyChanged() NN_NOEXCEPT
{
    IdleWindowController::EventType e = {};
    e.onIdlePolicyChanged = true;
    NotifyEvent<IdleWindowController>(e);
}

void WindowManager::NotifySleepRequiredByHighTemperature() NN_NOEXCEPT
{
    SpsmEventWindowController::EventType e = {};
    e.onSleepRequiredByHighTemperature = true;
    NotifyEvent<SpsmEventWindowController>(e);
}

void WindowManager::NotifySleepRequiredByLowBattery() NN_NOEXCEPT
{
    SpsmEventWindowController::EventType e = {};
    e.onSleepRequiredByLowBattery = true;
    NotifyEvent<SpsmEventWindowController>(e);
}

void WindowManager::NotifyAutoPowerDown() NN_NOEXCEPT
{
    SpsmEventWindowController::EventType e = {};
    e.onAutoPowerDown = true;
    NotifyEvent<SpsmEventWindowController>(e);
}

void WindowManager::NotifyCecSystemStandbyReceived() NN_NOEXCEPT
{
    CecWindowController::EventType e = {};
    e.onSystemStandbyReceived = true;
    NotifyEvent<CecWindowController>(e);
}

void WindowManager::NotifySdCardRemoved() NN_NOEXCEPT
{
    NN_AM_SERVICE_LOG(call, "NotifySdCardRemoved()\n");
    SdCardEventWindowController::EventType e = {};
    e.onSdCardRemoved = true;
    NotifyEvent<SdCardEventWindowController>(e);
}


void WindowManager::NotifyGameCardStateChanged() NN_NOEXCEPT
{
    GameCardEventWindowController::EventType e = {};
    e.gameCardStateChanged = true;
    NotifyEvent<GameCardEventWindowController>(e);
}

void WindowManager::NotifyProcessException(const os::ProcessId& processId) NN_NOEXCEPT
{
    ForegroundWindowController::EventType e = {};
    e.pExceptionProcessId = processId;
    NotifyEvent<ForegroundWindowController>(e);
}


void WindowManager::NotifyFatal(bool lockForever) NN_NOEXCEPT
{
    SuspendMonitoring();
    NN_AM_SERVICE_LOG(error, "suspending all processes...\n");
    ForegroundWindowController::EventType e = {};
    e.onFatal = true;
    NotifyEvent<ForegroundWindowController>(e);
    NN_AM_SERVICE_LOG(error, "suspending all processes is done.\n");
    if (lockForever)
    {
        NN_AM_SERVICE_LOG(error, "lock window manager and sleep forever.\n");
        auto lk1 = AcquireWindowAccessLock();
        auto lk2 = AcquireStructureLock();
        auto lk3 = AcquireUpdateLock();
        auto lk4 = AcquireWindowEventStatusLock();
        for (;;)
        {
            os::SleepThread(TimeSpan::FromDays(1));
        }
    }
}

}}}}
