﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/display/am_ScreenShotRequestQueue.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>

namespace nn{ namespace am{ namespace service{ namespace display{

    void ScreenShotRequestQueue::Initialize(DisplayWorker* pWorker, void(*function)(const ScreenShotRequest&, void*), void* userPtr) NN_NOEXCEPT
    {
        NN_SDK_REQUIRES_NOT_NULL(pWorker);
        NN_SDK_REQUIRES_NOT_NULL(function);

        nn::os::InitializeMutex(&m_Mutex, false, 0);
        nn::os::InitializeEvent(&m_HasEntryEvent, false, nn::os::EventClearMode_ManualClear);
        nn::os::InitializeMultiWaitHolder(&m_HasEntryHolder, &m_HasEntryEvent);

        NN_ABORT_UNLESS_NOT_NULL(pWorker->LinkMultiWait(&m_HasEntryHolder, OnRequestEnqueuedCallbackFunction, this));

        m_pWorker = pWorker;
        m_WorkerFunction = function;
        m_WorkerUserPointer = userPtr;
    }

    void ScreenShotRequestQueue::Finalize() NN_NOEXCEPT
    {
        m_pWorker->UnlinkMultiWait(&m_HasEntryHolder);

        nn::os::FinalizeMultiWaitHolder(&m_HasEntryHolder);
        nn::os::FinalizeEvent(&m_HasEntryEvent);
        nn::os::FinalizeMutex(&m_Mutex);

        m_pWorker = nullptr;
        m_WorkerFunction = nullptr;
        m_WorkerUserPointer = nullptr;
    }

    void ScreenShotRequestQueue::EnqueueRequest(const ScreenShotRequest& request) NN_NOEXCEPT
    {
        nn::os::LockMutex(&m_Mutex);
        NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };

        m_Entry = request;
        nn::os::SignalEvent(&m_HasEntryEvent);
    }

    void ScreenShotRequestQueue::OnRequestEnqueuedCallbackFunction(nn::os::MultiWaitHolderType* pHolder, void* arg) NN_NOEXCEPT
    {
        auto p = reinterpret_cast<ScreenShotRequestQueue*>(arg);
        p->OnRequestEnqueuedImpl();

        // 再度待ち
        NN_ABORT_UNLESS_NOT_NULL(p->m_pWorker->LinkMultiWait(pHolder, OnRequestEnqueuedCallbackFunction, arg));
    }

    void ScreenShotRequestQueue::OnRequestEnqueuedImpl() NN_NOEXCEPT
    {
        ScreenShotRequest request = {};

        {
            nn::os::LockMutex(&m_Mutex);
            NN_UTIL_SCOPE_EXIT{ nn::os::UnlockMutex(&m_Mutex); };
            NN_UTIL_SCOPE_EXIT{ nn::os::ClearEvent(&m_HasEntryEvent); };

            if(!m_Entry)
            {
                return;
            }

            request = *m_Entry;
        }

        m_WorkerFunction(request, m_WorkerUserPointer);
    }

}}}}

