﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/display/am_IntegratedDisplayLayer.h>

#include <nn/result/result_HandlingUtility.h>
#include <nn/util/util_ScopeExit.h>
#include <nn/am/service/display/am_DisplayControlLog.h>
#include <nn/am/service/display/am_DisplayPrimitiveOperation.h>

namespace nn{ namespace am{ namespace service{ namespace display{
    namespace detail{

        IntegratedDisplayLayer::IntegratedDisplayLayer() NN_NOEXCEPT
        {
            std::memset(this, 0, sizeof(*this));
        }

        IntegratedDisplayLayerState IntegratedDisplayLayer::GetState() const NN_NOEXCEPT
        {
            return m_State;
        }

        IntegratedDisplayLayerUserClass IntegratedDisplayLayer::GetUserClass() const NN_NOEXCEPT
        {
            return m_UserClass;
        }

        IntegratedDisplayLayerBufferOwner IntegratedDisplayLayer::GetBufferOwner() const NN_NOEXCEPT
        {
            return m_BufferOwner;
        }

        IntegratedDisplayLayerDestinationLayer IntegratedDisplayLayer::GetDestinationLayer() const NN_NOEXCEPT
        {
            return m_DestinationLayer;
        }

        applet::AppletResourceUserId IntegratedDisplayLayer::GetAruid() const NN_NOEXCEPT
        {
            return m_Aruid;
        }

        nn::vi::IndirectLayerHandleType IntegratedDisplayLayer::GetIndirectLayerHandle() const NN_NOEXCEPT
        {
            return m_IndirectLayerHandle;
        }

        nn::vi::LayerId IntegratedDisplayLayer::GetDirectLayerId() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_State != IntegratedDisplayLayerState_NotInitialized);
            return m_DirectLayerId;
        }

        nn::vi::IndirectProducerHandleType IntegratedDisplayLayer::GetIndirectProducerHandle() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_State != IntegratedDisplayLayerState_NotInitialized);
            return m_IndirectProducerHandle;
        }

        nn::vi::fbshare::SharedLayerHandle IntegratedDisplayLayer::GetSharedLayerHandle() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_State != IntegratedDisplayLayerState_NotInitialized);
            return m_SharedLayerHandle;
        }


        nn::os::SystemEventType* IntegratedDisplayLayer::GetSharedLayerDetachReadyEvent() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_State != IntegratedDisplayLayerState_NotInitialized);
            NN_SDK_REQUIRES(!m_SharedLayerHandle.IsInvalid());
            return &m_SharedLayerDetachReadyEvent;
        }

        nn::vi::LayerId IntegratedDisplayLayer::GetIndirectProducerHandleForSharedLayer() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_State != IntegratedDisplayLayerState_NotInitialized);
            return static_cast<nn::vi::LayerId>(m_IndirectProducerHandleForSharedLayer);
        }

        bool IntegratedDisplayLayer::IsVisible() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_State != IntegratedDisplayLayerState_NotInitialized);
            return m_IsVisible;
        }

        int32_t IntegratedDisplayLayer::GetZOrder() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_State != IntegratedDisplayLayerState_NotInitialized);
            return m_ZOrder;
        }

        bool IntegratedDisplayLayer::IsCaptureTarget() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_State != IntegratedDisplayLayerState_NotInitialized);
            return m_IsCaptureTarget;
        }

        bool IntegratedDisplayLayer::IsTransitionLayerRequested() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_State != IntegratedDisplayLayerState_NotInitialized);
            return m_IsTransitionLayerRequested;
        }

        bool IntegratedDisplayLayer::IsConductorIfForeground() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_State != IntegratedDisplayLayerState_NotInitialized);
            return m_IsConductorIfForeground;
        }

        bool IntegratedDisplayLayer::IsRecordingLayerStackRequested() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_State != IntegratedDisplayLayerState_NotInitialized);
            return m_IsRecordingLayerStackRequested;
        }

        bool IntegratedDisplayLayer::IsDebugLayerStackRequested() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_State != IntegratedDisplayLayerState_NotInitialized);
            return m_IsDebugLayerStackRequested;
        }

        bool IntegratedDisplayLayer::IsCallerAppletCaptureRequested() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_State != IntegratedDisplayLayerState_NotInitialized);
            return m_IsCallerAppletCaptureRequested;
        }

        bool IntegratedDisplayLayer::IsCallerAppletCaptureScreenshotPermitted() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_State != IntegratedDisplayLayerState_NotInitialized);
            return m_IsCallerAppletCaptureScreenshotPermitted;
        }

        bool IntegratedDisplayLayer::IsSharedBufferExported() const NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_State != IntegratedDisplayLayerState_NotInitialized);
            return !m_ExportedSharedBufferHandle.IsInvalid();
        }

        bool IntegratedDisplayLayer::InvokeScreenShotPermissionGetterFunction() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_State != IntegratedDisplayLayerState_NotInitialized);
            if(m_pScreenShotPermissionGetterFunction == nullptr)
            {
                return false;
            }
            return m_pScreenShotPermissionGetterFunction(m_pScreenShotPermissionGetterUserPtr);
        }

        void IntegratedDisplayLayer::Initialize() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES(m_State == IntegratedDisplayLayerState_NotInitialized);
            m_State = IntegratedDisplayLayerState_Undetermined;
            m_UserClass = IntegratedDisplayLayerUserClass_Unspecified;
            m_BufferOwner = IntegratedDisplayLayerBufferOwner_Unspecified;
            m_DestinationLayer = IntegratedDisplayLayerDestinationLayer_Unspecified;
            m_Aruid = applet::AppletResourceUserId::GetInvalidId();
            m_SharedBufferHandle = {};
            m_IndirectLayerHandle = {};
            m_IndirectProducerFlipOffset = {};
            m_pScreenShotPermissionGetterFunction = nullptr;
            m_pScreenShotPermissionGetterUserPtr = nullptr;

            m_IsVisible = false;
            m_ZOrder = IntegratedDisplayZOrder_Default;
            m_IsCaptureTarget = false;
            m_IsTransitionLayerRequested = false;
            m_IsConductorIfForeground = false;
            m_IsRecordingLayerStackRequested = false;
            m_IsDebugLayerStackRequested = false;
            m_IsCallerAppletCaptureRequested = false;
            m_IsCallerAppletCaptureScreenshotPermitted = false;

            m_DirectLayerId = 0;
            m_IndirectProducerHandle = 0;
            m_SharedLayerHandle = {};
            m_ExportedSharedBufferHandle = {};
            m_IndirectProducerHandleForSharedLayer = 0;

            NN_AM_DISPCTRL_LOG_TRACE("initialized IntegratedDisplayLayer\n");
        }

        void IntegratedDisplayLayer::Finalize() NN_NOEXCEPT
        {
            NN_AM_DISPCTRL_LOG_TRACE("finalizing IntegratedDisplayLayer\n");
            NN_SDK_REQUIRES(m_State != IntegratedDisplayLayerState_NotInitialized);

            // オブジェクトの破棄
            if(m_DirectLayerId)
            {
                NN_AM_DISPCTRL_LOG_TRACE("detaching presentation tracer\n");
                if(m_UserClass == IntegratedDisplayLayerUserClass_Application)
                {
                    DisplayPrimitiveOperation::DetachDisplayLayerPresentationTracer(m_DirectLayerId);
                }
                NN_AM_DISPCTRL_LOG_TRACE("destroying direct layer\n");
                DisplayPrimitiveOperation::DestroyDisplayLayer(m_DirectLayerId);
                m_DirectLayerId = 0;
            }
            if(m_IndirectLayerHandle && m_IndirectProducerHandle)
            {
                NN_AM_DISPCTRL_LOG_TRACE("destroying indirect producer(for applet's buffer)\n");
                DisplayPrimitiveOperation::DestroyIndirectProducerEndPoint(m_IndirectLayerHandle, m_IndirectProducerHandle);
                m_IndirectProducerHandle = 0;
            }
            if(!m_SharedLayerHandle.IsInvalid())
            {
                NN_AM_DISPCTRL_LOG_TRACE("destroying shared layer\n");
                DisplayPrimitiveOperation::FinalizeSharedLayerDetachReadyEvent(&m_SharedLayerDetachReadyEvent, m_SharedLayerHandle);
                DisplayPrimitiveOperation::DestroySharedLayer(m_SharedLayerHandle);
                m_SharedLayerHandle = {};
            }
            if(!m_ExportedSharedBufferHandle.IsInvalid())
            {
                NN_AM_DISPCTRL_LOG_TRACE("unregister shared buffer importer\n");
                DisplayPrimitiveOperation::UnregisterSharedBufferImporter(m_ExportedSharedBufferHandle, m_Aruid);
                m_ExportedSharedBufferHandle = {};
            }
            if(m_IndirectLayerHandle && m_IndirectProducerHandleForSharedLayer)
            {
                NN_AM_DISPCTRL_LOG_TRACE("destroying indirect producer(for shared buffer)\n");
                DisplayPrimitiveOperation::DisconnectSharedLowLevelLayerFromIndirectLayer(m_IndirectProducerHandleForSharedLayer);
                DisplayPrimitiveOperation::UnbindSharedLowLevelLayerFromIndirectLayer(m_IndirectProducerHandleForSharedLayer);
                DisplayPrimitiveOperation::DestroyIndirectProducerEndPoint(m_IndirectLayerHandle, m_IndirectProducerHandleForSharedLayer);
                m_IndirectProducerHandleForSharedLayer = 0;
            }

            // パラメータと状態をリセット
            m_State = IntegratedDisplayLayerState_NotInitialized;
            m_UserClass = IntegratedDisplayLayerUserClass_Unspecified;
            m_BufferOwner = IntegratedDisplayLayerBufferOwner_Unspecified;
            m_DestinationLayer = IntegratedDisplayLayerDestinationLayer_Unspecified;
            m_Aruid = applet::AppletResourceUserId::GetInvalidId();
            m_SharedBufferHandle = {};
            m_IndirectLayerHandle = {};
            m_IndirectProducerFlipOffset = {};
            m_pScreenShotPermissionGetterFunction = nullptr;
            m_pScreenShotPermissionGetterUserPtr = nullptr;

            m_IsVisible = false;
            m_ZOrder = IntegratedDisplayZOrder_Default;
            m_IsCaptureTarget = false;
            m_IsTransitionLayerRequested = false;
            m_IsConductorIfForeground = false;
            m_IsRecordingLayerStackRequested = false;
            m_IsDebugLayerStackRequested = false;
            m_IsCallerAppletCaptureRequested = false;
            m_IsCallerAppletCaptureScreenshotPermitted = false;

            NN_AM_DISPCTRL_LOG_TRACE("finalized IntegratedDisplayLayer\n");
        }

        bool IntegratedDisplayLayer::SpecifyUserClass(IntegratedDisplayLayerUserClass value, applet::AppletResourceUserId aruid) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_EQUAL(GetState(), IntegratedDisplayLayerState_Undetermined);
            NN_SDK_REQUIRES_EQUAL(GetUserClass(), IntegratedDisplayLayerUserClass_Unspecified);
            m_UserClass = value;
            m_Aruid = aruid;
            return SetupIfParameterDeterminedImpl();
        }

        bool IntegratedDisplayLayer::SpecifyBufferOwner(
            IntegratedDisplayLayerBufferOwner value,
            nn::vi::fbshare::SharedBufferHandle hBuffer
        ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_EQUAL(GetState(), IntegratedDisplayLayerState_Undetermined);
            NN_SDK_REQUIRES_EQUAL(GetBufferOwner(), IntegratedDisplayLayerBufferOwner_Unspecified);
            m_BufferOwner = value;
            m_SharedBufferHandle = hBuffer;
            return SetupIfParameterDeterminedImpl();
        }

        bool IntegratedDisplayLayer::SpecifyDestinationLayer(
            IntegratedDisplayLayerDestinationLayer value,
            nn::vi::IndirectLayerHandleType indirectLayerHandle,
            nn::TimeSpan indirectLayerFlipOffset
        ) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_EQUAL(GetState(), IntegratedDisplayLayerState_Undetermined);
            NN_SDK_REQUIRES_EQUAL(GetDestinationLayer(), IntegratedDisplayLayerDestinationLayer_Unspecified);
            m_DestinationLayer = value;
            m_IndirectLayerHandle = indirectLayerHandle;
            m_IndirectProducerFlipOffset = indirectLayerFlipOffset;
            return SetupIfParameterDeterminedImpl();
        }

        void IntegratedDisplayLayer::SetScreenShotPermissionGetterFunction(bool (*func)(void*), void* userPtr) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_EQUAL(GetState(), IntegratedDisplayLayerState_NotInitialized);
            m_pScreenShotPermissionGetterFunction = func;
            m_pScreenShotPermissionGetterUserPtr = userPtr;
        }

        void IntegratedDisplayLayer::SetVisibility(bool isVisible, int32_t zOrder, bool isCaptureTarget) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_EQUAL(GetState(), IntegratedDisplayLayerState_NotInitialized);
            m_IsVisible = isVisible;
            m_ZOrder = zOrder;
            m_IsCaptureTarget = isCaptureTarget;
        }

        void IntegratedDisplayLayer::SetTransitionLayerRequested(bool value) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_EQUAL(GetState(), IntegratedDisplayLayerState_NotInitialized);
            m_IsTransitionLayerRequested = value;
        }

        void IntegratedDisplayLayer::SetConductorIfForeground(bool value) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_EQUAL(GetState(), IntegratedDisplayLayerState_NotInitialized);
            m_IsConductorIfForeground = value;
        }

        void IntegratedDisplayLayer::SetRecordingLayerStackRequested(bool value) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_EQUAL(GetState(), IntegratedDisplayLayerState_NotInitialized);
            m_IsRecordingLayerStackRequested = value;
        }

        void IntegratedDisplayLayer::SetDebugLayerStackRequested(bool value) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_EQUAL(GetState(), IntegratedDisplayLayerState_NotInitialized);
            m_IsDebugLayerStackRequested = value;
        }

        void IntegratedDisplayLayer::SetCallerAppletCaptureRequested(bool value, bool isScreenshotPermitted) NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_NOT_EQUAL(GetState(), IntegratedDisplayLayerState_NotInitialized);
            m_IsCallerAppletCaptureRequested = value;
            m_IsCallerAppletCaptureScreenshotPermitted = isScreenshotPermitted;
        }

        void IntegratedDisplayLayer::ExportSharedBuffer(nn::vi::fbshare::SharedBufferHandle hBuffer) NN_NOEXCEPT
        {
            //NN_SDK_REQUIRES_EQUAL(GetState(), IntegratedDisplayLayerState_Determined);
            NN_SDK_REQUIRES(!IsSharedBufferExported());
            if(m_Aruid == applet::AppletResourceUserId::GetInvalidId())
            {
                return;
            }
            DisplayPrimitiveOperation::RegisterSharedBufferImporter(hBuffer, m_Aruid);
            m_ExportedSharedBufferHandle = hBuffer;
        }

        //-------------------

        bool IntegratedDisplayLayer::SetupIfParameterDeterminedImpl() NN_NOEXCEPT
        {
            NN_SDK_REQUIRES_EQUAL(GetState(), IntegratedDisplayLayerState_Undetermined);

            if(GetUserClass() == IntegratedDisplayLayerUserClass_Unspecified)
            {
                return false;
            }

            if(GetBufferOwner() == IntegratedDisplayLayerBufferOwner_Unspecified)
            {
                return false;
            }

            if(GetDestinationLayer() == IntegratedDisplayLayerDestinationLayer_Unspecified)
            {
                return false;
            }

            NN_AM_DISPCTRL_LOG_TRACE("all parameters are determined\n");
            NN_AM_DISPCTRL_LOG_TRACE("  user-class   = %d\n", static_cast<int>(GetUserClass()));
            NN_AM_DISPCTRL_LOG_TRACE("    |- aruid   = %lld\n", GetAruid().lower);
            NN_AM_DISPCTRL_LOG_TRACE("  buffer-owner = %d\n", static_cast<int>(GetBufferOwner()));
            NN_AM_DISPCTRL_LOG_TRACE("  destination  = %d\n", static_cast<int>(GetDestinationLayer()));
            NN_AM_DISPCTRL_LOG_TRACE("    |- indirect-layer   = %lld\n", GetIndirectLayerHandle());
            NN_AM_DISPCTRL_LOG_TRACE("    |- indirect-flip-us = %lld\n", m_IndirectProducerFlipOffset.GetMicroSeconds());

            if(GetUserClass() == IntegratedDisplayLayerUserClass_NoDisplay)
            {
                m_State = IntegratedDisplayLayerState_Determined;
                return true;
            }

            if(GetBufferOwner() == IntegratedDisplayLayerBufferOwner_OwnedByApplet)
            {
                if(GetDestinationLayer() == IntegratedDisplayLayerDestinationLayer_DisplayLayer)
                {
                    nn::vi::LayerSettings settings = {};
                    settings.Set<vi::LayerFlags::Fullscreen>(1);
                    if(GetUserClass() == IntegratedDisplayLayerUserClass_Application
                        || GetUserClass() == IntegratedDisplayLayerUserClass_FullscreenApplet
                        )
                    {
                        settings.Set<vi::LayerFlags::Opaque>(1);
                    }
                    NN_AM_DISPCTRL_LOG_TRACE("initializing direct layer\n");
                    NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::CreateDisplayLayer(&m_DirectLayerId, settings, m_Aruid));

                    // アプリの場合は PresentationTracer を有効にする
                    if(GetUserClass() == IntegratedDisplayLayerUserClass_Application)
                    {
                        NN_AM_DISPCTRL_LOG_TRACE("attaching presentation tracer\n");
                        NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::AttachDisplayLayerPresentationTracer(m_DirectLayerId));
                    }
                }
                else if(GetDestinationLayer() == IntegratedDisplayLayerDestinationLayer_IndirectLayer)
                {
                    NN_AM_DISPCTRL_LOG_TRACE("creating indirect producer\n");
                    NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::CreateIndirectProducerEndPoint(&m_IndirectProducerHandle, m_IndirectLayerHandle, m_Aruid));
                    DisplayPrimitiveOperation::SetIndirectProducerFlipOffset(m_IndirectLayerHandle, m_IndirectProducerHandle, m_IndirectProducerFlipOffset);
                }
                else
                {
                    NN_AM_DISPCTRL_LOG_WARN("invalid combination\n");
                    m_UserClass = IntegratedDisplayLayerUserClass_NoDisplay;
                }
            }
            else if(GetBufferOwner() == IntegratedDisplayLayerBufferOwner_SystemShared)
            {
                if(GetDestinationLayer() == IntegratedDisplayLayerDestinationLayer_DisplayLayer)
                {
                    NN_AM_DISPCTRL_LOG_TRACE("creating shared layer\n");
                    NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::CreateSharedLayer(&m_SharedLayerHandle, m_Aruid));
                    DisplayPrimitiveOperation::InitializeSharedLayerDetachReadyEvent(&m_SharedLayerDetachReadyEvent, m_SharedLayerHandle);
                }
                else if(GetDestinationLayer() == IntegratedDisplayLayerDestinationLayer_IndirectLayer)
                {
                    NN_AM_DISPCTRL_LOG_TRACE("creating shared layer(for shared indirect layer)\n");
                    NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::CreateSharedLayer(&m_SharedLayerHandle, m_Aruid));
                    DisplayPrimitiveOperation::InitializeSharedLayerDetachReadyEvent(&m_SharedLayerDetachReadyEvent, m_SharedLayerHandle);

                    NN_AM_DISPCTRL_LOG_TRACE("creating indirect producer(for shared indirect layer)\n");
                    NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::CreateIndirectProducerEndPoint(&m_IndirectProducerHandleForSharedLayer, m_IndirectLayerHandle, {0}));
                    DisplayPrimitiveOperation::SetIndirectProducerFlipOffset(m_IndirectLayerHandle, m_IndirectProducerHandleForSharedLayer, m_IndirectProducerFlipOffset);
                    NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::BindSharedLowLevelLayerToIndirectLayer(m_IndirectProducerHandleForSharedLayer, {0}));
                    NN_ABORT_UNLESS_RESULT_SUCCESS(DisplayPrimitiveOperation::ConnectSharedLowLevelLayerToIndirectLayer(m_IndirectProducerHandleForSharedLayer, m_SharedBufferHandle));
                }
                else
                {
                    NN_AM_DISPCTRL_LOG_WARN("invalid combination\n");
                    m_UserClass = IntegratedDisplayLayerUserClass_NoDisplay;
                }
            }
            else
            {
                NN_AM_DISPCTRL_LOG_WARN("invalid combination\n");
                m_UserClass = IntegratedDisplayLayerUserClass_NoDisplay;
            }

            m_State = IntegratedDisplayLayerState_Determined;
            return true;
        }


    }
}}}}

