﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_Storage.h>

#include <nn/nn_Common.h>
#include <nn/nn_SdkAssert.h>
#include <utility>
#include <nn/os/os_Mutex.h>
#include <mutex>
#include <nn/am/am_ResultPrivate.h>

#include <nn/am/service/am_Foundation.sfdl.h>

// for SimpleMemory
#include <cstring>
#include <nn/am/service/am_ServiceStaticAllocator.h>

// for TransferMemory
#include <nn/os/os_TransferMemory.h>
#include <nn/sf/sf_NativeHandle.h>

namespace nn { namespace am { namespace service {

namespace {

class StorageImpl
    : public sf::ISharedObject
{
public:

    explicit StorageImpl(std::shared_ptr<StorageBody> pBody) NN_NOEXCEPT
        : m_pBody(std::move(pBody))
    {
    }

    Result Open(sf::Out<sf::SharedPointer<IStorageAccessor>> pOut) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_pBody->GetKind() == StorageBody::Kind::Memory, ResultIllegalAccessToStorage());
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            NN_RESULT_THROW_UNLESS(m_Valid, ResultInvalidatedStorage());
            NN_RESULT_THROW_UNLESS(m_OpenCount == 0, ResultAlreadyOpenedStorage());
            ++this->m_OpenCount;
        }
        *pOut = sf::CreateSharedObjectEmplaced<IStorageAccessor, AccessorImpl>(this);
        NN_RESULT_SUCCESS;
    }

    Result OpenTransferStorage(sf::Out<sf::SharedPointer<ITransferStorageAccessor>> pOut) NN_NOEXCEPT
    {
        NN_RESULT_THROW_UNLESS(m_pBody->GetKind() == StorageBody::Kind::TransferMemory, ResultIllegalAccessToStorage());
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            NN_RESULT_THROW_UNLESS(m_Valid, ResultInvalidatedStorage());
            NN_RESULT_THROW_UNLESS(m_OpenCount == 0, ResultAlreadyOpenedStorage());
            ++this->m_OpenCount;
        }
        *pOut = sf::CreateSharedObjectEmplaced<ITransferStorageAccessor, TransferMemoryAccessorImpl>(this);
        NN_RESULT_SUCCESS;
    }

    Result GetAndInvalidate(sf::Out<sf::SharedPointer<IStorage>> pOut) NN_NOEXCEPT
    {
        {
            std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
            NN_RESULT_THROW_UNLESS(m_Valid, ResultInvalidatedStorage());
            NN_RESULT_THROW_UNLESS(m_OpenCount == 0, ResultAlreadyOpenedStorage());
            this->m_Valid = false;
        }
        // m_pBody への破壊的アクセスは m_Valid によって守られているのでロック外で問題ない
        *pOut = sf::CreateSharedObjectEmplaced<IStorage, StorageImpl>(std::move(m_pBody));
        NN_RESULT_SUCCESS;
    }

private:

    std::shared_ptr<StorageBody> m_pBody;
    os::Mutex m_Mutex{false};
    bool m_Valid{true};
    int m_OpenCount{0};

    void Close() NN_NOEXCEPT
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        NN_SDK_ASSERT(m_OpenCount > 0);
        --this->m_OpenCount;
    }

    class AccessorImpl
    {
    public:

        explicit AccessorImpl(StorageImpl* pStorageImpl) NN_NOEXCEPT
            : m_StorageImpl(pStorageImpl, true)
        {
        }

        ~AccessorImpl() NN_NOEXCEPT
        {
            m_StorageImpl->Close();
        }

        nn::Result GetSize(nn::sf::Out<std::int64_t> pOut) NN_NOEXCEPT
        {
            *pOut = GetStorageBody()->GetSize();
            NN_RESULT_SUCCESS;
        }

        nn::Result Write(std::int64_t offset, const nn::sf::InBuffer& buffer) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(buffer.GetSize() >= 0);
            NN_SDK_ASSERT(GetStorageBody()->GetSize() >= 0);
            auto storageSize = static_cast<size_t>(GetStorageBody()->GetSize());
            auto valid = true
                && 0 <= offset
                && buffer.GetSize() <= storageSize
                && static_cast<size_t>(offset) <= storageSize - buffer.GetSize();
            NN_RESULT_THROW_UNLESS(valid, ResultStorageOutOfRange());
            return GetStorageBody()->Write(offset, buffer.GetPointerUnsafe(), buffer.GetSize());
        }

        nn::Result Read(std::int64_t offset, const nn::sf::OutBuffer& buffer) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(buffer.GetSize() >= 0);
            NN_SDK_ASSERT(GetStorageBody()->GetSize() >= 0);
            auto storageSize = static_cast<size_t>(GetStorageBody()->GetSize());
            auto valid = true
                && 0 <= offset
                && buffer.GetSize() <= storageSize
                && static_cast<size_t>(offset) <= storageSize - buffer.GetSize();
            NN_RESULT_THROW_UNLESS(valid, ResultStorageOutOfRange());
            return GetStorageBody()->Read(offset, buffer.GetPointerUnsafe(), buffer.GetSize());
        }

    private:
        MemoryStorageBody* GetStorageBody() NN_NOEXCEPT
        {
            return reinterpret_cast<MemoryStorageBody*>(m_StorageImpl->m_pBody.get());
        }

        sf::SharedPointer<StorageImpl> m_StorageImpl;

    };

    class TransferMemoryAccessorImpl
    {
    public:

        explicit TransferMemoryAccessorImpl(StorageImpl* pStorageImpl) NN_NOEXCEPT
            : m_StorageImpl(pStorageImpl, true)
        {
        }

        ~TransferMemoryAccessorImpl() NN_NOEXCEPT
        {
            m_StorageImpl->Close();
        }

        nn::Result GetSize(nn::sf::Out<std::int64_t> pOut) NN_NOEXCEPT
        {
            *pOut = GetStorageBody()->GetSize();
            NN_RESULT_SUCCESS;
        }

        nn::Result GetHandle(sf::Out<sf::NativeHandle> pOutHandle, sf::Out<uint64_t> pOutSize) NN_NOEXCEPT
        {
            *pOutHandle = GetStorageBody()->GetHandle();
            *pOutSize   = GetStorageBody()->GetSize();
            NN_RESULT_SUCCESS;
        }

    private:
        TransferMemoryStorageBody* GetStorageBody() NN_NOEXCEPT
        {
            return reinterpret_cast<TransferMemoryStorageBody*>(m_StorageImpl->m_pBody.get());
        }

        sf::SharedPointer<StorageImpl> m_StorageImpl;

    };

};

}

sf::SharedPointer<IStorage> MakeTransferMemoryStorage(std::shared_ptr<TransferMemoryStorageBody> pBody) NN_NOEXCEPT
{
    return SfObjectFactory::CreateSharedEmplaced<IStorage, StorageImpl>(std::move(pBody));
}

sf::SharedPointer<IStorage> MakeMemoryStorage(std::shared_ptr<MemoryStorageBody> pBody) NN_NOEXCEPT
{
    return SfObjectFactory::CreateSharedEmplaced<IStorage, StorageImpl>(std::move(pBody));
}

Result MakeSimpleMemoryStorage(sf::Out<sf::SharedPointer<am::service::IStorage>> pOut, size_t size) NN_NOEXCEPT
{
    class SimpleMemoryStorageBody
        : public MemoryStorageBody
    {
    public:
        SimpleMemoryStorageBody() NN_NOEXCEPT
            : m_Buffer(nullptr)
        {
        }
        Result Initialize(size_t size) NN_NOEXCEPT
        {
            auto p = StaticAllocator<char>().allocate(size);
            NN_RESULT_THROW_UNLESS(p, ResultOutOfStorageMemory());
            this->m_Buffer = p;
            this->m_Size = size;
            NN_RESULT_SUCCESS;
        }
        ~SimpleMemoryStorageBody() NN_NOEXCEPT
        {
            if (m_Buffer)
            {
                StaticAllocator<char>().deallocate(m_Buffer, m_Size);
            }
        }
        virtual int64_t GetSize() NN_NOEXCEPT
        {
            return m_Size;
        }
        virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
        {
            std::memcpy(buffer, m_Buffer + offset, size);
            NN_RESULT_SUCCESS;
        }
        virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
        {
            std::memcpy(m_Buffer + offset, buffer, size);
            NN_RESULT_SUCCESS;
        }
    private:
        char* m_Buffer;
        size_t m_Size;
    };
    auto p = MakeShared<SimpleMemoryStorageBody>();
    NN_RESULT_DO(p->Initialize(size));
    *pOut = MakeMemoryStorage(std::move(p));
    NN_RESULT_SUCCESS;
}


Result MakeLargeStorage(sf::Out<sf::SharedPointer<am::service::IStorage>> pOut, sf::NativeHandle handle, size_t size, bool isWritable) NN_NOEXCEPT
{
    class LargeMemoryStorageBody
        : public MemoryStorageBody
    {
    public:
        LargeMemoryStorageBody() NN_NOEXCEPT
            : m_Buffer(nullptr)
        {
        }
        Result Initialize(sf::NativeHandle handle, size_t size, bool isWritable) NN_NOEXCEPT
        {
            m_TransferMemory.Attach(size, handle.GetOsHandle(), handle.IsManaged());
            handle.Detach();
            void* p;
            NN_RESULT_DO(m_TransferMemory.Map(&p, os::MemoryPermission_None));
            this->m_Writable = isWritable;
            this->m_Buffer = static_cast<char*>(p);
            this->m_Size = size;
            NN_RESULT_SUCCESS;
        }
        ~LargeMemoryStorageBody() NN_NOEXCEPT
        {
            if (m_Buffer)
            {
                m_TransferMemory.Unmap();
            }
        }
        virtual int64_t GetSize() NN_NOEXCEPT
        {
            NN_SDK_ASSERT(m_Buffer);
            return m_Size;
        }
        virtual Result Read(int64_t offset, void* buffer, size_t size) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(m_Buffer);
            std::memcpy(buffer, m_Buffer + offset, size);
            NN_RESULT_SUCCESS;
        }
        virtual Result Write(int64_t offset, const void* buffer, size_t size) NN_NOEXCEPT
        {
            NN_SDK_ASSERT(m_Buffer);
            NN_RESULT_THROW_UNLESS(m_Writable, ResultStorageIsNotWritable());
            std::memcpy(m_Buffer + offset, buffer, size);
            NN_RESULT_SUCCESS;
        }
    private:
        os::TransferMemory m_TransferMemory;
        bool m_Writable;
        char* m_Buffer;
        size_t m_Size;
    };
    auto p = MakeShared<LargeMemoryStorageBody>();
    NN_RESULT_DO(p->Initialize(std::move(handle), size, isWritable));
    *pOut = MakeMemoryStorage(std::move(p));
    NN_RESULT_SUCCESS;
}

Result MakeHandleStorage(sf::Out<sf::SharedPointer<am::service::IStorage>> pOut, sf::NativeHandle handle, size_t size) NN_NOEXCEPT
{
    class HandleStorageBody
        : public TransferMemoryStorageBody
    {
    public:
        HandleStorageBody(sf::NativeHandle handle, size_t size) NN_NOEXCEPT : TransferMemoryStorageBody(std::move(handle), size)
        {
        }
    };
    auto p = MakeShared<HandleStorageBody>(std::move(handle), size);
    *pOut = MakeTransferMemoryStorage(std::move(p));
    NN_RESULT_SUCCESS;
}

sf::SharedPointer<IStorage> StorageChannel::Pop() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    if (m_Queue.empty())
    {
        return nullptr;
    }
    auto ret = m_Queue.front();
    m_Queue.pop_front();
    if (m_PopEvent && m_Queue.empty())
    {
        m_PopEvent->Clear();
    }
    return ret;
}

void StorageChannel::Clear() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    m_Queue.clear();
    if (m_PopEvent)
    {
        m_PopEvent->Clear();
    }
}

sf::NativeHandle StorageChannel::GetPopEventHandle() NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    if (!m_PopEvent)
    {
        m_PopEvent.emplace(os::EventClearMode_ManualClear, true);
        if (!m_Queue.empty())
        {
            m_PopEvent->Signal();
        }
    }
    return sf::NativeHandle(m_PopEvent->GetReadableHandle(), false);
}

void StorageChannel::PushImpl(bool toFront, sf::SharedPointer<IStorage> p) NN_NOEXCEPT
{
    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    auto needsSignal = m_PopEvent && m_Queue.empty();
    if (toFront)
    {
        m_Queue.push_front(std::move(p));
    }
    else
    {
        m_Queue.push_back(std::move(p));
    }
    if (needsSignal)
    {
        m_PopEvent->Signal();
    }
}

}}}
