﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/am/service/am_GlobalStateManager.h>

#include <nn/nn_Common.h>
#include <nn/am/service/window/am_IWindowTransiter.h>
#include <nn/am/service/am_ServiceDiagnostics.h>
#include <nn/am/service/am_IntegratedApplet.h>
#include <nn/result/result_HandlingUtility.h>

#include <nn/os/os_ThreadApi.h>
#include <nn/settings/fwdbg/settings_SettingsGetterApi.h>
#include <nn/omm/omm_Api.h>
#include <nn/idle/idle_SystemApi.h>
#include <nn/ns/ns_ShutdownApi.h>
#include <nn/spsm/spsm_Api.h>
#include <nn/am/service/am_DisplayLayerControl.h>
#include <nn/am/service/am_PlayDataManager.h>
#include <nn/am/service/am_StuckChecker.h>
#include <nn/erpt/erpt_SystemApi.h>
#include "am_CaptureButton.h"

namespace nn { namespace am { namespace service {

class GlobalStateManager::SleepWindowTransiter
    : public window::IWindowTransiter
{
public:

    explicit SleepWindowTransiter(GlobalStateManager* pManager) NN_NOEXCEPT
        : m_pGlobalStateManager(pManager)
    {
    }

    static window::WindowProperty MakeShutterWindowInitialProperty() NN_NOEXCEPT
    {
        window::WindowProperty ret = {};
        ret.globalOrder = window::WindowGlobalOrder::Top;
        ret.controlsLcd = true;
        ret.foregroundMode = window::ForegroundMode::None;
        ret.transparentVolumeRate = 0; // 以降のレイヤの音の停止
        return ret;
    }

    virtual void SetLcd(bool onLcd) NN_NOEXCEPT NN_OVERRIDE
    {
        NN_AM_SERVICE_LOG(seq, "SetLcd(%d)\n", static_cast<int>(onLcd));
        if (onLcd)
        {
            // フェードインはスリープ復帰時に最速で行われる
        }
        else
        {
            am::service::SetSystemLayerVisible(false); // フェード時に遷移レイヤが見えないようにするため
            NN_AM_SERVICE_STUCK_CHECKED(omm_FadeOutDisplay, 60, omm::FadeOutDisplay());
        }
    }

    static window::WindowProperty MakeSleepWindowInitialProperty() NN_NOEXCEPT
    {
        window::WindowProperty ret = {};
        ret.globalOrder = window::WindowGlobalOrder::Bottom;
        ret.drivesSleepSequence = true;
        return ret;
    }

    virtual void TransitSleepSequence(window::SleepSequenceStage stage) NN_NOEXCEPT NN_OVERRIDE
    {
        m_pGlobalStateManager->TransitSleepSequenceImpl(stage);
    }

    virtual void OnPowerButtonShortPressed() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pGlobalStateManager->SignalSleepCancelEvent();
    }

    virtual void OnHomeButtonShortPressed() NN_NOEXCEPT NN_OVERRIDE
    {
        m_pGlobalStateManager->SignalSleepCancelEvent();
    }

private:

    GlobalStateManager* m_pGlobalStateManager;

};

GlobalStateManager::GlobalStateManager(window::WindowManager* pWindowManager) NN_NOEXCEPT
    : m_pWindowManager(pWindowManager)
    , m_pTransiter(MakeShared<SleepWindowTransiter>(this))
    , m_SleepCancelEvent(os::EventClearMode_ManualClear, true)
{
    this->m_pShutterWindow = pWindowManager->CreateWindow(nullptr, m_pTransiter.get(), SleepWindowTransiter::MakeShutterWindowInitialProperty());
    this->m_pSleepWindow = pWindowManager->CreateWindow(nullptr, m_pTransiter.get(), SleepWindowTransiter::MakeSleepWindowInitialProperty());
}

GlobalStateManager::~GlobalStateManager() NN_NOEXCEPT
{
    // TODO: sleep sequence 待ち
    m_pWindowManager->RemoveWindow(m_pSleepWindow);
    m_pWindowManager->RemoveWindow(m_pShutterWindow);
}

void GlobalStateManager::NotifyStart() NN_NOEXCEPT
{
    NotifyPowerStateChangeEvent(pdm::PowerStateChangeEventType::On);
    erpt::UpdatePowerOnTime();
}

Result GlobalStateManager::StartSleepSequence() NN_NOEXCEPT
{
    // OMM の出画のためディスプレイ制御をシステムモードに変更する。
    // GPU などのハードウェアを待つ必要があるため、スリープ時のみ変更する。
    am::service::ChangeDisplayLayerControlModeToSystem();
    return StartPowerDownSequenceImpl(PowerDownType_Sleep, false);
}

Result GlobalStateManager::StartShutdownSequence() NN_NOEXCEPT
{
    return StartPowerDownSequenceImpl(PowerDownType_Shutdown, true);
}

Result GlobalStateManager::StartRebootSequence() NN_NOEXCEPT
{
    return StartPowerDownSequenceImpl(PowerDownType_Reboot, true);
}

Result GlobalStateManager::StartPowerDownSequenceImpl(PowerDownType powerDownType, bool forcePowerDown) NN_NOEXCEPT
{
    decltype(m_LastClockForSleep) clockForSleep;
    {
        std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
        NN_RESULT_THROW_UNLESS(!m_Running, am::ResultInvalidCall());
        this->m_Running = true;
        clockForSleep = ++this->m_LastClockForSleep;
    }

    NN_AM_SERVICE_LOG(seq, "StartPowerDownSequenceImpl(clock=%llu, type=%d)\n", static_cast<unsigned long long>(clockForSleep), powerDownType);
    this->m_PowerDownType = powerDownType;

    auto updater = m_pWindowManager->BeginUpdate();
    updater.RefWindowProperty(m_pShutterWindow).requestsPowerDown = true;
    updater.RefWindowProperty(m_pShutterWindow).forcePowerDown = forcePowerDown;
    updater.RefWindowProperty(m_pShutterWindow).clockForSleep = clockForSleep;
    // スリープ／再起動／シャットダウンシーケンス中は、
    // HOME/POWER/CAPTURE ボタンを横取りしてそのまま破棄する
    auto& prop = updater.RefWindowProperty(m_pShutterWindow);
    prop.handlesHomeButtonShortPressed = true;
    prop.handlesHomeButtonLongPressed = true;
    prop.handlesPowerButtonShortPressed = true;
    prop.handlesPowerButtonMiddlePressed = true;
    prop.handlesPowerButtonLongPressed = true;
    prop.handlesCaptureButtonShortPressed = true;
    prop.handlesCaptureButtonLongPressed = true;

    updater.ActivateWindow(m_pShutterWindow);
    updater.ActivateWindow(m_pSleepWindow);
    NN_RESULT_SUCCESS;
}

void GlobalStateManager::TransitSleepSequenceImpl(window::SleepSequenceStage stage) NN_NOEXCEPT
{
    switch (stage)
    {
        case nn::am::service::window::SleepSequenceStage::Initializing:
        {
            // nop
            // deactivate による初期化
            return;
        }
        case nn::am::service::window::SleepSequenceStage::Sleep:
        {
            auto updater = m_pWindowManager->BeginUpdate();
            NN_AM_SERVICE_STUCK_CHECKED(omm_DisableAudioVisual, 60, omm::DisableAudioVisual());
            switch (m_PowerDownType)
            {
                case PowerDownType_Shutdown:
                {
                    ShutdownImpl();
                    break;
                }
                case PowerDownType_Reboot:
                {
                    RebootImpl();
                    break;
                }
                case PowerDownType_Sleep:
                {
                    SleepImpl();
                    break;
                }
                default: NN_UNEXPECTED_DEFAULT;
            }
            NN_AM_SERVICE_STUCK_CHECKED(omm_EnableAudioVisual, 60, omm::EnableAudioVisual());
            NN_AM_SERVICE_STUCK_CHECKED(omm_FadeInDisplay, 60, omm::FadeInDisplay());
            updater.RefWindowProperty(m_pShutterWindow).requestsPowerDown = false;
            return;
        }
        case nn::am::service::window::SleepSequenceStage::Finalizing:
        {
            {
                auto updater = m_pWindowManager->BeginUpdate();
                updater.DeactivateWindow(m_pShutterWindow);
                updater.DeactivateWindow(m_pSleepWindow);
            }
            EndSleepSequenceImpl();
            return;
        }
        default: NN_UNEXPECTED_DEFAULT;
    }
}

void GlobalStateManager::SleepImpl() NN_NOEXCEPT
{
    NN_AM_SERVICE_LOG(call, "begin Sleep\n");
    NotifyPowerStateChangeEvent(pdm::PowerStateChangeEventType::Sleep);

    m_SleepCancelEvent.Clear();
    SuspendMonitoring();
    omm::EnterSleepAndWait( &m_SleepCancelEvent );
    ResumeMonitoring(TimeSpan::FromSeconds(5));

    NotifyPowerStateChangeEvent(pdm::PowerStateChangeEventType::Awake);
    erpt::UpdateAwakeTime();
    NN_AM_SERVICE_LOG(call, "end Sleep\n");
}

NN_NORETURN void GlobalStateManager::ShutdownImpl() NN_NOEXCEPT
{
    NN_AM_SERVICE_LOG(call, "begin shutdown\n");
    NotifyPowerStateChangeEvent(pdm::PowerStateChangeEventType::Off);
    NN_AM_SERVICE_STUCK_CHECKED(ns_PrepareShutdown, 60, ns::PrepareShutdown());
    NN_AM_SERVICE_STUCK_CHECKED(ns_PrepareShutdownForSystemUpdate, 60, ns::PrepareShutdownForSystemUpdate());
    SuspendMonitoring();
    spsm::ShutdownSystem();
}

NN_NORETURN void GlobalStateManager::RebootImpl() NN_NOEXCEPT
{
    NN_AM_SERVICE_LOG(call, "begin reboot\n");
    NotifyPowerStateChangeEvent(pdm::PowerStateChangeEventType::Off);
    NN_AM_SERVICE_STUCK_CHECKED(ns_PrepareShutdown, 60, ns::PrepareShutdown());
    NN_AM_SERVICE_STUCK_CHECKED(ns_PrepareShutdownForSystemUpdate, 60, ns::PrepareShutdownForSystemUpdate());
    SuspendMonitoring();
    spsm::RebootSystem();
}

void GlobalStateManager::EndSleepSequenceImpl() NN_NOEXCEPT
{
    // スリープシーケンス終了時点でキャプチャボタンを押下している場合は、
    // それを無効化（次回、再押下時から再びキャプチャボタンが有効になる）。
    GetCaptureButton().InvalidateCurrentButtonPressing();

    // OMM はこれ以上描画を行わないのでディスプレイ制御モードをアプレットに戻す。
    am::service::ChangeDisplayLayerControlModeToApplet();

    std::lock_guard<decltype(m_Mutex)> lk(m_Mutex);
    NN_SDK_ASSERT(m_Running);
    NN_AM_SERVICE_LOG(seq, "EndSleepSequence(clock=%llu)\n", static_cast<unsigned long long>(m_LastClockForSleep));
    this->m_Running = false;
    OnFinishedSleepSequence();
}

Result GlobalStateManager::LoadAndApplyIdlePolicySettings() NN_NOEXCEPT
{
    NN_AM_SERVICE_LOG(seq, "Invoke idle::LoadAndApplySettings()");
    return idle::LoadAndApplySettings();
}

void GlobalStateManager::SignalSleepCancelEvent() NN_NOEXCEPT
{
    m_SleepCancelEvent.Signal();
}

}}}
